% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MRGmerge.R
\name{MRGmerge}
\alias{MRGmerge}
\title{Merge two or more multi-resolution grids to a common resolution}
\usage{
MRGmerge(
  himg1,
  himg2,
  vars1,
  vars2,
  addVars1,
  addVars2,
  useAllVars = FALSE,
  na.rm = TRUE,
  postProcess = FALSE,
  aggr = "merge",
  ...
)
}
\arguments{
\item{himg1}{Either a multi-resolution grid (typically resulting from a call to multiResGrid), or a list of such grids}

\item{himg2}{A multi-resolution grid, typically resulting from a call to multiResGrid}

\item{vars1}{Variable(s) of interest that should be merged from the first grid, or a list of variables,
one for each grid in the list \code{himg1}}

\item{vars2}{Variable(s) of interest that should be merged from the second grid}

\item{addVars1}{variable(s) to be added to vars1 (normally used if vars1 is found from 
the attributes of himg1, but new columns have later been added to it}

\item{addVars2}{variable(s) to be added to vars2 (normally used if vars2 is found from 
the attributes of himg2, but new columns have later been added to it}

\item{useAllVars}{logical; should all variables be used in the merging process (applies
to all objects)}

\item{na.rm}{Should NA values be removed when summing values (essentially
treating them equal to zero)}

\item{postProcess}{Logical; should the postprocessing be done as part
of creation of the multiresolution grid (TRUE), or be done in a separate 
step afterwards (FALSE). The second option is useful when wanting
to check the confidential grid cells of the final map}

\item{aggr}{Should data be aggregated to the largest grid cell (\code{aggr = "merge"}),
or should data from larger grid cells be disaggregated to smaller grid cells
   \code{(aggr = "disaggr")}}

\item{...}{Additional grids (himg3, himg4, ...) and variables (vars3, vars4, ...) to be merged. 
Additional grids and variables must be named.}
}
\value{
The function produces a new multiresolution grid, which is a
\code{\link[sf]{sf}}-object with polygons.
}
\description{
Merge two or more multi-resolution grids to a common resolution
}
\details{
This function can merge different multi-resolution grids to a common 
resolution, i.e., it will select the grid cells with the lowest resolution,
as these are the ones defining the restrictions.  

The function will merge the variable names in \code{vars1, vars2, ...}
if they exist. 
If the variables are not given as \code{vars} or attributes, the function
will try to guess them from the column names. Typical column names used by
MRG (mostly temporary variables such as \code{small}, \code{confidential} etc)
will be ignored. If variable names partly coincide with any of these names,
or with \code{count}, \code{res}, \code{geometry}, it is necessary to specify vars.

The function can also use "vars" from (\code{attr(himg, "vars")}). These
are added or updated by \code{\link{multiResGrid}} and \code{\link{multiResGrid}}, 
but might disappear if the
grid has been manipulated, or has been exported to another format for transmission.
The function will by default not merge the values of new columns
that have been added outside MRG-functions, instead it will give a warning
that the object has changed. This can be fixed in different ways


\enumerate{ 
\item{New variables are added to attr(himg, \"vars\") (The function will still issue this warning) }
\item{All merging variables are added to vars1, vars2, etc }
\item{New variables are added to addVars1, addVars2, etc }
\item{All columns are used (set \code{useAllVars = TRUE)}. This is used for all objects, and might 
      cause other isues with repeated variable names in different objects"}
}      

The multi-resolution grids must be passed as named parameters if more than two 
are given. 

Common variable names in different grids should be avoided.

The default of the function is to treat NA-values as zeroes when merging 
(through \code{na.rm} in sums).
It will therefore not be possible to separate restricted grid cells
from grid cells with zero observations after merging, except for the ones that
have been left as they were. The alternative would
be a much higher number of NA-values in the merged grids.

The resulting grid will most likely not have exactly the same values as a 
multi-resolution grid produced 
directly from the microdata. If the input-grids have been post-processed
(the normal situation when not having access to the microdata), the 
grid cell values have usually been rounded, and some might have been 
suppressed. As these rounded and potentially suppressed values are summed,
their values are likely to deviate from those that are computed directly
from the microdata through a joint gridding process.

The argument \code{aggr} will decide on the direction of aggregation. 
If \code{aggr == "merge"}, The values in high resolution grid cells will 
be aggregated to match those of lower resolution grid cells in the 
second grid. If \code{aggr == "disaggr"}, the values of the lower resolution
grid cells will be redistributed equally among higher resolution grid cells,
according to their area. 
Note that this will most likely result in grid cell values that are apparently
confidential (for example having less than 10 individuals). These are still
not confidential values, but are average values from a larger area. 
This will in most cases be fine if the data is used for analyses, 
but publication of such values should be done with care.

Also note that if more than 2 MRG-grids are merged at the same time, 
then the redistribution will occur more than once. If the resolution of some grid cells
becomes higher for each redistribution, with some of the high resolution grid cells missing,
then the average values might differ for different high resolution grid cells coming from
the same low value grid cell. See the plotted examples of h2 and h22.
}
\examples{
\donttest{
library(sf)
library(dplyr)
library(ggplot2)
library(viridis)

# These are SYNTHETIC agricultural FSS data 
data(ifs_dk) # Census data
ifs_weight = ifs_dk \%>\% dplyr::filter(Sample == 1) # Extract weighted subsample

# Create spatial data
ifg = fssgeo(ifs_dk, locAdj = "LL")
fsg = fssgeo(ifs_weight, locAdj = "LL")

# We use the numeric part of the farmtype to create a third variable. This 
# is done for the an example, the value does not have any meaning when treated 
# like this
ifg$ft = as.numeric(substr(ifg$FARMTYPE, 3, 4))^2

ress = c(1,5,10,20,40, 80, 160)*1000
# Create regular grid of the variables
ifl = gridData(ifg, vars = c("UAA", "UAAXK0000_ORG", "ft"), res = ress)

# Create the different multi-resolution grids
himg1 = multiResGrid(ifl, vars = "UAA", ifg = ifg, postProcess = FALSE)
himg2 = multiResGrid(ifl, vars = "UAAXK0000_ORG", ifg = ifg, postProcess = FALSE)
himg3 = multiResGrid(ifl, vars = "ft", ifg = ifg, postProcess = FALSE)

# The grids have different number of polygons
dim(himg1)
dim(himg2)
dim(himg3)

hh1 = MRGmerge(himg1, himg2, himg3 = himg3)
dim(hh1)
# Postprocessing can also be done on the merged object
hh11 = MRGmerge(himg1, himg2, himg3 = himg3, postProcess = TRUE, rounding = -1)
dim(hh11)
summary(hh1$UAA-hh11$UAA)

# Here the merging will instead redistribute average values to 
# the higher resolution grid cells, and also seeing the effect
# of merging a third layer
hh2 = MRGmerge(himg1, himg2, aggr = "disaggr")
hh22 = MRGmerge(himg1, himg2, himg3 = himg3, aggr = "disaggr")
himg2$orgShare = himg2$UAAXK0000_ORG/himg2$res^2 * 10000
hh2$orgShare = hh2$UAAXK0000_ORG/hh2$res^2 * 10000
hh22$orgShare = hh22$UAAXK0000_ORG/hh22$res^2 * 10000
# Plot the organic share (organic area relative to grid cell area) for
# the original MRG grid for organic area, and after merging with the higher
# resolution maps.
p1 = ggplot(himg2) + geom_sf(aes(fill = orgShare)) + ggtitle("original") +
      scale_fill_viridis()
p2 = ggplot(hh2) + geom_sf(aes(fill = orgShare)) + ggtitle("merged two")+
      scale_fill_viridis() 
p3 = ggplot(hh22) + geom_sf(aes(fill = orgShare)) + ggtitle("merged three")+
      scale_fill_viridis() 
if (require(patchwork)) p1 + p2 + p3 + plot_spacer() + plot_layout(guides = 'collect')

# If two data sets share the same variable, one of them has to be renamed.
# (A comparison of the two can act as a indication of possible errors 
# introduced through the post-processing)

himg21 = multiResGrid(ifl, vars = c("UAA", "UAAXK0000_ORG"), ifg = ifg, postProcess = FALSE)
hh3 = try(MRGmerge(himg1, himg21, himg3 = himg3))
himg21 = himg21 \%>\% rename(UAA2 = UAA, weight_UAA2 = weight_UAA) 
hh3 = MRGmerge(himg1, himg21, himg3 = himg3)


summary(hh3[, c("UAA", "UAA2")])

himg4 = multiResGrid(ifl, vars = c("UAA", "ft", "UAAXK0000_ORG"), ifg = ifg, postProcess = FALSE)
summary(hh1[, c("UAA", "UAAXK0000_ORG", "ft")])
summary(himg4[, c("UAA", "UAAXK0000_ORG", "ft")])
}
           
           
}
