# M <- runGMM(x = data, k=2 )

runGMM <-
  function(x, k, max_iter = 100, run_number = 10, smax_iter = 3,
           s_iter = 10, c_iter = 10,
           tol = 1e-6, burn_in = 3,
           verbose = FALSE) {
    
    all_init <- vector("list", 9)
    fail <- "K is too big, try to decrease k"
    
    .msg <- function(...) {
      if (isTRUE(verbose)) message(...)
    }
    
    has_na <- function(lst) {
      any(sapply(lst, function(sublist) any(is.na(sublist))))
    }
    
    # 1 Random choose
    .msg("Running Random")
    all_init[[1]] <- param_random(x = x, k = k)
    
    # 2 hierarchical average
    .msg("Running hierarchical average")
    # all_init[[2]] <-  tryCatch({param_hier_aver(x = x, k = k)}, error = function(e) { fail })
    param_ha <- param_hier_aver(x = x, k = k)
    all_init[[2]] <- if (has_na(param_ha)) fail else param_ha
    
    # 3 hierarchical ward
    .msg("Running hierarchical ward")
    param_hw <- param_hier_ward(x = x, k = k)
    all_init[[3]] <- if (has_na(param_hw)) fail else param_hw
    
    # 4 kmeans
    .msg("Running kmeans")
    param_k <- param_kmean(x = x, k = k)
    all_init[[4]] <- if (has_na(param_k)) fail else param_k
    
    # 5 emEM
    .msg("Running emEM")
    all_init[[5]] <- param_emEM(x = x, k = k, run_number = run_number,
                                max_iter = smax_iter, tol = tol)
    
    # 6 emAEM
    .msg("Running emAEM")
    all_init[[6]] <- rerun(function() param_emaEM(x = x, k = k, run_number = run_number,
                                                  max_iter = smax_iter, tol = tol))
    
    # 7 SEM
    .msg("Running sem")
    all_init[[7]] <- rerun(function() param_SEM(x = x, k = k, iter = s_iter))
    
    # 8 CEM
    .msg("Running cem")
    all_init[[8]] <- rerun(function() param_CEM(x = x, k = k, iter = c_iter))
    
    # 9 mclust
    .msg("Running mclust")
    param_m <- param_mclust(x = x, k = k)
    all_init[[9]] <- if (!is.null(param_m)) param_m else fail
    
    names(all_init) <- c('Random', 'hierarchical.average', 'hierarchical.ward', 'kmeans',
                         'emEM', 'emAEM', 'sem', 'cem', 'mclust')
    
    all_result <- vector("list", 9)
    
    for (run in seq_along(all_init)) {
      # cat("Running = ", run , "\n")
      if (is.numeric(unlist(all_init[[run]]))) {
        all_result[[run]] <- tryCatch(
          runEM(x = x, param = all_init[[run]], max_iter = max_iter),
          #  runEM(x = x, param = all_init[[2]] , max_iter = max_iter)
          
          error = function(e) fail
        )
      } else {
        all_result[[run]] <- fail
      }
    }
    
    names(all_result) <- names(all_init)
    class(all_result) <- "gmminit_run"
    all_result
    
    
  }

#' @export
print.gmminit_run <- function(x, ...) {
  if (!is.list(x)) stop("print.gmminit_run() expects a list.")
  
  nm <- names(x)
  if (is.null(nm)) nm <- paste0("method_", seq_along(x))
  
  #
  is_fail <- vapply(x, function(obj) is.character(obj) && length(obj) == 1, logical(1))
  
  # 
  bics <- rep(NA_real_, length(x))
  for (i in seq_along(x)) {
    if (is_fail[i]) next
    obj <- x[[i]]
    if (is.list(obj) && !is.null(obj$BIC) && is.numeric(obj$BIC)) {
      bics[i] <- obj$BIC[1]
    }
  }
  
  cat("GMMinit: runGMM results\n")
  cat("  Methods:", length(x), "\n")
  cat("  Successful:", sum(!is_fail), "\n")
  cat("  Failed:", sum(is_fail), "\n")
  
  if (any(!is_fail & is.finite(bics))) {
    best_idx <- which.min(bics)
    cat("  Best (by BIC):", nm[best_idx], sprintf("(BIC = %.4f)", bics[best_idx]), "\n")
  }
  
  cat("\n")
  cat(sprintf("%-22s %-5s %-12s\n", "Method", "Stat", "BIC"))
  cat(strrep("-", 22 + 1 + 5 + 1 + 12), "\n", sep = "")
  
  for (i in seq_along(x)) {
    if (is_fail[i]) {
      cat(sprintf("%-22s %-5s %-12s\n", nm[i], "FAIL", "NA"))
    } else {
      bic_str <- if (is.finite(bics[i])) sprintf("%.4f", bics[i]) else "NA"
      cat(sprintf("%-22s %-5s %-12s\n", nm[i], "OK", bic_str))
    }
  }
  cat("\n(Use x[['method']] to access details; see names(x).)\n")

  invisible(x)
}









