#' Compute the Shannon homogeneity for a vector
#'
#' Computes the Shannon homogeneity (normalized Shannon entropy) for
#' a vector, typically categorical but the procedure also works with
#' numerical vectors.  Returns a value in the range from 0 (for a highly
#' inhomogeneous vector, concentrated entirely on one of L > 1 levels)
#' to 1 (for a completely homogeneous vector).  By convention, vectors
#' of length 0 or 1 return homogeneity values of 1.
#'
#' @param x the vector to be characterized
#' @param dgts number of digits in the return value (default = 3)
#'
#' @return a numerical homogeneity measure between 0 and 1
#' @export
#'
#' @examples
#' x <- rep(c("a", "b", "c", "d", "e"), 200)
#' y <- c(rep("a", 497), rep("b", 497), rep("c", 2), rep("d", 2), rep("e", 2))
#' z <- c(rep("a", 996), "b", "c", "d", "e")
#' ShannonHomogeneity(x)
#' ShannonHomogeneity(y)
#' ShannonHomogeneity(z)
#'
ShannonHomogeneity <- function(x, dgts = 3){
  #
  stopifnot("dgts must be positive"= dgts > 0)
  #
  if (length(x) == 0){
    H <- 1
  } else {
    tbl <- table(x, useNA = "ifany")
    L <- length(tbl)
    if (L == 1){
      H <- 1
    } else {
      pVec <- as.numeric(tbl)/sum(tbl)
      pLogPvec <- unlist(lapply(pVec, FUN = plogpFun))
      H <- - sum(pLogPvec)/log(L)
    }
  }
  #
  return(round(H, digits = dgts))
}

plogpFun <- function(p){
  val <- ifelse(p == 0, 0, p * log(p))
  return(val)
}

