#' smoothSegs
#'
#' This function is used to attempt to smooth the input copy number
#' segments in order to reduce the biasing affect of the technology
#' and copy number caller used to make the segments. It does this by
#' trying to join together close segs and removing small abbarent 
#' segments.
#'
#' @param segData The segData to be smoothed
#' @param cores Number of cores to be used for parallel smoothing. Default is 1.
#' @return Returns the smoothed segments
#' @import foreach
#' @export
#' @examples
#' smoothSegs(segDataExp)
smoothSegs = function(segData, cores = 1){
  
  workers = parallel::makeCluster(cores, type = "SOCK")
  doParallel::registerDoParallel(workers)
  
  sampName = ""
  #Assigns each feature to a different worker
  convSegData = foreach(sampName = names(segData)) %dopar% {
    sample = segData[[sampName]]
    #Remove Duplicate Lines
    sample = sample[!duplicated(sample),]

    #Converts from log ratio to absolute
    if (min(sample[,4])<0)
      sample[,4] = round(2^(sample[,4])*2)
    
    #Scale the high CN's to between 10 and 25 by percentiles
    maxCN = max(sample[,4])
    if (maxCN > 25){
      indices = which(sample[,4]>10)
      percentile = stats::ecdf(sample[indices,4])
      for (i in indices){
        sample[i,4] = round(10 + percentile(sample[i,4])*15)
      }
    }
    
    convSamp = sample[1,]
    for (i in 2:(nrow(sample)-1)){
      if (sample[i,1] != convSamp[nrow(convSamp),1]){
        convSamp = rbind(convSamp,sample[i,])
        next()
      }
      dist = sample[i,2]-convSamp[nrow(convSamp),3]
      segSize = sample[i,3] - sample[i,2]
      prevSegSize = convSamp[nrow(convSamp),3] - convSamp[nrow(convSamp),2]
      nextSegSize = sample[i+1,3] - sample[i+1,2]
      
      #Goes through a number of steps to figure out whether to combine the segments
      #or to leave them alone. First checks if they overlap
      if (dist < 0 ){
        end = ifelse(convSamp[nrow(convSamp),3]>sample[i,3],convSamp[nrow(convSamp),3],sample[i,3])
        segVal = (convSamp[nrow(convSamp),4]+sample[i,4])/2
        convSamp[nrow(convSamp),3] = end
        convSamp[nrow(convSamp),4] = segVal
      }
      else if (segSize <=0){
        convSamp[nrow(convSamp),3] = sample[i,3]
      }
      #Then checks if they are too far away to smooth together.
      else if (dist >= 15000){
        convSamp = rbind(convSamp,sample[i,])
      } 
      #Then checks whether the segment values are the same and if so combines them
      else if (abs(sample[i,4] == convSamp[nrow(convSamp),4])){
        convSamp[nrow(convSamp),3] = sample[i,3]
      } 
      #Then checks whether the segment is small
      else if (segSize >= 100000){
        convSamp = rbind(convSamp,sample[i,])
      } 
      #Then checks whether the segments on either side are the same values 
      else if (abs(sample[i+1,4] - convSamp[nrow(convSamp),4]) >= 1) {
        convSamp = rbind(convSamp,sample[i,])
      } 
      #Then checks if the segments on either side are also small
      else if (prevSegSize < 100000 || nextSegSize < 100000) {
        convSamp = rbind(convSamp,sample[i,])
      } 
      #Removes the small segment if the neighbors are large and the same value
      else {
        convSamp[nrow(convSamp),3] = sample[i,3]
      }
    }
    convSamp
  }
  parallel::stopCluster(workers)
  
  #Renames values in new list and returns the converted data
  names(convSegData) = names(segData)
  return(convSegData)
}