% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/u5mr_trussell.R
\name{u5mr_trussell}
\alias{u5mr_trussell}
\title{Estimating under-five mortality using Coale-Demeny life table models}
\usage{
u5mr_trussell(
  data,
  women = "women",
  child_born = "child_born",
  child_dead = "child_dead",
  agegrp = "agegrp",
  model = "west",
  svy_year = 1976.5,
  sex
)
}
\arguments{
\item{data}{processed data}

\item{women}{total number of women}

\item{child_born}{children ever born}

\item{child_dead}{children dead}

\item{agegrp}{age grouping}

\item{model}{Coale-Demeny life table model:
\code{north}, \code{south}, \code{east}, and \code{west}}

\item{svy_year}{end of the survey}

\item{sex}{indicates sex-specific estimates: \code{both}, \code{male}, and \code{female}}
}
\value{
\code{data.frame}
\itemize{
\item \code{agegrp}      - five-year age groups
\item \code{women} - total number of women
\item \code{child_born}  - total number of children ever born
\item \code{child_dead}  - number of children dead
\item \code{pi} - average parity
\item \code{di} - proportion of dead among children ever born
\item \code{ki} - multipliers
\item \code{qx} - probabilities of dying at age x
\item \code{ti} - time between survey year and interpolated year
\item \code{year} - reference year
\item \code{h}  - interpolation factor
\item \code{q5} - under-five mortality
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

\code{u5mr_trussell()} uses the Trussell version of the BRASS method
and calculates under-five mortality estimates.
}
\details{
T. J. Trussell developed the Trussell version of the Brass method to estimate
child mortality using summary birth history (United Nations,  1983b, Chapter III).
It is based on the Coale-Demeny life table models of either North, South, East, or West.

\strong{Computational Procedure}

Step 1. Preparing the dataset

The function needs four variables from the input data:

a) \code{agegrp}: age groups representing \code{15-19}, \code{20-24}, \code{25-29}, \code{30-34},
\code{35-39}, \code{40-44}, and \code{45-49}.

b) \code{women}: the total number of women in the age group irrespective of their marital
or reporting status

c) \code{child_born}: the total number of children ever borne by these women

d) \code{child_dead}: the number of children dead reported by these women

Step 1.1. Calculation of average parity per woman \code{P(i)}

\deqn{P(i) = CEB(i) / FP(i)}
\itemize{
\item \code{CEB(i)} is the total number of children ever borne by these women
\item \code{FP(i)} is the total number of women in the
age group irrespective of their marital or reporting status.
}

Step 1.2. Calculation of the proportions dead among children ever borne \code{D(i)}

\deqn{D(i) = CD(i) / CEB(i)}
\itemize{
\item \code{CD(i)} is the number of children dead for women of age group i
}

Step 2. Calculating the multipliers \code{k(i)} and probabilities of dying by age x \code{q(x)}

\deqn{k(i) = a(i) + b(i) P(1)/P(2) + c(i) P(2)/P(3)}

where \code{a(i)}, \code{b(i)}, and \code{c(i)} are coefficients estimated by regression analysis of
simulated model cases, and \code{P(1)/P(2)} and \code{P(2)/P(3)} are parity ratios.

\deqn{q(x) = k(i) x D(i)}

Step 3. Calculating the reference dates for \code{q(x)} and interpolating \code{q5}

Under conditions of steady mortality change over time, a reference time \code{t(i)}
can be estimated for each \code{q(x)}.

\deqn{t(i) = a(i) + b(i) P(1)/P(2) + c(i) P(2)/P(3)}

Actual dates can be obtained by subtracting \code{t(i)} from the reference date of
the survey or census, expressed in decimal terms.

Step 4. Interpolating between \code{q(x)} and model life table

A common index, in this case, under-five mortality \code{q(5)} can be obtained by
conversions of corresponding \code{q(x)} values to the specified family of
the Coale-Demeny life table models. In a given life table family and sex,
the first step is to identify the mortality levels with q(x) values that
enclose the estimated value \eqn{q^e(x)}.

\deqn{q^j(x) > q^e(x) > q^j+1(x)}

where \eqn{q^j(x)} and \eqn{q^j+1(x)} are the model values of \code{q(x)} for
levels \code{j} and \code{j+1}, and \eqn{q^e(x)} is the estimated value.

The desired common index \eqn{q^c(5)}, or \code{q5} is given by

\deqn{q^c(5) = (1.0 - h) x q^j(5) + h x q^e(5)}

where h is the interpolation factor calculated in the following way:

\deqn{h = q^e(x) - q^j(x) / q^j+1(x) - q^j(x)}

Step 5. Finalizing the calculation

The final output is combined into a \code{data.frame}, which contains original dataset
as well as statistics produced during the computational procedure.
}
\examples{

## Using Bangladesh survey data to estimate child mortality
data("bangladesh")
bang_both <- u5mr_trussell(bangladesh, sex = "both", model = "south", svy_year = 1974.3)
bang_male <- u5mr_trussell(bangladesh, child_born = "male_born",
                 child_dead = "male_dead", sex = "male",
                 model = "south", svy_year = 1974.3)
bang_female <- u5mr_trussell(bangladesh, child_born = "female_born",
                 child_dead = "female_dead", sex = "female",
                 model = "south", svy_year = 1974.3)

## plotting all data points
with(bang_both,
    plot(year, q5, type = "b", pch = 19,
         ylim = c(0, .45),
         col = "black", xlab = "Reference date", ylab = "u5MR",
         main = paste0("Under-five mortality, q(5) in Bangladesh, estimated\n",
                       "using model South and the Trussell version of the Brass method")))
with(bang_both, text(year, q5, agegrp, cex=0.65, pos=3,col="purple"))
with(bang_male,
    lines(year, q5, pch = 18, col = "red", type = "b", lty = 2))
with(bang_female,
    lines(year, q5, pch = 18, col = "blue", type = "b", lty = 3))
legend("bottomright", legend=c("Both sexes", "Male", "Female"),
      col = c("Black", "red", "blue"), lty = 1:3, cex=0.8)


## Using panama survey data to estimate child mortality
data("panama")
pnm_both <- u5mr_trussell(panama, sex = "both", model = "west", svy_year = 1976.5)
pnm_male <- u5mr_trussell(panama, child_born = "male_born",
                child_dead = "male_dead", sex = "male",
                model = "west", svy_year = 1976.5)
pnm_female <- u5mr_trussell(panama, child_born = "female_born",
                child_dead = "female_dead", sex = "female",
                model = "west", svy_year = 1976.5)

## plotting all data points
with(pnm_both,
    plot(year, q5, type = "b", pch = 19,
        ylim = c(0, .2), col = "black", xlab = "Reference date", ylab = "u5MR",
         main = paste0("Under-five mortality, q(5) in Panama, estimated\n",
                       "using model West and the Trussell version of the Brass method")))
with(pnm_both, text(year, q5, agegrp, cex=0.65, pos=3,col="purple"))
with(pnm_male,
    lines(year, q5, pch = 18, col = "red", type = "b", lty = 2))
with(pnm_female,
    lines(year, q5, pch = 18, col = "blue", type = "b", lty = 3))
legend("bottomleft", legend=c("Both sexes", "Male", "Female"),
      col = c("Black", "red", "blue"), lty = 1:3, cex=0.8)

}
\references{
\enumerate{
\item United Nations (1990) "Step-by-step guide to the estimation of the child mortality"
\url{https://www.un.org/en/development/desa/population/publications/pdf/mortality/stepguide_childmort.pdf}
\item United Nations (1983) "Manual X indirect techniques for demographic estimation"
\url{https://www.un.org/en/development/desa/population/publications/pdf/mortality/stepguide_childmort.pdf}
}
}
