% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classes__OAuthProvider.R
\name{oauth_provider}
\alias{oauth_provider}
\title{Create generic \link{OAuthProvider}}
\usage{
oauth_provider(
  name,
  auth_url,
  token_url,
  userinfo_url = NA_character_,
  introspection_url = NA_character_,
  revocation_url = NA_character_,
  issuer = NA_character_,
  use_nonce = NULL,
  use_pkce = TRUE,
  pkce_method = "S256",
  userinfo_required = NULL,
  userinfo_id_token_match = NULL,
  userinfo_id_selector = function(userinfo) userinfo$sub,
  id_token_required = NULL,
  id_token_validation = NULL,
  extra_auth_params = list(),
  extra_token_params = list(),
  extra_token_headers = character(),
  token_auth_style = "header",
  jwks_cache = NULL,
  jwks_pins = character(),
  jwks_pin_mode = "any",
  jwks_host_issuer_match = NULL,
  jwks_host_allow_only = NULL,
  allowed_algs = c("RS256", "RS384", "RS512", "PS256", "PS384", "PS512", "ES256",
    "ES384", "ES512", "EdDSA"),
  allowed_token_types = c("Bearer"),
  leeway = getOption("shinyOAuth.leeway", 30)
)
}
\arguments{
\item{name}{Provider name (e.g., "github", "google"). Cosmetic
only; used in logging and audit events}

\item{auth_url}{Authorization endpoint URL}

\item{token_url}{Token endpoint URL}

\item{userinfo_url}{User info endpoint URL (optional)}

\item{introspection_url}{Token introspection endpoint URL (optional; RFC 7662)}

\item{revocation_url}{Token revocation endpoint URL (optional; RFC 7009)}

\item{issuer}{OIDC issuer URL (optional; required for ID token validation).
This is the base URL that identifies the OpenID Provider (OP). It is used
during ID token validation to verify the \code{iss} claim in the ID
token matches the expected issuer. It is also used to fetch the provider's
JSON Web Key Set (JWKS) for verifying ID token signatures (typically via
the OIDC discovery document located at \verb{/.well-known/openid-configuration}
relative to the issuer URL)}

\item{use_nonce}{Whether to use OIDC nonce. This adds a \code{nonce} parameter to
the authorization request and validates the \code{nonce} claim in the ID token.
This is recommended for OIDC flows to mitigate replay attacks}

\item{use_pkce}{Whether to use PKCE. This adds a \code{code_challenge} parameter to
the authorization request and requires a \code{code_verifier} when exchanging
the authorization code for tokens. This is prevents authorization code
interception attacks}

\item{pkce_method}{PKCE code challenge method ("S256" or "plain"). "S256" is
recommended. "plain" should only be used for non-compliant providers that
do not support "S256"}

\item{userinfo_required}{Whether to fetch userinfo after token exchange.
User information will be stored in the \code{userinfo} field of the returned
\code{OAuthToken} object. This requires a valid \code{userinfo_url} to be set.
If fetching the userinfo fails, the token exchange will fail.

For the low-level constructor \code{\link[=oauth_provider]{oauth_provider()}}, when not explicitly
supplied, this is inferred from the presence of a non-empty \code{userinfo_url}:
if a \code{userinfo_url} is provided, \code{userinfo_required} defaults to \code{TRUE},
otherwise it defaults to \code{FALSE}. This avoids unexpected validation errors
when \code{userinfo_url} is omitted (since it is optional).}

\item{userinfo_id_token_match}{Whether to verify that the user ID ("sub") from the ID token
matches the user ID extracted from the userinfo response. This requires both
\code{userinfo_required} and \code{id_token_validation} to be TRUE (and thus a valid \code{userinfo_url}
and \code{issuer} to be set, plus potentially setting the client's scope to include "openid",
so that an ID token is returned). Furthermore, the provider's \code{userinfo_id_selector} must be configured
to extract the user ID from the userinfo response. This check helps ensure
the integrity of the user information by confirming that both sources agree on the user's identity.

For \code{\link[=oauth_provider]{oauth_provider()}}, when not explicitly supplied, this is inferred as
\code{TRUE} only if both \code{userinfo_required} and \code{id_token_validation} are \code{TRUE};
otherwise it defaults to \code{FALSE}.}

\item{userinfo_id_selector}{A function that extracts the user ID from the userinfo response.#'
Should take a single argument (the userinfo list) and return the user ID
as a string.

This is used when \code{userinfo_id_token_match} is TRUE.
Optional otherwise; when not supplied, some features (like subject matching)
will be unavailable. Helper constructors like \code{\link[=oauth_provider]{oauth_provider()}} and \code{\link[=oauth_provider_oidc]{oauth_provider_oidc()}}
provide a default selector that extracts the \code{sub} field.}

\item{id_token_required}{Whether to require an ID token to be returned
during token exchange. If no ID token is returned, the token exchange
will fail. This requires the provider to be a valid OpenID Connect
provider and may require setting the client's scope to include "openid".

Note: At the S7 class level, this defaults to FALSE so that pure OAuth 2.0
providers can be configured without OIDC. Helper constructors like
\code{\link[=oauth_provider]{oauth_provider()}} and \code{\link[=oauth_provider_oidc]{oauth_provider_oidc()}} will enable this when an
issuer is supplied or OIDC is explicitly requested.}

\item{id_token_validation}{Whether to perform ID token validation after token exchange.
This requires the provider to be a valid OpenID Connect provider with a configured
\code{issuer} and the token response to include an ID token (may require setting
the client's scope to include "openid").

Note: At the S7 class level, this defaults to FALSE. Helper constructors like
\code{\link[=oauth_provider]{oauth_provider()}} and \code{\link[=oauth_provider_oidc]{oauth_provider_oidc()}} turn this on when an issuer
is provided or when OIDC is used.}

\item{extra_auth_params}{Extra parameters for authorization URL}

\item{extra_token_params}{Extra parameters for token exchange}

\item{extra_token_headers}{Extra headers for token exchange requests (named character vector)}

\item{token_auth_style}{How to authenticate when exchanging tokens. One of:
\itemize{
\item "header": HTTP Basic (client_secret_basic)
\item "body": Form body (client_secret_post)
\item "client_secret_jwt": JWT client assertion signed with HMAC using client_secret
(RFC 7523)
\item "private_key_jwt": JWT client assertion signed with an asymmetric key
(RFC 7523)
}}

\item{jwks_cache}{JWKS cache backend. If not provided, a \code{cachem::cache_mem(max_age = 3600)}
(1 hour) cache will be created. May be any cachem‑compatible backend, including
\code{\link[cachem:cache_disk]{cachem::cache_disk()}} for a filesystem cache shared across workers, or a custom
implementation created via \code{\link[=custom_cache]{custom_cache()}} (e.g., database/Redis backed).

TTL guidance: Choose \code{max_age} in line with your identity platform’s JWKS rotation
and cache‑control cadence. A range of 15 minutes to 2 hours is typically sensible;
the default is 1 hour. Shorter TTLs adopt new keys faster at the cost of more JWKS
traffic; longer TTLs reduce traffic but may delay new keys slightly. Signature
verification will automatically perform a one‑time JWKS refresh when a new \code{kid}
appears in an ID token.

Cache keys are internal, hashed by issuer and pinning configuration. Cache values are
lists with elements \code{jwks} and \code{fetched_at} (numeric epoch seconds)}

\item{jwks_pins}{Optional character vector of RFC 7638 JWK thumbprints
(base64url) to pin against. If non-empty, fetched JWKS must contain keys
whose thumbprints match these values depending on \code{jwks_pin_mode}.
Use to reduce key substitution risks by pre-authorizing expected keys}

\item{jwks_pin_mode}{Pinning policy when \code{jwks_pins} is provided. Either
"any" (default; at least one key in JWKS must match) or "all" (every
RSA/EC/OKP public key in JWKS must match one of the configured pins)}

\item{jwks_host_issuer_match}{When TRUE, enforce that the discovery \code{jwks_uri} host
matches the issuer host (or a subdomain). Defaults to FALSE at the class
level, but helper constructors for OIDC (e.g., \code{\link[=oauth_provider_oidc]{oauth_provider_oidc()}} and
\code{\link[=oauth_provider_oidc_discover]{oauth_provider_oidc_discover()}}) enable this by default for safer config.
The generic helper \code{\link[=oauth_provider]{oauth_provider()}} will also automatically set this to
TRUE when an \code{issuer} is provided and either \code{id_token_validation} or
\code{id_token_required} is TRUE (OIDC-like configuration). Set explicitly to
FALSE to opt out. For providers that legitimately publish JWKS on a
different host (e.g., Google), prefer setting \code{jwks_host_allow_only} to
the exact hostname rather than disabling this check}

\item{jwks_host_allow_only}{Optional explicit hostname that the jwks_uri must match.
When provided, jwks_uri host must equal this value (exact match). You can
pass either just the host (e.g., "www.googleapis.com") or a full URL; only
the host component will be used. If you need to include a port or an IPv6
literal, pass a full URL (e.g., \verb{https://[::1]:8443}) — the port is ignored
and only the hostname part is used for matching. Takes precedence over
\code{jwks_host_issuer_match}}

\item{allowed_algs}{Optional vector of allowed JWT algorithms for ID tokens.
Use to restrict acceptable \code{alg} values on a per-provider basis. Supported
asymmetric algorithms include \code{RS256}, \code{RS384}, \code{RS512}, \code{PS256}, \code{PS384},
\code{PS512}, \code{ES256}, \code{ES384}, \code{ES512}, and \code{EdDSA} (Ed25519/Ed448 via OKP).
Symmetric HMAC algorithms \code{HS256}, \code{HS384}, \code{HS512} are also supported but
require that you supply a \code{client_secret} and explicitly enable HMAC
verification via the option \code{options(shinyOAuth.allow_hs = TRUE)}.
Defaults to \code{c("RS256","RS384","RS512","PS256","PS384","PS512", "ES256","ES384","ES512","EdDSA")}, which intentionally excludes HS*.
Only include \verb{HS*} if you are certain the \code{client_secret} is stored strictly
server-side and is never shipped to, or derivable by, the browser or other
untrusted environments. Prefer rotating secrets regularly when enabling this.}

\item{allowed_token_types}{Character vector of acceptable OAuth token types
returned by the token endpoint (case-insensitive). When non-empty, the
token response MUST include \code{token_type} and it must be one of the allowed
values; otherwise the flow fails fast with a \code{shinyOAuth_token_error}.
When empty, no check is performed and \code{token_type} may be omitted by the
provider. The \code{\link[=oauth_provider]{oauth_provider()}} helper defaults to \code{c("Bearer")} for all
providers because the package only supports Bearer tokens (i.e.,
\code{\link[=client_bearer_req]{client_bearer_req()}} sends \code{Authorization: Bearer}). This ensures that if
a provider returns a non-Bearer token type (e.g., DPoP, MAC), the flow
fails fast rather than misusing the token. Set \code{allowed_token_types = character()} explicitly to opt out of enforcement.}

\item{leeway}{Clock skew leeway (seconds) applied to ID token \code{exp}/\code{iat}/\code{nbf} checks
and state payload \code{issued_at} future check. Default 30. Can be globally
overridden via option \code{shinyOAuth.leeway}}
}
\value{
\link{OAuthProvider} object
}
\description{
Helper function to create an \link{OAuthProvider} object.
This function provides sensible defaults and infers
some settings based on the provided parameters.
}
\examples{
# Configure generic OAuth 2.0 provider (no OIDC)
generic_provider <- oauth_provider(
  name = "example",
  auth_url = "https://example.com/oauth/authorize",
  token_url = "https://example.com/oauth/token",
  # Optional URL for fetching user info:
  userinfo_url = "https://example.com/oauth/userinfo"
)

# Configure generic OIDC provider manually
# (This defaults to using nonce & ID token validation)
generic_oidc_provider <- oauth_provider_oidc(
  name = "My OIDC",
  base_url = "https://my-issuer.example.com"
)

# Configure a OIDC provider via OIDC discovery
# (requires network access)
\donttest{
# Using Auth0 sample issuer as an example
oidc_discovery_provider <- oauth_provider_oidc_discover(
  issuer = "https://samples.auth0.com"
)
}

# GitHub preconfigured provider 
github_provider <- oauth_provider_github()

# Google preconfigured provider
google_provider <- oauth_provider_google()

# Microsoft preconfigured provider
# See `?oauth_provider_microsoft` for example using a custom tenant ID

# Spotify preconfigured provider 
spotify_provider <- oauth_provider_spotify()

# Slack via OIDC discovery
# (requires network access)
\donttest{
slack_provider <- oauth_provider_slack()
}
  
# Keycloak
# (requires configured Keycloak realm; example below is therefore not run)
\dontrun{
oauth_provider_keycloak(base_url = "http://localhost:8080", realm = "myrealm")
}

# Auth0
# (requires configured Auth0 domain; example below is therefore not run)
\dontrun{
oauth_provider_auth0(domain = "your-tenant.auth0.com")
}

# Okta
# (requires configured Okta domain; example below is therefore not run)
\dontrun{
oauth_provider_okta(domain = "dev-123456.okta.com")
}
}
