#' File I/O for profiler data
#'
#' @description
#' These functions read profile data from files, or store profile data to files.
#' Readers call [validate_profile()] on input, writers on output.
#'
#' `read_rprof()` reads a file generated by [Rprof()], `write_rprof()` writes
#' in a compatible format.
#'
#' @param path File name
#' @param ... Ignored
#' @param version Version of the data, currently only `"1.0"` is supported.
#'   Pass an explicit value to this argument if your code depends on the data
#'   format.
#' @export
#' @examples
#' rprof_file <- system.file("samples/rprof/1.out", package = "profile")
#' ds <- read_rprof(rprof_file)
#' ds
#' if (requireNamespace("RProtoBuf", quietly = TRUE)) {
#'   pprof_file <- tempfile("profile", fileext = ".pb.gz")
#'   write_pprof(ds, pprof_file)
#' }
read_rprof <- function(path, ..., version = "1.0") {
  stopifnot(version == get_default_meta()$value)
  rprof_ll <- read_rprof_ll(path)
  ds <- rprof_to_ds(rprof_ll)
  validate_profile(ds)
  ds
}

read_rprof_ll <- function(path) {
  lines <- readLines(path)

  header <- 1L
  files <- grep("^#File ", lines)
  traces <- setdiff(seq_along(lines), c(header, files))
  list(
    header = lines[header],
    files = lines[files],
    traces = lines[traces]
  )
}
