% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/object_ops.R
\name{export_report}
\alias{export_report}
\title{Export an \emph{agent}, \emph{informant}, \emph{multiagent}, or table scan to HTML}
\usage{
export_report(x, filename, path = NULL, quiet = FALSE)
}
\arguments{
\item{x}{\emph{One of several types of objects}

\verb{<object>} // \strong{required}

An \emph{agent} object of class \code{ptblank_agent}, an \emph{informant} of class
\code{ptblank_informant}, a \emph{multiagent} of class \code{ptblank_multiagent}, a table
scan of class \code{ptblank_tbl_scan}, or, customized reporting objects
(\code{ptblank_agent_report}, \code{ptblank_informant_report},
\code{ptblank_multiagent_report.wide}, \code{ptblank_multiagent_report.long}).}

\item{filename}{\emph{File name}

\verb{scalar<character>} // \strong{required}

The filename to create on disk for the HTML export of the object provided.
It's recommended that the extension \code{".html"} is included.}

\item{path}{\emph{File path}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional path to which the file should be saved (this is automatically
combined with \code{filename}).}

\item{quiet}{\emph{Inform (or not) upon file writing}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

Should the function \emph{not} inform when the file is written?}
}
\value{
Invisibly returns \code{TRUE} if the file has been written.
}
\description{
The \emph{agent}, \emph{informant}, \emph{multiagent}, and the table scan object can be
easily written as HTML with \code{export_report()}. Furthermore, any report
objects from the \emph{agent}, \emph{informant}, and \emph{multiagent} (generated using
\code{\link[=get_agent_report]{get_agent_report()}}, \code{\link[=get_informant_report]{get_informant_report()}}, and
\code{\link[=get_multiagent_report]{get_multiagent_report()}}) can be provided here for HTML export. Each HTML
document written to disk is self-contained and easily viewable in a web
browser.
}
\section{Examples}{

\subsection{A: Writing an agent report as HTML}{

Let's go through the process of (1) developing an agent with a validation
plan (to be used for the data quality analysis of the \code{\link{small_table}}
dataset), (2) interrogating the agent with the \code{\link[=interrogate]{interrogate()}} function, and
(3) writing the agent and all its intel to a file.

Creating an \code{action_levels} object is a common workflow step when creating a
pointblank agent. We designate failure thresholds to the \code{warn}, \code{stop}, and
\code{notify} states using \code{\link[=action_levels]{action_levels()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )
}\if{html}{\out{</div>}}

Now create a pointblank \code{agent} object and give it the \code{al} object (which
serves as a default for all validation steps which can be overridden). The
data will be referenced in the \code{tbl} argument with a leading \code{~}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <- 
  create_agent(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`export_report()`",
    actions = al
  )
}\if{html}{\out{</div>}}

As with any agent object, we can add steps to the validation plan by using as
many validation functions as we want. Then, we \code{\link[=interrogate]{interrogate()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <-
  agent \%>\% 
  col_exists(columns = c(date, date_time)) \%>\%
  col_vals_regex(
    columns = b,
    regex = "[0-9]-[a-z]\{3\}-[0-9]\{3\}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(columns = d, value = 100) \%>\%
  col_vals_lte(columns = c, value = 5) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

The agent report can be written to an HTML file with \code{export_report()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{export_report(
  agent,
  filename = "agent-small_table.html"
)
}\if{html}{\out{</div>}}

If you're consistently writing agent reports when periodically checking data,
we could make use of \code{affix_date()} or \code{affix_datetime()} depending on the
granularity you need. Here's an example that writes the file with the format:
\code{"<filename>-YYYY-mm-dd_HH-MM-SS.html"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{export_report(
  agent,
  filename = affix_datetime(
    "agent-small_table.html"
  )
)
}\if{html}{\out{</div>}}
}

\subsection{B: Writing an informant report as HTML}{

Let's go through the process of (1) creating an informant object that
minimally describes the \code{\link{small_table}} dataset, (2) ensuring that data is
captured from the target table using the \code{\link[=incorporate]{incorporate()}} function, and (3)
writing the informant report to HTML.

Create a pointblank \code{informant} object with \code{\link[=create_informant]{create_informant()}} and the
\code{\link{small_table}} dataset. Use \code{\link[=incorporate]{incorporate()}} so that info snippets are
integrated into the text.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{informant <- 
  create_informant(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`export_report()`"
  ) \%>\%
  info_snippet(
    snippet_name = "high_a",
    fn = snip_highest(column = "a")
  ) \%>\%
  info_snippet(
    snippet_name = "low_a",
    fn = snip_lowest(column = "a")
  ) \%>\%
  info_columns(
    columns = a,
    info = "From \{low_a\} to \{high_a\}."
  ) \%>\%
  info_columns(
    columns = starts_with("date"),
    info = "Time-based values."
  ) \%>\%
  info_columns(
    columns = date,
    info = "The date part of `date_time`."
  ) \%>\%
  incorporate()
}\if{html}{\out{</div>}}

The informant report can be written to an HTML file with \code{export_report()}.
Let's do this with \code{\link[=affix_date]{affix_date()}} so the filename has a datestamp.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{export_report(
  informant,
  filename = affix_date(
    "informant-small_table.html"
  )
)
}\if{html}{\out{</div>}}
}

\subsection{C: Writing a table scan as HTML}{

We can get a report that describes all of the data in the \code{storms} dataset.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl_scan <- scan_data(tbl = dplyr::storms)
}\if{html}{\out{</div>}}

The table scan object can be written to an HTML file with \code{export_report()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{export_report(
  tbl_scan,
  filename = "tbl_scan-storms.html"
)
}\if{html}{\out{</div>}}
}
}

\section{Function ID}{

9-3
}

\seealso{
Other Object Ops: 
\code{\link{activate_steps}()},
\code{\link{deactivate_steps}()},
\code{\link{remove_steps}()},
\code{\link{set_tbl}()},
\code{\link{x_read_disk}()},
\code{\link{x_write_disk}()}
}
\concept{Object Ops}
