% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/layerviewcontrol.R
\name{customizeLayersControl}
\alias{customizeLayersControl}
\title{Customize Layers Control of a Leaflet Map}
\usage{
customizeLayersControl(
  map,
  view_settings,
  home_btns = FALSE,
  home_btn_options = list(),
  setviewonselect = TRUE,
  opacityControl = list(),
  includelegends = TRUE,
  addCollapseButton = FALSE,
  layersControlCSS = list(),
  increaseOpacityOnHover = FALSE
)
}
\arguments{
\item{map}{A \code{leaflet} or \code{mapview} object to which the extended layers control will be added.}

\item{view_settings}{A list specifying the view settings for each layer. Each list element should contain
either:
\itemize{
\item \code{coords}: A vector of length 2 (latitude, longitude) for setting the view, or length 4
(bounding box: lat1, lng1, lat2, lng2) for fitting the bounds.
\item \code{zoom}: The zoom level (used for \code{setView}).
\item \code{fly} (optional): A logical indicating whether to use \code{flyTo} or \code{flyToBounds} instead of \code{setView} or \code{fitBounds}.
\item \code{options} (optional): Additional options to pass to \code{setView}, \code{fitBounds}, or \code{flyTo}.
}}

\item{home_btns}{Logical. If \code{TRUE}, adds a "home" button next to each layer name in the layer control.
Clicking the home button zooms the map to the view specified for that layer in \code{view_settings}.}

\item{home_btn_options}{A list of options to customize the home button appearance and behavior.
Possible options include:
\itemize{
\item \code{text}: The text or emoji to display on the button (default is \code{fontawesome::fa("home")}).
\item \code{cursor}: CSS cursor style for the button (default is 'pointer').
\item \code{class}: CSS class name for the button (default is 'leaflet-home-btn').
\item \code{styles}: Semicolon separated CSS-string (default is 'float: inline-end;').
}}

\item{setviewonselect}{Logical. If \code{TRUE} (default) sets the view when the layer is selected.}

\item{opacityControl}{A list specifying the opacity control settings for each layer. Each list element should contain:
\itemize{
\item \code{min}: Minimum opacity value (default is 0).
\item \code{max}: Maximum opacity value (default is 1).
\item \code{step}: Step size for the opacity slider (default is 0.1).
\item \code{default}: Default opacity value (default is 1).
\item \code{width}: Width of the opacity slider (default is '100\%').
\item \code{class}: CSS class name for the slider (default is 'leaflet-opacity-slider').
}}

\item{includelegends}{Logical. If \code{TRUE} (default), appends legends to the layer control. Legends are matched
to layers by their group name. The legends need to be added with corresponding layer IDs.}

\item{addCollapseButton}{Logical. If True a button will be added on top of the
LayersControl which, when clicked, will expand/collapse the view. This is mainly
relevant when the original Control was set to \code{collapsed = FALSE}.}

\item{layersControlCSS}{a list of valid CSS key-value pairs to modify the appearance
of the layersControl.}

\item{increaseOpacityOnHover}{Logical. If \code{TRUE} the layersControl will be fully opaque
when hovered. Mainly relevant if \code{opacity} was adjusted in \code{layersControlCSS}.}
}
\value{
A modified \code{leaflet} map object with extended layers control including view controls, home buttons, opacity controls, and legends.
}
\description{
This function enables customization of an existing layers control in a \code{leaflet} map by adding custom views, home buttons,
opacity controls, and legends. It enhances the functionality of a layers control created with \code{leaflet}
or \code{leaflet.extras}. It also allows to customize the layersControl appearance via CSS.
}
\details{
This function generates JavaScript that listens for \code{overlayadd} or \code{baselayerchange} events
and automatically sets the view or zoom level according to the specified \code{view_settings}.
If \code{home_btns} is enabled, a home button is added next to each layer in the layer control.
When clicked, it zooms the map to the predefined view of that layer.
The opacity control slider allows users to adjust the opacity of layers. The legend will be appended
to the corresponding layer control, matched by the layer's group name.
}
\examples{
library(sf)
library(leaflet)
library(leafem)

# Example data ##########
breweries91 <- st_as_sf(breweries91)
lines <- st_as_sf(atlStorms2005)
polys <- st_as_sf(gadmCHE)

# View settings ##########
view_settings <- list(
  "Base_tiles1" = list(
    coords = c(20, 50),
    zoom = 3
  ),
  "Base_tiles2" = list(
    coords = c(-110, 50),
    zoom = 5
  ),
  "breweries91" = list(
    coords = as.numeric(st_coordinates(st_centroid(st_union(breweries91)))),
    zoom = 8
  ),
  "atlStorms2005" = list(
    coords = as.numeric(st_bbox(lines)),
    options = list(padding = c(110, 110))
  ),
  "gadmCHE" = list(
    coords = as.numeric(st_bbox(polys)),
    options = list(padding = c(2, 2)),
    fly = TRUE
  )
)

# Opacity control settings ##########
opacityControl <- list(
  "breweries91" = list(
    min = 0,
    max = 1,
    step = 0.1,
    default = 1,
    width = '100\%',
    class = 'opacity-slider'
  )
)

# Legends ##########
legends <- list(
  "breweries91" = "<div>Legend for breweries</div>"
)

leaflet() \%>\%
  ## Baselayer
  addTiles(group = "Base_tiles1") \%>\%
  addProviderTiles("CartoDB", group = "Base_tiles2") \%>\%

  ## Overlays
  addCircleMarkers(data = breweries91, group = "breweries91") \%>\%
  addPolylines(data = lines, group = "atlStorms2005") \%>\%
  addPolygons(data = polys, group = "gadmCHE") \%>\%

  ## LayersControl
  addLayersControl(
    baseGroups = c("Base_tiles1", "Base_tiles2"),
    overlayGroups = c("breweries91", "atlStorms2005", "gadmCHE"),
    options = layersControlOptions(collapsed = FALSE, autoZIndex = TRUE)
  ) \%>\%

  ## Customize Layers Control
  customizeLayersControl(
    view_settings = view_settings,
    home_btns = TRUE,
    home_btn_options = list(
      "Base_tiles1" = list(
        text = fontawesome::fa("home"),
        cursor = 'ns-resize',
        class = 'homebtn'
      ),
      "Base_tiles2" = list(
        text = fontawesome::fa("home"),
        cursor = 'pointer'
      ),
      "atlStorms2005" = list(
        text = fontawesome::fa("wind"),
        cursor = 'all-scroll'
      ),
      "breweries91" = list(
        text = fontawesome::fa("beer-mug-empty"),
        styles = 'background-color: lightgreen; float: inline-end'
      ),
      "gadmCHE" = list(
        text = fontawesome::fa("mountain"),
        styles = 'float: none;'
      )
    ),
    opacityControl = opacityControl,
    includelegends = TRUE,
    addCollapseButton = TRUE,
    layersControlCSS = list("opacity" = 0.6),
    increaseOpacityOnHover = TRUE
  )


}
