% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimizers-schedules.R
\name{learning_rate_schedule_exponential_decay}
\alias{learning_rate_schedule_exponential_decay}
\title{A \code{LearningRateSchedule} that uses an exponential decay schedule.}
\usage{
learning_rate_schedule_exponential_decay(
  initial_learning_rate,
  decay_steps,
  decay_rate,
  staircase = FALSE,
  name = "ExponentialDecay"
)
}
\arguments{
\item{initial_learning_rate}{A float. The initial learning rate.}

\item{decay_steps}{A integer. Must be positive. See the decay
computation above.}

\item{decay_rate}{A float. The decay rate.}

\item{staircase}{Boolean.  If \code{TRUE} decay the learning rate at discrete
intervals.}

\item{name}{String.  Optional name of the operation.  Defaults to
\verb{"ExponentialDecay}".}
}
\value{
A 1-arg callable learning rate schedule that takes the current optimizer
step and outputs the decayed learning rate, a scalar tensor of the
same type as \code{initial_learning_rate}.
}
\description{
When training a model, it is often useful to lower the learning rate as
the training progresses. This schedule applies an exponential decay function
to an optimizer step, given a provided initial learning rate.

The schedule is a 1-arg callable that produces a decayed learning
rate when passed the current optimizer step. This can be useful for changing
the learning rate value across different invocations of optimizer functions.
It is computed as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step) \{
  initial_learning_rate * decay_rate ^ (step / decay_steps)
\}
}\if{html}{\out{</div>}}

If the argument \code{staircase} is \code{TRUE}, then \code{step / decay_steps} is
an integer division and the decayed learning rate follows a
staircase function.

You can pass this schedule directly into a \code{optimizer}
as the learning rate.
}
\section{Examples}{
When fitting a Keras model, decay every 100000 steps with a base
of 0.96:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{initial_learning_rate <- 0.1
lr_schedule <- learning_rate_schedule_exponential_decay(
    initial_learning_rate,
    decay_steps=100000,
    decay_rate=0.96,
    staircase=TRUE)

model \%>\% compile(
  optimizer = optimizer_sgd(learning_rate = lr_schedule),
  loss = 'sparse_categorical_crossentropy',
  metrics = c('accuracy'))

model \%>\% fit(data, labels, epochs=5)
}\if{html}{\out{</div>}}

The learning rate schedule is also serializable and deserializable using
\code{keras$optimizers$schedules$serialize} and
\code{keras$optimizers$schedules$deserialize}.
}

\seealso{
\itemize{
\item \url{https://keras.io/api/optimizers/learning_rate_schedules/exponential_decay#exponentialdecay-class}
}

Other optimizer learning rate schedules: \cr
\code{\link{LearningRateSchedule}()} \cr
\code{\link{learning_rate_schedule_cosine_decay}()} \cr
\code{\link{learning_rate_schedule_cosine_decay_restarts}()} \cr
\code{\link{learning_rate_schedule_inverse_time_decay}()} \cr
\code{\link{learning_rate_schedule_piecewise_constant_decay}()} \cr
\code{\link{learning_rate_schedule_polynomial_decay}()} \cr
}
\concept{optimizer learning rate schedules}
