/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Pprivate.h"  
#include "H5VLprivate.h" 

static hid_t  H5G__create_api_common(hid_t loc_id, const char *name, hid_t lcpl_id, hid_t gcpl_id,
                                     hid_t gapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr);
static hid_t  H5G__open_api_common(hid_t loc_id, const char *name, hid_t gapl_id, void **token_ptr,
                                   H5VL_object_t **_vol_obj_ptr);
static herr_t H5G__get_info_api_common(hid_t loc_id, H5G_info_t *group_info , void **token_ptr,
                                       H5VL_object_t **_vol_obj_ptr);
static herr_t H5G__get_info_by_name_api_common(hid_t loc_id, const char *name, H5G_info_t *group_info ,
                                               hid_t lapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr);
static herr_t H5G__get_info_by_idx_api_common(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                              H5_iter_order_t order, hsize_t n,
                                              H5G_info_t *group_info , hid_t lapl_id, void **token_ptr,
                                              H5VL_object_t **_vol_obj_ptr);

static hid_t
H5G__create_api_common(hid_t loc_id, const char *name, hid_t lcpl_id, hid_t gcpl_id, hid_t gapl_id,
                       void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    void           *grp         = NULL; 
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_loc_params_t loc_params;                     
    hid_t             ret_value = H5I_INVALID_HID;    

    FUNC_ENTER_PACKAGE

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    if (H5VL_setup_acc_args(loc_id, H5P_CLS_GACC, true, &gapl_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");

    
    if (H5P_DEFAULT == lcpl_id)
        lcpl_id = H5P_LINK_CREATE_DEFAULT;
    else if (true != H5P_isa_class(lcpl_id, H5P_LINK_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a link creation property list");

    
    if (H5P_DEFAULT == gcpl_id)
        gcpl_id = H5P_GROUP_CREATE_DEFAULT;
    else if (true != H5P_isa_class(gcpl_id, H5P_GROUP_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a group creation property list");

    
    H5CX_set_lcpl(lcpl_id);

    
    if (NULL == (grp = H5VL_group_create(*vol_obj_ptr, &loc_params, name, lcpl_id, gcpl_id, gapl_id,
                                         H5P_DATASET_XFER_DEFAULT, token_ptr)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5I_INVALID_HID, "unable to create group");

    
    if ((ret_value = H5VL_register(H5I_GROUP, grp, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to get ID for group handle");

done:
    if (H5I_INVALID_HID == ret_value)
        if (grp && H5VL_group_close(*vol_obj_ptr, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Gcreate2(hid_t loc_id, const char *name, hid_t lcpl_id, hid_t gcpl_id, hid_t gapl_id)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5G__create_api_common(loc_id, name, lcpl_id, gcpl_id, gapl_id, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, H5I_INVALID_HID, "unable to synchronously create group");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Gcreate_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
                hid_t lcpl_id, hid_t gcpl_id, hid_t gapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5G__create_api_common(loc_id, name, lcpl_id, gcpl_id, gapl_id, token_ptr, &vol_obj)) <
        0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, H5I_INVALID_HID, "unable to asynchronously create group");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE9(__func__, "*s*sIui*siiii", app_file, app_func, app_line, loc_id, name, lcpl_id, gcpl_id, gapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on group ID");
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Gcreate_anon(hid_t loc_id, hid_t gcpl_id, hid_t gapl_id)
{
    void             *grp     = NULL;              
    H5VL_object_t    *vol_obj = NULL;              
    H5VL_loc_params_t loc_params;                  
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5P_DEFAULT == gcpl_id)
        gcpl_id = H5P_GROUP_CREATE_DEFAULT;
    else if (true != H5P_isa_class(gcpl_id, H5P_GROUP_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not group create property list");

    if (H5P_DEFAULT == gapl_id)
        gapl_id = H5P_GROUP_ACCESS_DEFAULT;
    else if (true != H5P_isa_class(gapl_id, H5P_GROUP_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not group access property list");

    
    if (H5CX_set_apl(&gapl_id, H5P_CLS_GACC, loc_id, true) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, H5I_INVALID_HID, "can't set access property list info");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL == (grp = H5VL_group_create(vol_obj, &loc_params, NULL, H5P_LINK_CREATE_DEFAULT, gcpl_id,
                                         gapl_id, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5I_INVALID_HID, "unable to create group");

    
    if ((ret_value = H5VL_register(H5I_GROUP, grp, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to get ID for group handle");

done:
    
    if (H5I_INVALID_HID == ret_value)
        if (grp && H5VL_group_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release group");

    FUNC_LEAVE_API(ret_value)
} 

static hid_t
H5G__open_api_common(hid_t loc_id, const char *name, hid_t gapl_id, void **token_ptr,
                     H5VL_object_t **_vol_obj_ptr)
{
    void           *grp         = NULL; 
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_loc_params_t loc_params;                     
    hid_t             ret_value = H5I_INVALID_HID;    

    FUNC_ENTER_PACKAGE

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    if (H5VL_setup_acc_args(loc_id, H5P_CLS_GACC, false, &gapl_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");

    if (NULL == (grp = H5VL_group_open(*vol_obj_ptr, &loc_params, name, gapl_id, H5P_DATASET_XFER_DEFAULT,
                                       token_ptr)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open group");

    
    if ((ret_value = H5VL_register(H5I_GROUP, grp, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register group");

done:
    if (H5I_INVALID_HID == ret_value)
        if (grp && H5VL_group_close(*vol_obj_ptr, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Gopen2(hid_t loc_id, const char *name, hid_t gapl_id)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5G__open_api_common(loc_id, name, gapl_id, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, H5I_INVALID_HID, "unable to synchronously open group");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Gopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
              hid_t gapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5G__open_api_common(loc_id, name, gapl_id, token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, H5I_INVALID_HID, "unable to asynchronously open group");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE7(__func__, "*s*sIui*sii", app_file, app_func, app_line, loc_id, name, gapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on group ID");
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Gget_create_plist(hid_t group_id)
{
    H5VL_object_t        *vol_obj;     
    H5VL_group_get_args_t vol_cb_args; 
    hid_t                 ret_value = H5I_INVALID_HID;

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(group_id, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a group ID");

    
    vol_cb_args.op_type               = H5VL_GROUP_GET_GCPL;
    vol_cb_args.args.get_gcpl.gcpl_id = H5I_INVALID_HID;

    
    if (H5VL_group_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5I_INVALID_HID, "can't get group's creation property list");

    
    ret_value = vol_cb_args.args.get_gcpl.gcpl_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5G__get_info_api_common(hid_t loc_id, H5G_info_t *group_info , void **token_ptr,
                         H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_group_get_args_t vol_cb_args;                
    H5I_type_t            id_type;                    
    herr_t                ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    id_type = H5I_get_type(loc_id);
    if (!(H5I_GROUP == id_type || H5I_FILE == id_type))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid group (or file) ID");
    if (!group_info)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "group_info parameter cannot be NULL");

    
    vol_cb_args.op_type = H5VL_GROUP_GET_INFO;
    if (H5VL_setup_self_args(loc_id, vol_obj_ptr, &vol_cb_args.args.get_info.loc_params) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set object access arguments");
    vol_cb_args.args.get_info.ginfo = group_info;

    
    if (H5VL_group_get(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get group info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Gget_info(hid_t loc_id, H5G_info_t *group_info )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5G__get_info_api_common(loc_id, group_info, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to synchronously get group info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gget_info_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                  H5G_info_t *group_info , hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5G__get_info_api_common(loc_id, group_info, token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to asynchronously get group info");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                H5ARG_TRACE6(__func__, "*s*sIui*GIi", app_file, app_func, app_line, loc_id, group_info, es_id)) < 0)
            
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5G__get_info_by_name_api_common(hid_t loc_id, const char *name, H5G_info_t *group_info ,
                                 hid_t lapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_group_get_args_t vol_cb_args;                
    herr_t                ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    if (!group_info)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "group_info parameter cannot be NULL");

    
    vol_cb_args.op_type = H5VL_GROUP_GET_INFO;
    if (H5VL_setup_name_args(loc_id, name, false, lapl_id, vol_obj_ptr,
                             &vol_cb_args.args.get_info.loc_params) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set object access arguments");
    vol_cb_args.args.get_info.ginfo = group_info;

    
    if (H5VL_group_get(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get group info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Gget_info_by_name(hid_t loc_id, const char *name, H5G_info_t *group_info , hid_t lapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5G__get_info_by_name_api_common(loc_id, name, group_info, lapl_id, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't synchronously retrieve group info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gget_info_by_name_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                          const char *name, H5G_info_t *group_info , hid_t lapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5G__get_info_by_name_api_common(loc_id, name, group_info, lapl_id, token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't asynchronously retrieve group info");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE8(__func__, "*s*sIui*s*GIii", app_file, app_func, app_line, loc_id, name, group_info, lapl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5G__get_info_by_idx_api_common(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                H5_iter_order_t order, hsize_t n, H5G_info_t *group_info ,
                                hid_t lapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_group_get_args_t vol_cb_args;                
    herr_t                ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    if (!group_info)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "group_info parameter cannot be NULL");

    
    vol_cb_args.op_type = H5VL_GROUP_GET_INFO;
    if (H5VL_setup_idx_args(loc_id, group_name, idx_type, order, n, false, lapl_id, vol_obj_ptr,
                            &vol_cb_args.args.get_info.loc_params) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set object access arguments");
    vol_cb_args.args.get_info.ginfo = group_info;

    
    if (H5VL_group_get(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get group info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Gget_info_by_idx(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                   hsize_t n, H5G_info_t *group_info , hid_t lapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5G__get_info_by_idx_api_common(loc_id, group_name, idx_type, order, n, group_info, lapl_id, NULL,
                                        NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't synchronously retrieve group info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gget_info_by_idx_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                         const char *group_name, H5_index_t idx_type, H5_iter_order_t order, hsize_t n,
                         H5G_info_t *group_info , hid_t lapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5G__get_info_by_idx_api_common(loc_id, group_name, idx_type, order, n, group_info, lapl_id,
                                        token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't asynchronously retrieve group info");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE11(__func__, "*s*sIui*sIiIoh*GIii", app_file, app_func, app_line, loc_id, group_name, idx_type, order, n, group_info, lapl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gclose(hid_t group_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GROUP != H5I_get_type(group_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group ID");

    
    if (H5I_dec_app_ref(group_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDEC, FAIL, "decrementing group ID failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t group_id, hid_t es_id)
{
    H5VL_object_t    *vol_obj   = NULL;            
    H5VL_connector_t *connector = NULL;            
    void             *token     = NULL;            
    void            **token_ptr = H5_REQUEST_NULL; 
    herr_t            ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5I_GROUP != H5I_get_type(group_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group ID");

    
    if (H5ES_NONE != es_id) {
        
        if (NULL == (vol_obj = H5VL_vol_object(group_id)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get VOL object for group");

        
        connector = H5VL_OBJ_CONNECTOR(vol_obj);
        H5VL_conn_inc_rc(connector);

        
        token_ptr = &token;
    } 

    
    if (H5I_dec_app_ref_async(group_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDEC, FAIL, "decrementing group ID failed");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, group_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    if (connector && H5VL_conn_dec_rc(connector) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTDEC, FAIL, "can't decrement ref count on connector");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Gflush(hid_t group_id)
{
    H5VL_object_t             *vol_obj;             
    H5VL_group_specific_args_t vol_cb_args;         
    herr_t                     ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(group_id, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group ID");

    
    if (H5CX_set_loc(group_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    vol_cb_args.op_type           = H5VL_GROUP_FLUSH;
    vol_cb_args.args.flush.grp_id = group_id;

    
    if (H5VL_group_specific(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTFLUSH, FAIL, "unable to flush group");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Grefresh(hid_t group_id)
{
    H5VL_object_t             *vol_obj;             
    H5VL_group_specific_args_t vol_cb_args;         
    herr_t                     ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(group_id, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group ID");

    
    if (H5CX_set_loc(group_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    vol_cb_args.op_type             = H5VL_GROUP_REFRESH;
    vol_cb_args.args.refresh.grp_id = group_id;

    
    if (H5VL_group_specific(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTLOAD, FAIL, "unable to refresh group");

done:
    FUNC_LEAVE_API(ret_value)
} 
