/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 
#define H5D_FRIEND     

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5Ppkg.h"      
#include "H5Sprivate.h"  
#include "H5Tprivate.h"  
#include "H5VMprivate.h" 
#include "H5Zprivate.h"  

#define H5D_DEF_STORAGE_COMPACT_INIT                                                                         \
    {                                                                                                        \
        false, (size_t)0, NULL                                                                               \
    }
#define H5D_DEF_STORAGE_CONTIG_INIT                                                                          \
    {                                                                                                        \
        HADDR_UNDEF, (hsize_t)0                                                                              \
    }
#define H5D_DEF_STORAGE_CHUNK_INIT                                                                           \
    {                                                                                                        \
        H5D_CHUNK_IDX_BTREE, HADDR_UNDEF, H5D_COPS_BTREE,                                                    \
        {                                                                                                    \
            {                                                                                                \
                HADDR_UNDEF, NULL                                                                            \
            }                                                                                                \
        }                                                                                                    \
    }
#define H5D_DEF_LAYOUT_CHUNK_INIT                                                                            \
    {                                                                                                        \
        H5D_CHUNK_IDX_BTREE, (uint8_t)0, (unsigned)0, {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,    \
                                                       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},      \
            (unsigned)0, (uint32_t)0, (hsize_t)0, (hsize_t)0,                                                \
            {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                                              \
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},                                                \
            {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                                              \
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},                                                \
            {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                                              \
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},                                                \
            {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                                              \
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},                                                \
        {                                                                                                    \
            {                                                                                                \
                {                                                                                            \
                    (uint8_t)0                                                                               \
                }                                                                                            \
            }                                                                                                \
        }                                                                                                    \
    }
#define H5D_DEF_STORAGE_VIRTUAL_INIT                                                                         \
    {                                                                                                        \
        {HADDR_UNDEF, 0}, 0, NULL, 0, {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                       \
                                       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},                      \
            H5D_VDS_ERROR, HSIZE_UNDEF, -1, -1, false, NULL, NULL, NULL, 0, 0, NULL,                         \
    }
#define H5D_DEF_STORAGE_COMPACT                                                                              \
    {                                                                                                        \
        H5D_COMPACT,                                                                                         \
        {                                                                                                    \
            .compact = H5D_DEF_STORAGE_COMPACT_INIT                                                          \
        }                                                                                                    \
    }
#define H5D_DEF_STORAGE_CONTIG                                                                               \
    {                                                                                                        \
        H5D_CONTIGUOUS,                                                                                      \
        {                                                                                                    \
            .contig = H5D_DEF_STORAGE_CONTIG_INIT                                                            \
        }                                                                                                    \
    }
#define H5D_DEF_STORAGE_CHUNK                                                                                \
    {                                                                                                        \
        H5D_CHUNKED,                                                                                         \
        {                                                                                                    \
            .chunk = H5D_DEF_STORAGE_CHUNK_INIT                                                              \
        }                                                                                                    \
    }
#define H5D_DEF_STORAGE_VIRTUAL                                                                              \
    {                                                                                                        \
        H5D_VIRTUAL,                                                                                         \
        {                                                                                                    \
            .virt = H5D_DEF_STORAGE_VIRTUAL_INIT                                                             \
        }                                                                                                    \
    }
#define H5D_DEF_LAYOUT_COMPACT                                                                               \
    {                                                                                                        \
        H5D_COMPACT, H5O_LAYOUT_VERSION_DEFAULT, H5D_LOPS_COMPACT, {H5D_DEF_LAYOUT_CHUNK_INIT},              \
            H5D_DEF_STORAGE_COMPACT                                                                          \
    }
#define H5D_DEF_LAYOUT_CONTIG                                                                                \
    {                                                                                                        \
        H5D_CONTIGUOUS, H5O_LAYOUT_VERSION_DEFAULT, H5D_LOPS_CONTIG, {H5D_DEF_LAYOUT_CHUNK_INIT},            \
            H5D_DEF_STORAGE_CONTIG                                                                           \
    }
#define H5D_DEF_LAYOUT_CHUNK                                                                                 \
    {                                                                                                        \
        H5D_CHUNKED, H5O_LAYOUT_VERSION_DEFAULT, H5D_LOPS_CHUNK, {H5D_DEF_LAYOUT_CHUNK_INIT},                \
            H5D_DEF_STORAGE_CHUNK                                                                            \
    }
#define H5D_DEF_LAYOUT_VIRTUAL                                                                               \
    {                                                                                                        \
        H5D_VIRTUAL, H5O_LAYOUT_VERSION_4, H5D_LOPS_VIRTUAL, {H5D_DEF_LAYOUT_CHUNK_INIT},                    \
            H5D_DEF_STORAGE_VIRTUAL                                                                          \
    }

#define H5D_CRT_LAYOUT_SIZE       sizeof(H5O_layout_t)
#define H5D_CRT_LAYOUT_DEF        H5D_DEF_LAYOUT_CONTIG
#define H5D_CRT_LAYOUT_SET        H5P__dcrt_layout_set
#define H5D_CRT_LAYOUT_GET        H5P__dcrt_layout_get
#define H5D_CRT_LAYOUT_ENC        H5P__dcrt_layout_enc
#define H5D_CRT_LAYOUT_DEC        H5P__dcrt_layout_dec
#define H5D_CRT_LAYOUT_DEL        H5P__dcrt_layout_del
#define H5D_CRT_LAYOUT_COPY       H5P__dcrt_layout_copy
#define H5D_CRT_LAYOUT_CMP        H5P__dcrt_layout_cmp
#define H5D_CRT_LAYOUT_CLOSE      H5P__dcrt_layout_close
#define H5D_CRT_LAYOUT_VERSION_0  0
#define H5D_CRT_LAYOUT_VERSION_1  1
#define H5D_CRT_LAYOUT_MAGIC_TYPE (uint8_t)0xff

#define H5D_CRT_FILL_VALUE_SIZE sizeof(H5O_fill_t)
#define H5D_CRT_FILL_VALUE_DEF                                                                               \
    {                                                                                                        \
        {0, NULL, H5O_NULL_ID, {{0, HADDR_UNDEF}}}, H5O_FILL_VERSION_2, NULL, 0, NULL, H5D_ALLOC_TIME_LATE,  \
            H5D_FILL_TIME_IFSET, false                                                                       \
    }
#define H5D_CRT_FILL_VALUE_SET   H5P__dcrt_fill_value_set
#define H5D_CRT_FILL_VALUE_GET   H5P__dcrt_fill_value_get
#define H5D_CRT_FILL_VALUE_ENC   H5P__dcrt_fill_value_enc
#define H5D_CRT_FILL_VALUE_DEC   H5P__dcrt_fill_value_dec
#define H5D_CRT_FILL_VALUE_DEL   H5P__dcrt_fill_value_del
#define H5D_CRT_FILL_VALUE_COPY  H5P__dcrt_fill_value_copy
#define H5D_CRT_FILL_VALUE_CMP   H5P_fill_value_cmp
#define H5D_CRT_FILL_VALUE_CLOSE H5P__dcrt_fill_value_close

#define H5D_CRT_ALLOC_TIME_STATE_SIZE sizeof(unsigned)
#define H5D_CRT_ALLOC_TIME_STATE_DEF  1
#define H5D_CRT_ALLOC_TIME_STATE_ENC  H5P__encode_unsigned
#define H5D_CRT_ALLOC_TIME_STATE_DEC  H5P__decode_unsigned

#define H5D_CRT_EXT_FILE_LIST_SIZE sizeof(H5O_efl_t)
#define H5D_CRT_EXT_FILE_LIST_DEF                                                                            \
    {                                                                                                        \
        HADDR_UNDEF, 0, 0, NULL                                                                              \
    }
#define H5D_CRT_EXT_FILE_LIST_SET   H5P__dcrt_ext_file_list_set
#define H5D_CRT_EXT_FILE_LIST_GET   H5P__dcrt_ext_file_list_get
#define H5D_CRT_EXT_FILE_LIST_ENC   H5P__dcrt_ext_file_list_enc
#define H5D_CRT_EXT_FILE_LIST_DEC   H5P__dcrt_ext_file_list_dec
#define H5D_CRT_EXT_FILE_LIST_DEL   H5P__dcrt_ext_file_list_del
#define H5D_CRT_EXT_FILE_LIST_COPY  H5P__dcrt_ext_file_list_copy
#define H5D_CRT_EXT_FILE_LIST_CMP   H5P__dcrt_ext_file_list_cmp
#define H5D_CRT_EXT_FILE_LIST_CLOSE H5P__dcrt_ext_file_list_close

#define H5D_CRT_MIN_DSET_HDR_SIZE_SIZE sizeof(bool)
#define H5D_CRT_MIN_DSET_HDR_SIZE_DEF  false
#define H5D_CRT_MIN_DSET_HDR_SIZE_ENC  H5P__encode_bool
#define H5D_CRT_MIN_DSET_HDR_SIZE_DEC  H5P__decode_bool

static herr_t H5P__set_layout(H5P_genplist_t *plist, const H5O_layout_t *layout);

static herr_t H5P__dcrt_reg_prop(H5P_genclass_t *pclass);

static herr_t H5P__dcrt_layout_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_layout_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_layout_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dcrt_layout_dec(const void **pp, void *value);
static herr_t H5P__dcrt_layout_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_layout_copy(const char *name, size_t size, void *value);
static int    H5P__dcrt_layout_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__dcrt_layout_close(const char *name, size_t size, void *value);
static herr_t H5P__dcrt_fill_value_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_fill_value_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_fill_value_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dcrt_fill_value_dec(const void **pp, void *value);
static herr_t H5P__dcrt_fill_value_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_fill_value_copy(const char *name, size_t size, void *value);
static herr_t H5P__dcrt_fill_value_close(const char *name, size_t size, void *value);
static herr_t H5P__dcrt_ext_file_list_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_ext_file_list_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_ext_file_list_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dcrt_ext_file_list_dec(const void **pp, void *value);
static herr_t H5P__dcrt_ext_file_list_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dcrt_ext_file_list_copy(const char *name, size_t size, void *value);
static int    H5P__dcrt_ext_file_list_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__dcrt_ext_file_list_close(const char *name, size_t size, void *value);

const H5P_libclass_t H5P_CLS_DCRT[1] = {{
    "dataset create",        
    H5P_TYPE_DATASET_CREATE, 

    &H5P_CLS_OBJECT_CREATE_g,     
    &H5P_CLS_DATASET_CREATE_g,    
    &H5P_CLS_DATASET_CREATE_ID_g, 
    &H5P_LST_DATASET_CREATE_ID_g, 
    H5P__dcrt_reg_prop,           

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

H5FL_BLK_EXTERN(type_conv);

static const H5O_layout_t H5D_def_layout_g = H5D_CRT_LAYOUT_DEF;     
static const H5O_fill_t   H5D_def_fill_g   = H5D_CRT_FILL_VALUE_DEF; 
static const unsigned     H5D_def_alloc_time_state_g =
    H5D_CRT_ALLOC_TIME_STATE_DEF;                                     
static const H5O_efl_t H5D_def_efl_g = H5D_CRT_EXT_FILE_LIST_DEF;     
static const unsigned H5O_ohdr_min_g = H5D_CRT_MIN_DSET_HDR_SIZE_DEF; 

static const H5O_layout_t H5D_def_layout_compact_g = H5D_DEF_LAYOUT_COMPACT;
static const H5O_layout_t H5D_def_layout_contig_g  = H5D_DEF_LAYOUT_CONTIG;
static const H5O_layout_t H5D_def_layout_chunk_g   = H5D_DEF_LAYOUT_CHUNK;
static const H5O_layout_t H5D_def_layout_virtual_g = H5D_DEF_LAYOUT_VIRTUAL;

static herr_t
H5P__dcrt_reg_prop(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5D_CRT_LAYOUT_NAME, H5D_CRT_LAYOUT_SIZE, &H5D_def_layout_g, NULL,
                           H5D_CRT_LAYOUT_SET, H5D_CRT_LAYOUT_GET, H5D_CRT_LAYOUT_ENC, H5D_CRT_LAYOUT_DEC,
                           H5D_CRT_LAYOUT_DEL, H5D_CRT_LAYOUT_COPY, H5D_CRT_LAYOUT_CMP,
                           H5D_CRT_LAYOUT_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_CRT_FILL_VALUE_NAME, H5D_CRT_FILL_VALUE_SIZE, &H5D_def_fill_g, NULL,
                           H5D_CRT_FILL_VALUE_SET, H5D_CRT_FILL_VALUE_GET, H5D_CRT_FILL_VALUE_ENC,
                           H5D_CRT_FILL_VALUE_DEC, H5D_CRT_FILL_VALUE_DEL, H5D_CRT_FILL_VALUE_COPY,
                           H5D_CRT_FILL_VALUE_CMP, H5D_CRT_FILL_VALUE_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_CRT_ALLOC_TIME_STATE_NAME, H5D_CRT_ALLOC_TIME_STATE_SIZE,
                           &H5D_def_alloc_time_state_g, NULL, NULL, NULL, H5D_CRT_ALLOC_TIME_STATE_ENC,
                           H5D_CRT_ALLOC_TIME_STATE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_CRT_EXT_FILE_LIST_NAME, H5D_CRT_EXT_FILE_LIST_SIZE, &H5D_def_efl_g,
                           NULL, H5D_CRT_EXT_FILE_LIST_SET, H5D_CRT_EXT_FILE_LIST_GET,
                           H5D_CRT_EXT_FILE_LIST_ENC, H5D_CRT_EXT_FILE_LIST_DEC, H5D_CRT_EXT_FILE_LIST_DEL,
                           H5D_CRT_EXT_FILE_LIST_COPY, H5D_CRT_EXT_FILE_LIST_CMP,
                           H5D_CRT_EXT_FILE_LIST_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_CRT_MIN_DSET_HDR_SIZE_NAME, H5D_CRT_MIN_DSET_HDR_SIZE_SIZE,
                           &H5O_ohdr_min_g, NULL, NULL, NULL, H5D_CRT_MIN_DSET_HDR_SIZE_ENC,
                           H5D_CRT_MIN_DSET_HDR_SIZE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                     size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_layout_t *layout = (H5O_layout_t *)value; 
    H5O_layout_t  new_layout;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (NULL == H5O_msg_copy(H5O_LAYOUT_ID, layout, &new_layout))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy layout");

    
    *layout = new_layout;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                     size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_layout_t *layout = (H5O_layout_t *)value; 
    H5O_layout_t  new_layout;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (NULL == H5O_msg_copy(H5O_LAYOUT_ID, layout, &new_layout))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy layout");

    
    *layout = new_layout;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_enc(const void *value, void **_pp, size_t *size)
{
    const H5O_layout_t *layout = (const H5O_layout_t *)value; 
    uint8_t           **pp     = (uint8_t **)_pp;
    uint8_t            *tmp_p;
    size_t              tmp_size;
    unsigned            version = H5D_CRT_LAYOUT_VERSION_0;
    H5F_libver_t        low_bound;
    H5F_libver_t        high_bound;
    size_t              u;                   
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(layout);
    assert(size);

    
    if (H5CX_get_libver_bounds(&low_bound, &high_bound) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get low/high bounds from API context");

    
    if (H5D_CHUNKED == layout->type)
        for (u = 0; u < (size_t)layout->u.chunk.ndims; u++)
            if (layout->u.chunk.dim[u] > (hsize_t)0xffffffff) {
                if (high_bound < H5F_LIBVER_V200)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL,
                                "can't encode chunk dimensions >= 2^32 with old encoding format - see "
                                "H5Pset_libver_bounds()");
                version = H5D_CRT_LAYOUT_VERSION_1;
            }

    if (NULL != *pp) {
        
        if (version > H5D_CRT_LAYOUT_VERSION_0) {
            *(*pp)++ = H5D_CRT_LAYOUT_MAGIC_TYPE;
            *(*pp)++ = (uint8_t)version;
        }

        
        *(*pp)++ = (uint8_t)layout->type;
        *size += sizeof(uint8_t);

        
        if (H5D_CHUNKED == layout->type) {
            
            *(*pp)++ = (uint8_t)layout->u.chunk.ndims;
            *size += sizeof(uint8_t);

            
            if (version >= H5D_CRT_LAYOUT_VERSION_1) {
                for (u = 0; u < (size_t)layout->u.chunk.ndims; u++) {
                    UINT64ENCODE(*pp, (uint64_t)layout->u.chunk.dim[u]);
                    *size += sizeof(uint64_t);
                } 
            }
            else
                for (u = 0; u < (size_t)layout->u.chunk.ndims; u++) {
                    UINT32ENCODE(*pp, (uint32_t)layout->u.chunk.dim[u]);
                    *size += sizeof(uint32_t);
                } 
        }         
        else if (H5D_VIRTUAL == layout->type) {
            uint64_t nentries = (uint64_t)layout->storage.u.virt.list_nused;

            
            UINT64ENCODE(*pp, nentries);
            *size += (size_t)8;

            
            for (u = 0; u < layout->storage.u.virt.list_nused; u++) {
                
                tmp_size = strlen(layout->storage.u.virt.list[u].source_file_name) + (size_t)1;
                H5MM_memcpy(*pp, layout->storage.u.virt.list[u].source_file_name, tmp_size);
                *pp += tmp_size;
                *size += tmp_size;

                
                tmp_size = strlen(layout->storage.u.virt.list[u].source_dset_name) + (size_t)1;
                H5MM_memcpy(*pp, layout->storage.u.virt.list[u].source_dset_name, tmp_size);
                *pp += tmp_size;
                *size += tmp_size;

                
                tmp_size = (size_t)-1;
                tmp_p    = *pp;
                if (H5S_encode(layout->storage.u.virt.list[u].source_select, pp, &tmp_size) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "unable to serialize source selection");
                *size += (size_t)(*pp - tmp_p);

                
                tmp_size = (size_t)-1;
                tmp_p    = *pp;
                if (H5S_encode(layout->storage.u.virt.list[u].source_dset.virtual_select, pp, &tmp_size) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "unable to serialize virtual selection");
                *size += (size_t)(*pp - tmp_p);
            } 
        }     
    }         
    else {
        
        *size += sizeof(uint8_t);

        
        if (H5D_CHUNKED == layout->type) {
            *size += sizeof(uint8_t);
            *size += layout->u.chunk.ndims *
                     ((version >= H5D_CRT_LAYOUT_VERSION_1) ? sizeof(uint64_t) : sizeof(uint32_t));
        } 
        else if (H5D_VIRTUAL == layout->type) {
            
            
            *size += (size_t)8;

            
            for (u = 0; u < layout->storage.u.virt.list_nused; u++) {
                
                tmp_size = strlen(layout->storage.u.virt.list[u].source_file_name) + (size_t)1;
                *size += tmp_size;

                
                tmp_size = strlen(layout->storage.u.virt.list[u].source_dset_name) + (size_t)1;
                *size += tmp_size;

                
                tmp_size = (size_t)0;
                tmp_p    = NULL;
                if (H5S_encode(layout->storage.u.virt.list[u].source_select, &tmp_p, &tmp_size) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "unable to serialize source selection");
                *size += tmp_size;

                
                tmp_size = (size_t)0;
                tmp_p    = NULL;
                if (H5S_encode(layout->storage.u.virt.list[u].source_dset.virtual_select, &tmp_p, &tmp_size) <
                    0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "unable to serialize virtual selection");
                *size += tmp_size;
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_dec(const void **_pp, void *value)
{
    const H5O_layout_t *layout;                               
    H5O_layout_t        tmp_layout;                           
    H5D_layout_t        type;                                 
    unsigned            version   = H5D_CRT_LAYOUT_VERSION_0; 
    const uint8_t     **pp        = (const uint8_t **)_pp;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(value);

    
    type = (H5D_layout_t) * (*pp)++;

    
    if (type == H5D_CRT_LAYOUT_MAGIC_TYPE) {
        version = (unsigned)*(*pp)++;
        type    = (H5D_layout_t) * (*pp)++;
    }

    
    switch (type) {
        case H5D_COMPACT:
            layout = &H5D_def_layout_compact_g;
            break;

        case H5D_CONTIGUOUS:
            layout = &H5D_def_layout_contig_g;
            break;

        case H5D_CHUNKED: {
            unsigned ndims; 

            
            ndims = *(*pp)++;

            
            if (0 == ndims)
                layout = &H5D_def_layout_chunk_g;
            else {          
                unsigned u; 

                
                tmp_layout = H5D_def_layout_chunk_g;

                
                tmp_layout.u.chunk.ndims = (unsigned)ndims;

                
                if (version >= H5D_CRT_LAYOUT_VERSION_1) {
                    for (u = 0; u < ndims; u++) {
                        uint64_t dim64;
                        UINT64DECODE(*pp, dim64);
                        tmp_layout.u.chunk.dim[u] = (hsize_t)dim64;
                        if ((uint64_t)tmp_layout.u.chunk.dim[u] != dim64)
                            HGOTO_ERROR(H5E_PLIST, H5E_BADRANGE, FAIL,
                                        "chunk dimension too big to fit in hsize_t");
                    }
                }
                else
                    for (u = 0; u < ndims; u++)
                        UINT32DECODE(*pp, tmp_layout.u.chunk.dim[u]);

                
                layout = &tmp_layout;
            } 
        } break;

        case H5D_VIRTUAL: {
            uint64_t nentries; 

            
            UINT64DECODE(*pp, nentries);

            if (nentries == (uint64_t)0)
                
                layout = &H5D_def_layout_virtual_g;
            else {
                size_t tmp_size;
                size_t u; 

                
                tmp_layout = H5D_def_layout_virtual_g;

                
                if (NULL == (tmp_layout.storage.u.virt.list = (H5O_storage_virtual_ent_t *)H5MM_calloc(
                                 (size_t)nentries * sizeof(H5O_storage_virtual_ent_t))))
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "unable to allocate heap block");
                tmp_layout.storage.u.virt.list_nalloc = (size_t)nentries;
                tmp_layout.storage.u.virt.list_nused  = (size_t)nentries;

                
                for (u = 0; u < (size_t)nentries; u++) {
                    H5O_storage_virtual_ent_t
                        *tmp_ent; 

                    
                    tmp_ent  = NULL;
                    tmp_size = strlen((const char *)*pp) + 1;
                    if (tmp_layout.storage.u.virt.list_nused > 0)
                        HASH_FIND(hh_source_file, tmp_layout.storage.u.virt.source_file_hash_table, *pp,
                                  tmp_size - 1, tmp_ent);
                    if (tmp_ent) {
                        
                        assert(tmp_ent >= tmp_layout.storage.u.virt.list &&
                               tmp_ent < &tmp_layout.storage.u.virt.list[u]);
                        tmp_layout.storage.u.virt.list[u].source_file_orig =
                            (size_t)(tmp_ent - tmp_layout.storage.u.virt.list);
                        tmp_layout.storage.u.virt.list[u].source_file_name = tmp_ent->source_file_name;
                    }
                    else {
                        
                        if (NULL == (tmp_layout.storage.u.virt.list[u].source_file_name =
                                         (char *)H5MM_xstrdup((const char *)*pp)))
                            HGOTO_ERROR(H5E_PLIST, H5E_RESOURCE, FAIL, "can't duplicate source file name");
                        tmp_layout.storage.u.virt.list[u].source_file_orig = SIZE_MAX;
                        HASH_ADD_KEYPTR(hh_source_file, tmp_layout.storage.u.virt.source_file_hash_table,
                                        tmp_layout.storage.u.virt.list[u].source_file_name, tmp_size - 1,
                                        &tmp_layout.storage.u.virt.list[u]);
                    }
                    *pp += tmp_size;

                    
                    tmp_ent  = NULL;
                    tmp_size = strlen((const char *)*pp) + 1;
                    if (tmp_layout.storage.u.virt.list_nused > 0)
                        HASH_FIND(hh_source_dset, tmp_layout.storage.u.virt.source_dset_hash_table, *pp,
                                  tmp_size - 1, tmp_ent);
                    if (tmp_ent) {
                        
                        assert(tmp_ent >= tmp_layout.storage.u.virt.list &&
                               tmp_ent < &tmp_layout.storage.u.virt.list[u]);
                        tmp_layout.storage.u.virt.list[u].source_dset_orig =
                            (size_t)(tmp_ent - tmp_layout.storage.u.virt.list);
                        tmp_layout.storage.u.virt.list[u].source_dset_name = tmp_ent->source_dset_name;
                    }
                    else {
                        
                        if (NULL == (tmp_layout.storage.u.virt.list[u].source_dset_name =
                                         (char *)H5MM_xstrdup((const char *)*pp)))
                            HGOTO_ERROR(H5E_PLIST, H5E_RESOURCE, FAIL, "can't duplicate source dataset name");
                        tmp_layout.storage.u.virt.list[u].source_dset_orig = SIZE_MAX;
                        HASH_ADD_KEYPTR(hh_source_dset, tmp_layout.storage.u.virt.source_dset_hash_table,
                                        tmp_layout.storage.u.virt.list[u].source_dset_name, tmp_size - 1,
                                        &tmp_layout.storage.u.virt.list[u]);
                    }
                    *pp += tmp_size;

                    
                    if (NULL == (tmp_layout.storage.u.virt.list[u].source_select = H5S_decode(pp)))
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTDECODE, FAIL, "can't decode source space selection");
                    tmp_layout.storage.u.virt.list[u].source_space_status = H5O_VIRTUAL_STATUS_USER;

                    
                    if (NULL ==
                        (tmp_layout.storage.u.virt.list[u].source_dset.virtual_select = H5S_decode(pp)))
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTDECODE, FAIL, "can't decode virtual space selection");
                    tmp_layout.storage.u.virt.list[u].virtual_space_status = H5O_VIRTUAL_STATUS_USER;

                    
                    if (H5D_virtual_parse_source_name(
                            tmp_layout.storage.u.virt.list[u].source_file_name,
                            &tmp_layout.storage.u.virt.list[u].parsed_source_file_name,
                            &tmp_layout.storage.u.virt.list[u].psfn_static_strlen,
                            &tmp_layout.storage.u.virt.list[u].psfn_nsubs) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't parse source file name");
                    if (H5D_virtual_parse_source_name(
                            tmp_layout.storage.u.virt.list[u].source_dset_name,
                            &tmp_layout.storage.u.virt.list[u].parsed_source_dset_name,
                            &tmp_layout.storage.u.virt.list[u].psdn_static_strlen,
                            &tmp_layout.storage.u.virt.list[u].psdn_nsubs) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't parse source dataset name");

                    
                    if ((tmp_layout.storage.u.virt.list[u].psfn_nsubs == 0) &&
                        (tmp_layout.storage.u.virt.list[u].psdn_nsubs == 0)) {
                        if (tmp_layout.storage.u.virt.list[u].parsed_source_file_name)
                            tmp_layout.storage.u.virt.list[u].source_dset.file_name =
                                tmp_layout.storage.u.virt.list[u].parsed_source_file_name->name_segment;
                        else
                            tmp_layout.storage.u.virt.list[u].source_dset.file_name =
                                tmp_layout.storage.u.virt.list[u].source_file_name;
                        if (tmp_layout.storage.u.virt.list[u].parsed_source_dset_name)
                            tmp_layout.storage.u.virt.list[u].source_dset.dset_name =
                                tmp_layout.storage.u.virt.list[u].parsed_source_dset_name->name_segment;
                        else
                            tmp_layout.storage.u.virt.list[u].source_dset.dset_name =
                                tmp_layout.storage.u.virt.list[u].source_dset_name;
                    } 

                    
                    tmp_layout.storage.u.virt.list[u].unlim_dim_source =
                        H5S_get_select_unlim_dim(tmp_layout.storage.u.virt.list[u].source_select);
                    tmp_layout.storage.u.virt.list[u].unlim_dim_virtual = H5S_get_select_unlim_dim(
                        tmp_layout.storage.u.virt.list[u].source_dset.virtual_select);
                    tmp_layout.storage.u.virt.list[u].unlim_extent_source  = HSIZE_UNDEF;
                    tmp_layout.storage.u.virt.list[u].unlim_extent_virtual = HSIZE_UNDEF;
                    tmp_layout.storage.u.virt.list[u].clip_size_source     = HSIZE_UNDEF;
                    tmp_layout.storage.u.virt.list[u].clip_size_virtual    = HSIZE_UNDEF;

                    
                    if (tmp_layout.storage.u.virt.list[u].unlim_dim_virtual < 0) {
                        tmp_layout.storage.u.virt.list[u].source_dset.clipped_source_select =
                            tmp_layout.storage.u.virt.list[u].source_select;
                        tmp_layout.storage.u.virt.list[u].source_dset.clipped_virtual_select =
                            tmp_layout.storage.u.virt.list[u].source_dset.virtual_select;
                    } 

                    
                    if (H5D_virtual_update_min_dims(&tmp_layout, u) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL,
                                    "unable to update virtual dataset minimum dimensions");
                } 

                
                layout = &tmp_layout;
            } 
        }     
        break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "bad layout type");
    } 

    
    H5MM_memcpy(value, layout, sizeof(H5O_layout_t));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                     size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5O_msg_reset(H5O_LAYOUT_ID, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release layout message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_layout_t *layout = (H5O_layout_t *)value; 
    H5O_layout_t  new_layout;
    herr_t        ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(layout);

    
    if (NULL == H5O_msg_copy(H5O_LAYOUT_ID, layout, &new_layout))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy layout");

    
    *layout = new_layout;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__dcrt_layout_cmp(const void *_layout1, const void *_layout2, size_t H5_ATTR_UNUSED size)
{
    const H5O_layout_t *layout1 = (const H5O_layout_t *)_layout1, 
        *layout2                = (const H5O_layout_t *)_layout2;
    herr_t ret_value            = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(layout1);
    assert(layout2);
    assert(size == sizeof(H5O_layout_t));

    
    if (layout1->type < layout2->type)
        HGOTO_DONE(-1);
    if (layout1->type > layout2->type)
        HGOTO_DONE(1);

    
    switch (layout1->type) {
        case H5D_COMPACT:
        case H5D_CONTIGUOUS:
            break;

        case H5D_CHUNKED: {
            unsigned u; 

            
            if (layout1->u.chunk.ndims < layout2->u.chunk.ndims)
                HGOTO_DONE(-1);
            if (layout1->u.chunk.ndims > layout2->u.chunk.ndims)
                HGOTO_DONE(1);

            
            for (u = 0; u < layout1->u.chunk.ndims - 1; u++) {
                if (layout1->u.chunk.dim[u] < layout2->u.chunk.dim[u])
                    HGOTO_DONE(-1);
                if (layout1->u.chunk.dim[u] > layout2->u.chunk.dim[u])
                    HGOTO_DONE(1);
            } 
        }     
        break;

        case H5D_VIRTUAL: {
            htri_t equal;
            int    strcmp_ret;
            size_t u; 

            
            if (layout1->storage.u.virt.list_nused < layout2->storage.u.virt.list_nused)
                HGOTO_DONE(-1);
            if (layout1->storage.u.virt.list_nused > layout2->storage.u.virt.list_nused)
                HGOTO_DONE(1);

            
            for (u = 0; u < layout1->storage.u.virt.list_nused; u++) {
                
                if ((equal = H5S_extent_equal(layout1->storage.u.virt.list[u].source_dset.virtual_select,
                                              layout2->storage.u.virt.list[u].source_dset.virtual_select)) <
                    0)
                    HGOTO_DONE(-1);
                if (!equal)
                    HGOTO_DONE(1);
                if ((equal = H5S_SELECT_SHAPE_SAME(
                         layout1->storage.u.virt.list[u].source_dset.virtual_select,
                         layout2->storage.u.virt.list[u].source_dset.virtual_select)) < 0)
                    HGOTO_DONE(-1);
                if (!equal)
                    HGOTO_DONE(1);

                
                strcmp_ret = strcmp(layout1->storage.u.virt.list[u].source_file_name,
                                    layout2->storage.u.virt.list[u].source_file_name);
                if (strcmp_ret < 0)
                    HGOTO_DONE(-1);
                if (strcmp_ret > 0)
                    HGOTO_DONE(1);

                
                strcmp_ret = strcmp(layout1->storage.u.virt.list[u].source_dset_name,
                                    layout2->storage.u.virt.list[u].source_dset_name);
                if (strcmp_ret < 0)
                    HGOTO_DONE(-1);
                if (strcmp_ret > 0)
                    HGOTO_DONE(1);

                
                if ((equal = H5S_extent_equal(layout1->storage.u.virt.list[u].source_select,
                                              layout2->storage.u.virt.list[u].source_select)) < 0)
                    HGOTO_DONE(-1);
                if (!equal)
                    HGOTO_DONE(1);
                if ((equal = H5S_SELECT_SHAPE_SAME(layout1->storage.u.virt.list[u].source_select,
                                                   layout2->storage.u.virt.list[u].source_select)) < 0)
                    HGOTO_DONE(-1);
                if (!equal)
                    HGOTO_DONE(1);
            } 
        }     
        break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            assert(0 && "Unknown layout type!");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_layout_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5O_msg_reset(H5O_LAYOUT_ID, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release layout message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_fill_t *fill = (H5O_fill_t *)value; 
    H5O_fill_t  new_fill;
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (NULL == H5O_msg_copy(H5O_FILL_ID, fill, &new_fill))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy fill value");

    
    *fill = new_fill;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_fill_t *fill = (H5O_fill_t *)value; 
    H5O_fill_t  new_fill;
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (NULL == H5O_msg_copy(H5O_FILL_ID, fill, &new_fill))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy fill value");

    
    *fill = new_fill;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_enc(const void *value, void **_pp, size_t *size)
{
    const H5O_fill_t *fill      = (const H5O_fill_t *)value; 
    size_t            dt_size   = 0;                         
    herr_t            ret_value = SUCCEED;                   
    uint8_t         **pp        = (uint8_t **)_pp;
    uint64_t          enc_value;
    unsigned          enc_size = 0;

    FUNC_ENTER_PACKAGE

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    HDcompile_assert(sizeof(ssize_t) <= sizeof(int64_t));
    assert(fill);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)fill->alloc_time;
        *(*pp)++ = (uint8_t)fill->fill_time;

        
        INT64ENCODE(*pp, fill->size);

        
        if (fill->size > 0) {
            
            H5MM_memcpy(*pp, (uint8_t *)fill->buf, (size_t)fill->size);
            *pp += fill->size;

            
            assert(fill->type);

            if (H5T_encode(fill->type, NULL, &dt_size) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "can't encode datatype");

            
            enc_value = (uint64_t)dt_size;
            enc_size  = H5VM_limit_enc_size(enc_value);
            assert(enc_size < 256);

            
            *(*pp)++ = (uint8_t)enc_size;

            
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);

            if (H5T_encode(fill->type, *pp, &dt_size) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "can't encode datatype");
            *pp += dt_size;
        } 
    }     

    
    *size += 2;
    *size += sizeof(int64_t);
    if (fill->size > 0) {
        
        *size += (size_t)fill->size;

        
        if (NULL == *pp) {
            
            assert(fill->type);
            if (H5T_encode(fill->type, NULL, &dt_size) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "can't encode datatype");
            enc_value = (uint64_t)dt_size;
            enc_size  = H5VM_limit_enc_size(enc_value);
        }
        *size += (1 + enc_size);
        *size += dt_size;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_dec(const void **_pp, void *_value)
{
    H5O_fill_t     *fill      = (H5O_fill_t *)_value; 
    const uint8_t **pp        = (const uint8_t **)_pp;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    HDcompile_assert(sizeof(ssize_t) <= sizeof(int64_t));

    
    *fill = H5D_def_fill_g;

    
    fill->alloc_time = (H5D_alloc_time_t) * (*pp)++;
    fill->fill_time  = (H5D_fill_time_t) * (*pp)++;

    
    INT64DECODE(*pp, fill->size);

    
    if (fill->size > 0) {
        size_t   dt_size = 0;
        uint64_t enc_value;
        unsigned enc_size;

        
        if (NULL == (fill->buf = H5MM_malloc((size_t)fill->size)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed for fill value buffer");
        H5MM_memcpy((uint8_t *)fill->buf, *pp, (size_t)fill->size);
        *pp += fill->size;

        enc_size = *(*pp)++;
        assert(enc_size < 256);

        
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        dt_size = (size_t)enc_value;

        
        if (NULL == (fill->type = H5T_decode(dt_size, *pp)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTDECODE, FAIL, "can't decode fill value datatype");
        *pp += dt_size;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5O_msg_reset(H5O_FILL_ID, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release fill value message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_fill_t *fill = (H5O_fill_t *)value; 
    H5O_fill_t  new_fill;
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(fill);

    
    if (NULL == H5O_msg_copy(H5O_FILL_ID, fill, &new_fill))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy fill value");

    
    *fill = new_fill;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5P_fill_value_cmp(const void *_fill1, const void *_fill2, size_t H5_ATTR_UNUSED size)
{
    const H5O_fill_t *fill1 = (const H5O_fill_t *)_fill1, 
        *fill2              = (const H5O_fill_t *)_fill2;
    int    cmp_value;     
    herr_t ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fill1);
    assert(fill2);
    assert(size == sizeof(H5O_fill_t));

    
    if (fill1->size < fill2->size)
        HGOTO_DONE(-1);
    if (fill1->size > fill2->size)
        HGOTO_DONE(1);

    
    if (fill1->type == NULL && fill2->type != NULL)
        HGOTO_DONE(-1);
    if (fill1->type != NULL && fill2->type == NULL)
        HGOTO_DONE(1);
    if (fill1->type != NULL)
        if ((cmp_value = H5T_cmp(fill1->type, fill2->type, false)) != 0)
            HGOTO_DONE(cmp_value);

    
    if (fill1->buf == NULL && fill2->buf != NULL)
        HGOTO_DONE(-1);
    if (fill1->buf != NULL && fill2->buf == NULL)
        HGOTO_DONE(1);
    if (fill1->buf != NULL)
        if ((cmp_value = memcmp(fill1->buf, fill2->buf, (size_t)fill1->size)) != 0)
            HGOTO_DONE(cmp_value);

    
    if (fill1->alloc_time < fill2->alloc_time)
        HGOTO_DONE(-1);
    if (fill1->alloc_time > fill2->alloc_time)
        HGOTO_DONE(1);

    
    if (fill1->fill_time < fill2->fill_time)
        HGOTO_DONE(-1);
    if (fill1->fill_time > fill2->fill_time)
        HGOTO_DONE(1);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_fill_value_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5O_msg_reset(H5O_FILL_ID, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release fill value message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_ext_file_list_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                            size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_efl_t *efl = (H5O_efl_t *)value; 
    H5O_efl_t  new_efl;
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (NULL == H5O_msg_copy(H5O_EFL_ID, efl, &new_efl))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy external file list");

    
    *efl = new_efl;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_ext_file_list_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                            size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_efl_t *efl = (H5O_efl_t *)value; 
    H5O_efl_t  new_efl;
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (NULL == H5O_msg_copy(H5O_EFL_ID, efl, &new_efl))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy external file list");

    
    *efl = new_efl;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_ext_file_list_enc(const void *value, void **_pp, size_t *size)
{
    const H5O_efl_t *efl = (const H5O_efl_t *)value; 
    size_t           len = 0;                        
    size_t           u;                              
    uint8_t        **pp = (uint8_t **)_pp;
    unsigned         enc_size;
    uint64_t         enc_value;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(efl);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    HDcompile_assert(sizeof(HDoff_t) <= sizeof(uint64_t));
    HDcompile_assert(sizeof(hsize_t) <= sizeof(uint64_t));
    assert(size);

    if (NULL != *pp) {
        
        enc_value = (uint64_t)efl->nused;
        enc_size  = H5VM_limit_enc_size(enc_value);
        assert(enc_size < 256);
        *(*pp)++ = (uint8_t)enc_size;
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);

        
        for (u = 0; u < efl->nused; u++) {
            
            len       = strlen(efl->slot[u].name) + 1;
            enc_value = (uint64_t)len;
            enc_size  = H5VM_limit_enc_size(enc_value);
            assert(enc_size < 256);
            *(*pp)++ = (uint8_t)enc_size;
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);

            
            H5MM_memcpy(*pp, (uint8_t *)(efl->slot[u].name), len);
            *pp += len;

            
            enc_value = (uint64_t)efl->slot[u].offset;
            enc_size  = H5VM_limit_enc_size(enc_value);
            assert(enc_size < 256);
            *(*pp)++ = (uint8_t)enc_size;
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);

            
            enc_value = (uint64_t)efl->slot[u].size;
            enc_size  = H5VM_limit_enc_size(enc_value);
            assert(enc_size < 256);
            *(*pp)++ = (uint8_t)enc_size;
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);
        }
    }

    
    *size += (1 + H5VM_limit_enc_size((uint64_t)efl->nused));
    for (u = 0; u < efl->nused; u++) {
        len = strlen(efl->slot[u].name) + 1;
        *size += (1 + H5VM_limit_enc_size((uint64_t)len));
        *size += len;
        *size += (1 + H5VM_limit_enc_size((uint64_t)efl->slot[u].offset));
        *size += (1 + H5VM_limit_enc_size((uint64_t)efl->slot[u].size));
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dcrt_ext_file_list_dec(const void **_pp, void *_value)
{
    H5O_efl_t      *efl = (H5O_efl_t *)_value; 
    const uint8_t **pp  = (const uint8_t **)_pp;
    size_t          u, nused;
    unsigned        enc_size;
    uint64_t        enc_value;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(efl);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    HDcompile_assert(sizeof(HDoff_t) <= sizeof(uint64_t));
    HDcompile_assert(sizeof(hsize_t) <= sizeof(uint64_t));

    
    *efl = H5D_def_efl_g;

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    nused = (size_t)enc_value;

    
    for (u = 0; u < nused; u++) {
        size_t len;
        if (efl->nused >= efl->nalloc) {
            size_t           na = efl->nalloc + H5O_EFL_ALLOC;
            H5O_efl_entry_t *x  = (H5O_efl_entry_t *)H5MM_realloc(efl->slot, na * sizeof(H5O_efl_entry_t));
            if (!x)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed");

            efl->nalloc = na;
            efl->slot   = x;
        }

        
        enc_size = *(*pp)++;
        assert(enc_size < 256);
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        len = (size_t)enc_value;

        
        efl->slot[u].name = H5MM_xstrdup((const char *)(*pp));
        *pp += len;

        
        enc_size = *(*pp)++;
        assert(enc_size < 256);
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        efl->slot[u].offset = (HDoff_t)enc_value;

        
        enc_size = *(*pp)++;
        assert(enc_size < 256);
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        efl->slot[u].size = (hsize_t)enc_value;

        efl->slot[u].name_offset = 0; 
        efl->nused++;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_ext_file_list_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                            size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5O_msg_reset(H5O_EFL_ID, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release external file list message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_ext_file_list_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    H5O_efl_t *efl = (H5O_efl_t *)value; 
    H5O_efl_t  new_efl;
    herr_t     ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(efl);

    
    if (NULL == H5O_msg_copy(H5O_EFL_ID, efl, &new_efl))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy external file list");

    
    *efl = new_efl;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__dcrt_ext_file_list_cmp(const void *_efl1, const void *_efl2, size_t H5_ATTR_UNUSED size)
{
    const H5O_efl_t *efl1 = (const H5O_efl_t *)_efl1, 
        *efl2             = (const H5O_efl_t *)_efl2;
    int    cmp_value;     
    herr_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(efl1);
    assert(efl2);
    assert(size == sizeof(H5O_efl_t));

    
    if (efl1->nalloc < efl2->nalloc)
        HGOTO_DONE(-1);
    if (efl1->nalloc > efl2->nalloc)
        HGOTO_DONE(1);

    
    if (efl1->nused < efl2->nused)
        HGOTO_DONE(-1);
    if (efl1->nused > efl2->nused)
        HGOTO_DONE(1);

    
    if (efl1->slot == NULL && efl2->slot != NULL)
        HGOTO_DONE(-1);
    if (efl1->slot != NULL && efl2->slot == NULL)
        HGOTO_DONE(1);
    if (efl1->slot != NULL && efl1->nused > 0) {
        size_t u; 

        
        for (u = 0; u < efl1->nused; u++) {
            
            if (efl1->slot[u].name_offset < efl2->slot[u].name_offset)
                HGOTO_DONE(-1);
            if (efl1->slot[u].name_offset > efl2->slot[u].name_offset)
                HGOTO_DONE(1);

            
            if (efl1->slot[u].name == NULL && efl2->slot[u].name != NULL)
                HGOTO_DONE(-1);
            if (efl1->slot[u].name != NULL && efl2->slot[u].name == NULL)
                HGOTO_DONE(1);
            if (efl1->slot[u].name != NULL)
                if ((cmp_value = strcmp(efl1->slot[u].name, efl2->slot[u].name)) != 0)
                    HGOTO_DONE(cmp_value);

            
            if (efl1->slot[u].offset < efl2->slot[u].offset)
                HGOTO_DONE(-1);
            if (efl1->slot[u].offset > efl2->slot[u].offset)
                HGOTO_DONE(1);

            
            if (efl1->slot[u].size < efl2->slot[u].size)
                HGOTO_DONE(-1);
            if (efl1->slot[u].size > efl2->slot[u].size)
                HGOTO_DONE(1);
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dcrt_ext_file_list_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5O_msg_reset(H5O_EFL_ID, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release external file list message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__set_layout(H5P_genplist_t *plist, const H5O_layout_t *layout)
{
    unsigned alloc_time_state;    
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P_get(plist, H5D_CRT_ALLOC_TIME_STATE_NAME, &alloc_time_state) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get space allocation time state");

    
    if (alloc_time_state) {
        H5O_fill_t fill; 

        
        if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

        
        switch (layout->type) {
            case H5D_COMPACT:
                fill.alloc_time = H5D_ALLOC_TIME_EARLY;
                break;

            case H5D_CONTIGUOUS:
                fill.alloc_time = H5D_ALLOC_TIME_LATE;
                break;

            case H5D_CHUNKED:
            case H5D_VIRTUAL:
                fill.alloc_time = H5D_ALLOC_TIME_INCR;
                break;

            case H5D_LAYOUT_ERROR:
            case H5D_NLAYOUTS:
            default:
                HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unknown layout type");
        } 

        
        if (H5P_poke(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set space allocation time");
    } 

    
    if (H5P_set(plist, H5D_CRT_LAYOUT_NAME, layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't set layout");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pset_layout(hid_t plist_id, H5D_layout_t layout_type)
{
    H5P_genplist_t     *plist;               
    const H5O_layout_t *layout;              
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (layout_type < 0 || layout_type >= H5D_NLAYOUTS)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "raw data layout method is not valid");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    switch (layout_type) {
        case H5D_COMPACT:
            layout = &H5D_def_layout_compact_g;
            break;

        case H5D_CONTIGUOUS:
            layout = &H5D_def_layout_contig_g;
            break;

        case H5D_CHUNKED:
            layout = &H5D_def_layout_chunk_g;
            break;

        case H5D_VIRTUAL:
            layout = &H5D_def_layout_virtual_g;
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unknown layout type");
    } 

    
    if (H5P__set_layout(plist, layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't set layout");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5D_layout_t
H5Pget_layout(hid_t plist_id)
{
    H5P_genplist_t *plist;     
    H5O_layout_t    layout;    
    H5D_layout_t    ret_value; 

    FUNC_ENTER_API(H5D_LAYOUT_ERROR)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, H5D_LAYOUT_ERROR, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, H5D_LAYOUT_ERROR, "can't get layout");

    
    ret_value = layout.type;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_chunk(hid_t plist_id, int ndims, const hsize_t dim[])
{
    H5P_genplist_t *plist;               
    H5O_layout_t    chunk_layout;        
    unsigned        u;                   
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (ndims <= 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "chunk dimensionality must be positive");
    if (ndims > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "chunk dimensionality is too large");
    if (!dim)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no chunk dimensions specified");

    
    H5MM_memcpy(&chunk_layout, &H5D_def_layout_chunk_g, sizeof(H5D_def_layout_chunk_g));
    memset(&chunk_layout.u.chunk.dim, 0, sizeof(chunk_layout.u.chunk.dim));
    for (u = 0; u < (unsigned)ndims; u++) {
        if (dim[u] == 0)
            HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "all chunk dimensions must be positive");
        chunk_layout.u.chunk.dim[u] = dim[u]; 
    }                                         

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    chunk_layout.u.chunk.ndims = (unsigned)ndims;
    if (H5P__set_layout(plist, &chunk_layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set layout");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Pget_chunk(hid_t plist_id, int max_ndims, hsize_t dim[] )
{
    H5P_genplist_t *plist;     
    H5O_layout_t    layout;    
    int             ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_CHUNKED != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a chunked storage layout");

    if (dim) {
        unsigned u; 

        
        for (u = 0; u < layout.u.chunk.ndims && u < (unsigned)max_ndims; u++)
            dim[u] = layout.u.chunk.dim[u];
    } 

    
    ret_value = (int)layout.u.chunk.ndims;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_virtual(hid_t dcpl_id, hid_t vspace_id, const char *src_file_name, const char *src_dset_name,
               hid_t src_space_id)
{
    H5P_genplist_t            *plist = NULL;               
    H5O_layout_t               virtual_layout;             
    H5S_t                     *vspace;                     
    H5S_t                     *src_space;                  
    H5O_storage_virtual_ent_t *old_list = NULL;            
    H5O_storage_virtual_ent_t *ent      = NULL;            
    size_t                     tmp_len;                    
    bool                       retrieved_layout = false;   
    bool                       free_list        = false;   
    herr_t                     ret_value        = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!src_file_name)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "source file name not provided");
    if (!src_dset_name)
        HGOTO_ERROR(H5E_PLIST, H5E_BADRANGE, FAIL, "source dataset name not provided");
    if (NULL == (vspace = (H5S_t *)H5I_object_verify(vspace_id, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a dataspace");
    if (NULL == (src_space = (H5S_t *)H5I_object_verify(src_space_id, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a dataspace");

    
    if (H5D_virtual_check_mapping_pre(vspace, src_space, H5O_VIRTUAL_STATUS_USER) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "invalid mapping selections");

    
    if (NULL == (plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &virtual_layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get layout");
    retrieved_layout = true;

    
    if (virtual_layout.type == H5D_VIRTUAL)
        
        old_list = virtual_layout.storage.u.virt.list;
    else {
        
        if (H5O_msg_reset(H5O_LAYOUT_ID, &virtual_layout) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTRESET, FAIL, "can't release layout message");

        
        H5MM_memcpy(&virtual_layout, &H5D_def_layout_virtual_g, sizeof(H5D_def_layout_virtual_g));

        
        assert(virtual_layout.storage.u.virt.list_nalloc == 0);
    } 

    
    if (virtual_layout.storage.u.virt.list_nused == virtual_layout.storage.u.virt.list_nalloc) {
        H5O_storage_virtual_ent_t *x; 
        size_t    new_alloc = MAX(H5D_VIRTUAL_DEF_LIST_SIZE, virtual_layout.storage.u.virt.list_nalloc * 2);
        ptrdiff_t buf_diff;

        
        if (NULL == (x = (H5O_storage_virtual_ent_t *)H5MM_realloc(
                         virtual_layout.storage.u.virt.list, new_alloc * sizeof(H5O_storage_virtual_ent_t))))
            HGOTO_ERROR(H5E_PLIST, H5E_RESOURCE, FAIL, "can't reallocate virtual dataset mapping list");
        buf_diff                                  = (char *)x - (char *)virtual_layout.storage.u.virt.list;
        virtual_layout.storage.u.virt.list        = x;
        virtual_layout.storage.u.virt.list_nalloc = new_alloc;

        
        HASH_ADJUST_PTRS(hh_source_file, virtual_layout.storage.u.virt.source_file_hash_table, buf_diff);
        HASH_ADJUST_PTRS(hh_source_dset, virtual_layout.storage.u.virt.source_dset_hash_table, buf_diff);
    } 

    
    assert((virtual_layout.storage.u.virt.list_nused &&
            virtual_layout.storage.u.virt.source_file_hash_table &&
            virtual_layout.storage.u.virt.source_dset_hash_table) ||
           (!virtual_layout.storage.u.virt.source_file_hash_table &&
            !virtual_layout.storage.u.virt.source_dset_hash_table));
    if (virtual_layout.storage.u.virt.list_nused && !virtual_layout.storage.u.virt.source_file_hash_table) {
        for (size_t i = 0; i < virtual_layout.storage.u.virt.list_nused; i++) {
            if (virtual_layout.storage.u.virt.list[i].source_file_orig == SIZE_MAX)
                HASH_ADD_KEYPTR(hh_source_file, virtual_layout.storage.u.virt.source_file_hash_table,
                                virtual_layout.storage.u.virt.list[i].source_file_name,
                                strlen(virtual_layout.storage.u.virt.list[i].source_file_name),
                                &(virtual_layout.storage.u.virt.list[i]));
            if (virtual_layout.storage.u.virt.list[i].source_dset_orig == SIZE_MAX)
                HASH_ADD_KEYPTR(hh_source_dset, virtual_layout.storage.u.virt.source_dset_hash_table,
                                virtual_layout.storage.u.virt.list[i].source_dset_name,
                                strlen(virtual_layout.storage.u.virt.list[i].source_dset_name),
                                &(virtual_layout.storage.u.virt.list[i]));
        }
    }

    
    ent = &virtual_layout.storage.u.virt.list[virtual_layout.storage.u.virt.list_nused];
    memset(ent, 0, sizeof(H5O_storage_virtual_ent_t)); 

    if (NULL == (ent->source_dset.virtual_select = H5S_copy(vspace, false, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy virtual selection");

    
    tmp_len = strlen(src_file_name);
    H5D_VIRTUAL_FIND_OR_ADD_NAME(file, &virtual_layout, src_file_name, tmp_len, ent, FAIL);

    
    tmp_len = strlen(src_dset_name);
    H5D_VIRTUAL_FIND_OR_ADD_NAME(dset, &virtual_layout, src_dset_name, tmp_len, ent, FAIL);

    if (NULL == (ent->source_select = H5S_copy(src_space, false, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy source selection");
    if (H5D_virtual_parse_source_name(ent->source_file_name, &ent->parsed_source_file_name,
                                      &ent->psfn_static_strlen, &ent->psfn_nsubs) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't parse source file name");
    if (H5D_virtual_parse_source_name(ent->source_dset_name, &ent->parsed_source_dset_name,
                                      &ent->psdn_static_strlen, &ent->psdn_nsubs) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't parse source dataset name");
    if ((ent->psfn_nsubs == 0) && (ent->psdn_nsubs == 0)) {
        if (ent->parsed_source_file_name)
            ent->source_dset.file_name = ent->parsed_source_file_name->name_segment;
        else
            ent->source_dset.file_name = ent->source_file_name;
        if (ent->parsed_source_dset_name)
            ent->source_dset.dset_name = ent->parsed_source_dset_name->name_segment;
        else
            ent->source_dset.dset_name = ent->source_dset_name;
    } 
    ent->unlim_dim_source  = H5S_get_select_unlim_dim(src_space);
    ent->unlim_dim_virtual = H5S_get_select_unlim_dim(vspace);
    if (ent->unlim_dim_virtual < 0) {
        ent->source_dset.clipped_source_select  = ent->source_select;
        ent->source_dset.clipped_virtual_select = ent->source_dset.virtual_select;
    } 
    ent->unlim_extent_source  = HSIZE_UNDEF;
    ent->unlim_extent_virtual = HSIZE_UNDEF;
    ent->clip_size_source     = HSIZE_UNDEF;
    ent->clip_size_virtual    = HSIZE_UNDEF;
    ent->source_space_status  = H5O_VIRTUAL_STATUS_USER;
    ent->virtual_space_status = H5O_VIRTUAL_STATUS_USER;

    
    if (H5D_virtual_check_mapping_post(ent) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid mapping entry");

    
    if (H5D_virtual_update_min_dims(&virtual_layout, virtual_layout.storage.u.virt.list_nused) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "unable to update virtual dataset minimum dimensions");

    
    virtual_layout.storage.u.virt.list_nused++;
done:
    
    
    if (retrieved_layout) {
        if (H5P_poke(plist, H5D_CRT_LAYOUT_NAME, &virtual_layout) < 0) {
            HDONE_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set layout");
            if (old_list != virtual_layout.storage.u.virt.list)
                free_list = true;
        } 
    }     

    
    if (ret_value < 0) {
        
        if (ent) {
            if (ent->source_file_orig == SIZE_MAX)
                ent->source_file_name = (char *)H5MM_xfree(ent->source_file_name);
            else
                HASH_DELETE(hh_source_file, virtual_layout.storage.u.virt.source_file_hash_table, ent);
            if (ent->source_dset_orig == SIZE_MAX)
                ent->source_dset_name = (char *)H5MM_xfree(ent->source_dset_name);
            else
                HASH_DELETE(hh_source_dset, virtual_layout.storage.u.virt.source_dset_hash_table, ent);
            if (ent->source_dset.virtual_select && H5S_close(ent->source_dset.virtual_select) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release virtual selection");
            ent->source_dset.virtual_select = NULL;
            if (ent->source_select && H5S_close(ent->source_select) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release source selection");
            ent->source_select = NULL;
            H5D_virtual_free_parsed_name(ent->parsed_source_file_name);
            ent->parsed_source_file_name = NULL;
            H5D_virtual_free_parsed_name(ent->parsed_source_dset_name);
            ent->parsed_source_dset_name = NULL;
        } 

        
        if (free_list)
            virtual_layout.storage.u.virt.list =
                (H5O_storage_virtual_ent_t *)H5MM_xfree(virtual_layout.storage.u.virt.list);
    } 
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_virtual_count(hid_t dcpl_id, size_t *count )
{
    H5P_genplist_t *plist;               
    H5O_layout_t    layout;              
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (count) {
        
        if (NULL == (plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
        if (H5D_VIRTUAL != layout.type)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a virtual storage layout");

        
        *count = layout.storage.u.virt.list_nused;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pget_virtual_vspace(hid_t dcpl_id, size_t idx)
{
    H5P_genplist_t *plist;        
    H5O_layout_t    layout;       
    H5S_t          *space = NULL; 
    hid_t           ret_value;    

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_VIRTUAL != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a virtual storage layout");

    
    if (idx >= layout.storage.u.virt.list_nused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid index (out of range)");
    assert(layout.storage.u.virt.list_nused <= layout.storage.u.virt.list_nalloc);
    if (NULL == (space = H5S_copy(layout.storage.u.virt.list[idx].source_dset.virtual_select, false, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy virtual selection");

    
    if ((ret_value = H5I_register(H5I_DATASPACE, space, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register dataspace");

done:
    
    if ((ret_value < 0) && space)
        if (H5S_close(space) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release source selection");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pget_virtual_srcspace(hid_t dcpl_id, size_t idx)
{
    H5P_genplist_t *plist;            
    H5O_layout_t    layout;           
    H5S_t          *space     = NULL; 
    hid_t           ret_value = FAIL; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_VIRTUAL != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a virtual storage layout");

    
    if (idx >= layout.storage.u.virt.list_nused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid index (out of range)");
    assert(layout.storage.u.virt.list_nused <= layout.storage.u.virt.list_nalloc);

    
    
    if ((H5O_VIRTUAL_STATUS_INVALID == layout.storage.u.virt.list[idx].source_space_status) &&
        (layout.storage.u.virt.list[idx].unlim_dim_source < 0)) {
        hsize_t bounds_start[H5S_MAX_RANK];
        hsize_t bounds_end[H5S_MAX_RANK];
        int     rank;
        int     i;

        
        if ((rank = H5S_GET_EXTENT_NDIMS(layout.storage.u.virt.list[idx].source_select)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get source space rank");

        
        if (H5S_SELECT_BOUNDS(layout.storage.u.virt.list[idx].source_select, bounds_start, bounds_end) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get selection bounds");

        
        for (i = 0; i < rank; i++)
            bounds_end[i]++;

        
        if (H5S_set_extent_simple(layout.storage.u.virt.list[idx].source_select, (unsigned)rank, bounds_end,
                                  NULL) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set source space extent");

        
        layout.storage.u.virt.list[idx].source_space_status = H5O_VIRTUAL_STATUS_SEL_BOUNDS;
    } 

    
    if (NULL == (space = H5S_copy(layout.storage.u.virt.list[idx].source_select, false, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy source selection");

    
    if ((ret_value = H5I_register(H5I_DATASPACE, space, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register dataspace");

done:
    
    if ((ret_value < 0) && space)
        if (H5S_close(space) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release source selection");

    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Pget_virtual_filename(hid_t dcpl_id, size_t idx, char *name , size_t size)
{
    H5P_genplist_t *plist;     
    H5O_layout_t    layout;    
    ssize_t         ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_VIRTUAL != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a virtual storage layout");

    
    if (idx >= layout.storage.u.virt.list_nused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid index (out of range)");
    assert(layout.storage.u.virt.list_nused <= layout.storage.u.virt.list_nalloc);
    assert(layout.storage.u.virt.list[idx].source_file_name);
    if (name && (size > 0))
        (void)strncpy(name, layout.storage.u.virt.list[idx].source_file_name, size);
    ret_value = (ssize_t)strlen(layout.storage.u.virt.list[idx].source_file_name);

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Pget_virtual_dsetname(hid_t dcpl_id, size_t idx, char *name , size_t size)
{
    H5P_genplist_t *plist;     
    H5O_layout_t    layout;    
    ssize_t         ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_VIRTUAL != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a virtual storage layout");

    
    if (idx >= layout.storage.u.virt.list_nused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid index (out of range)");
    assert(layout.storage.u.virt.list_nused <= layout.storage.u.virt.list_nalloc);
    assert(layout.storage.u.virt.list[idx].source_dset_name);
    if (name && (size > 0))
        (void)strncpy(name, layout.storage.u.virt.list[idx].source_dset_name, size);
    ret_value = (ssize_t)strlen(layout.storage.u.virt.list[idx].source_dset_name);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_chunk_opts(hid_t plist_id, unsigned options)
{
    H5P_genplist_t *plist;                  
    H5O_layout_t    layout;                 
    uint8_t         layout_flags = 0;       
    herr_t          ret_value    = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (options & ~(H5D_CHUNK_DONT_FILTER_PARTIAL_CHUNKS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "unknown chunk options");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_CHUNKED != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a chunked storage layout");

    
    if (options & H5D_CHUNK_DONT_FILTER_PARTIAL_CHUNKS)
        layout_flags |= H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS;

    
    
    layout.u.chunk.flags = layout_flags;
    if (layout.version < H5O_LAYOUT_VERSION_4)
        layout.version = H5O_LAYOUT_VERSION_4;

    
    if (H5P_poke(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't set layout");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_chunk_opts(hid_t plist_id, unsigned *options )
{
    H5P_genplist_t *plist;               
    H5O_layout_t    layout;              
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't get layout");
    if (H5D_CHUNKED != layout.type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a chunked storage layout");

    if (options) {
        
        *options = 0;
        if (layout.u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS)
            *options |= H5D_CHUNK_DONT_FILTER_PARTIAL_CHUNKS;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_external(hid_t plist_id, const char *name, HDoff_t offset, hsize_t size)
{
    size_t          idx;
    hsize_t         total, tmp;
    H5O_efl_t       efl;
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name given");
    if (offset < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "negative external file offset");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (H5P_peek(plist, H5D_CRT_EXT_FILE_LIST_NAME, &efl) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get external file list");
    if (efl.nused > 0 && H5O_EFL_UNLIMITED == efl.slot[efl.nused - 1].size)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "previous file size is unlimited");

    if (H5O_EFL_UNLIMITED != size) {
        for (idx = 0, total = size; idx < efl.nused; idx++, total = tmp) {
            tmp = total + efl.slot[idx].size;
            if (tmp <= total)
                HGOTO_ERROR(H5E_EFL, H5E_OVERFLOW, FAIL, "total external data size overflowed");
        }
    }

    
    if (efl.nused >= efl.nalloc) {
        size_t           na = efl.nalloc + H5O_EFL_ALLOC;
        H5O_efl_entry_t *x  = (H5O_efl_entry_t *)H5MM_realloc(efl.slot, na * sizeof(H5O_efl_entry_t));

        if (!x)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed");
        efl.nalloc = na;
        efl.slot   = x;
    }

    idx                       = efl.nused;
    efl.slot[idx].name_offset = 0; 
    efl.slot[idx].name        = H5MM_xstrdup(name);
    efl.slot[idx].offset      = offset;
    efl.slot[idx].size        = size;
    efl.nused++;

    if (H5P_poke(plist, H5D_CRT_EXT_FILE_LIST_NAME, &efl) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set external file list");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Pget_external_count(hid_t plist_id)
{
    H5O_efl_t       efl;
    H5P_genplist_t *plist;     
    int             ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_EXT_FILE_LIST_NAME, &efl) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get external file list");

    
    ret_value = (int)efl.nused;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_external(hid_t plist_id, unsigned idx, size_t name_size, char *name , HDoff_t *offset ,
                hsize_t *size )
{
    H5O_efl_t       efl;
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_EXT_FILE_LIST_NAME, &efl) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get external file list");

    if (idx >= efl.nused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "external file index is out of range");

    
    if (name_size > 0 && name)
        strncpy(name, efl.slot[idx].name, name_size);
    if (offset)
        *offset = efl.slot[idx].offset;
    if (size)
        *size = efl.slot[idx].size;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_szip(hid_t plist_id, unsigned options_mask, unsigned pixels_per_block)
{
    H5O_pline_t     pline;
    H5P_genplist_t *plist;        
    unsigned        cd_values[2]; 
    unsigned int    config_flags;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (H5Z_get_filter_info(H5Z_FILTER_SZIP, &config_flags) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "can't get filter info");

    if (!(config_flags & H5Z_FILTER_CONFIG_ENCODE_ENABLED))
        HGOTO_ERROR(H5E_PLINE, H5E_NOENCODER, FAIL, "Filter present but encoding is disabled.");

    
    if ((pixels_per_block % 2) == 1)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "pixels_per_block is not even");
    if (pixels_per_block > H5_SZIP_MAX_PIXELS_PER_BLOCK)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "pixels_per_block is too large");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    options_mask &= (unsigned)(~H5_SZIP_CHIP_OPTION_MASK);
    options_mask |= H5_SZIP_ALLOW_K13_OPTION_MASK;

    
    options_mask |= H5_SZIP_RAW_OPTION_MASK;

    
    
    options_mask &= (unsigned)(~(H5_SZIP_LSB_OPTION_MASK | H5_SZIP_MSB_OPTION_MASK));

    
    cd_values[0] = options_mask;
    cd_values[1] = pixels_per_block;

    
    if (H5P_peek(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get pipeline");
    if (H5Z_append(&pline, H5Z_FILTER_SZIP, H5Z_FLAG_OPTIONAL, (size_t)2, cd_values) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to add szip filter to pipeline");
    if (H5P_poke(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to set pipeline");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_shuffle(hid_t plist_id)
{
    H5O_pline_t     pline;
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (true != H5P_isa_class(plist_id, H5P_DATASET_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset creation property list");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(plist_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get pipeline");
    if (H5Z_append(&pline, H5Z_FILTER_SHUFFLE, H5Z_FLAG_OPTIONAL, (size_t)0, NULL) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to shuffle the data");
    if (H5P_poke(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to set pipeline");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_nbit(hid_t plist_id)
{
    H5O_pline_t     pline;
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (true != H5P_isa_class(plist_id, H5P_DATASET_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset creation property list");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(plist_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get pipeline");
    if (H5Z_append(&pline, H5Z_FILTER_NBIT, H5Z_FLAG_OPTIONAL, (size_t)0, NULL) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to add nbit filter to pipeline");
    if (H5P_poke(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to set pipeline");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_scaleoffset(hid_t plist_id, H5Z_SO_scale_type_t scale_type, int scale_factor)
{
    H5O_pline_t     pline;
    H5P_genplist_t *plist;               
    unsigned        cd_values[2];        
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (true != H5P_isa_class(plist_id, H5P_DATASET_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset creation property list");

    if (scale_factor < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "scale factor must be >= 0");
    if (scale_type != H5Z_SO_FLOAT_DSCALE && scale_type != H5Z_SO_FLOAT_ESCALE && scale_type != H5Z_SO_INT)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid scale type");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(plist_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    cd_values[0] = scale_type;
    cd_values[1] = (unsigned)scale_factor;

    
    if (H5P_peek(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get pipeline");
    if (H5Z_append(&pline, H5Z_FILTER_SCALEOFFSET, H5Z_FLAG_OPTIONAL, (size_t)2, cd_values) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to add scaleoffset filter to pipeline");
    if (H5P_poke(plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to set pipeline");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_fill_value(hid_t plist_id, hid_t type_id, const void *value)
{
    H5P_genplist_t *plist;               
    H5O_fill_t      fill;                
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

    
    H5O_fill_reset_dyn(&fill);

    if (value) {
        H5T_t      *type;  
        H5T_path_t *tpath; 

        
        if (NULL == (type = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

        
        if (NULL == (fill.type = H5T_copy(type, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy datatype");
        fill.size = (ssize_t)H5T_get_size(type);
        if (NULL == (fill.buf = H5MM_malloc((size_t)fill.size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, FAIL, "memory allocation failed for fill value");
        H5MM_memcpy(fill.buf, value, (size_t)fill.size);

        
        if (NULL == (tpath = H5T_path_find(type, type)))
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                        "unable to convert between src and dest data types");

        
        if (!H5T_path_noop(tpath)) {
            uint8_t *bkg_buf = NULL; 

            
            if (H5T_path_bkg(tpath) && NULL == (bkg_buf = H5FL_BLK_CALLOC(type_conv, (size_t)fill.size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

            
            if (H5T_convert(tpath, type, type, (size_t)1, (size_t)0, (size_t)0, fill.buf, bkg_buf) < 0) {
                if (bkg_buf)
                    bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");
            } 

            
            if (bkg_buf)
                bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);
        } 
    }     
    else
        fill.size = (-1);

    
    if (H5P_poke(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't set fill value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5P_get_fill_value(H5P_genplist_t *plist, const H5T_t *type, void *value )
{
    H5O_fill_t  fill;                
    H5T_path_t *tpath;               
    void       *buf       = NULL;    
    void       *bkg       = NULL;    
    H5T_t      *src_type  = NULL;    
    H5T_t      *tmp_type  = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");
    if (fill.size == -1)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "fill value is undefined");

    
    if (fill.size == 0) {
        memset(value, 0, H5T_get_size(type));
        HGOTO_DONE(SUCCEED);
    } 

    
    if (NULL == (tpath = H5T_path_find(fill.type, type)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "unable to convert between src and dst datatypes");

    src_type = fill.type;
    if (H5T_detect_class(src_type, H5T_VLEN, false) > 0 ||
        H5T_detect_class(src_type, H5T_REFERENCE, false) > 0) {
        if (NULL == (tmp_type = H5T_copy(src_type, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy fill value datatype");
        src_type = tmp_type;
    }

    
    if (H5T_get_size(type) >= H5T_get_size(fill.type)) {
        buf = value;
        if (H5T_path_bkg(tpath) && NULL == (bkg = H5MM_calloc(H5T_get_size(type))))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed for type conversion");
    } 
    else {
        if (NULL == (buf = H5MM_calloc(H5T_get_size(fill.type))))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed for type conversion");
        if (H5T_path_bkg(tpath) && NULL == (bkg = H5MM_calloc(H5T_get_size(fill.type))))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed for type conversion");
    } 
    H5MM_memcpy(buf, fill.buf, H5T_get_size(fill.type));

    
    if (H5T_convert(tpath, src_type, type, (size_t)1, (size_t)0, (size_t)0, buf, bkg) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "datatype conversion failed");
    if (buf != value)
        H5MM_memcpy(value, buf, H5T_get_size(type));

done:
    if (buf != value)
        H5MM_xfree(buf);
    if (bkg != value)
        H5MM_xfree(bkg);
    if (tmp_type && H5T_close(tmp_type) < 0)
        HDONE_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL, "unable to close temporary datatype");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pget_fill_value(hid_t plist_id, hid_t type_id, void *value )
{
    H5P_genplist_t *plist;               
    H5T_t          *type;                
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (type = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (!value)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no fill value output buffer");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get_fill_value(plist, type, value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5P_is_fill_value_defined(const H5O_fill_t *fill, H5D_fill_value_t *status)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(fill);
    assert(status);

    
    if (fill->size == -1 && !fill->buf)
        *status = H5D_FILL_VALUE_UNDEFINED;
    
    else if (fill->size == 0 && !fill->buf)
        *status = H5D_FILL_VALUE_DEFAULT;
    
    else if (fill->size > 0 && fill->buf)
        *status = H5D_FILL_VALUE_USER_DEFINED;
    else {
        *status = H5D_FILL_VALUE_ERROR;
        HGOTO_ERROR(H5E_PLIST, H5E_BADRANGE, FAIL, "invalid combination of fill-value info");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_fill_value_defined(H5P_genplist_t *plist, H5D_fill_value_t *status)
{
    H5O_fill_t fill; 
    herr_t     ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(status);

    
    if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

    
    if (H5P_is_fill_value_defined(&fill, status) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't check fill value status");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pfill_value_defined(hid_t plist_id, H5D_fill_value_t *status)
{
    H5P_genplist_t *plist; 
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    assert(status);

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_fill_value_defined(plist, status) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't check fill value status");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_alloc_time(hid_t plist_id, H5D_alloc_time_t alloc_time)
{
    H5P_genplist_t *plist;               
    H5O_fill_t      fill;                
    unsigned        alloc_time_state;    
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (alloc_time < H5D_ALLOC_TIME_DEFAULT || alloc_time > H5D_ALLOC_TIME_INCR)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid allocation time setting");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (alloc_time == H5D_ALLOC_TIME_DEFAULT) {
        H5O_layout_t layout; 

        
        if (H5P_peek(plist, H5D_CRT_LAYOUT_NAME, &layout) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get layout");

        
        switch (layout.type) {
            case H5D_COMPACT:
                alloc_time = H5D_ALLOC_TIME_EARLY;
                break;

            case H5D_CONTIGUOUS:
                alloc_time = H5D_ALLOC_TIME_LATE;
                break;

            case H5D_CHUNKED:
                alloc_time = H5D_ALLOC_TIME_INCR;
                break;

            case H5D_VIRTUAL:
                alloc_time = H5D_ALLOC_TIME_INCR;
                break;

            case H5D_LAYOUT_ERROR:
            case H5D_NLAYOUTS:
            default:
                HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unknown layout type");
        } 

        
        alloc_time_state = 1;
    } 
    else
        
        alloc_time_state = 0;

    
    if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

    
    fill.alloc_time = alloc_time;

    
    if (H5P_poke(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set fill value");
    if (H5P_set(plist, H5D_CRT_ALLOC_TIME_STATE_NAME, &alloc_time_state) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set space allocation time");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_alloc_time(hid_t plist_id, H5D_alloc_time_t *alloc_time )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (alloc_time) {
        H5P_genplist_t *plist; 
        H5O_fill_t      fill;  

        
        if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

        
        *alloc_time = fill.alloc_time;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_fill_time(hid_t plist_id, H5D_fill_time_t fill_time)
{
    H5P_genplist_t *plist;               
    H5O_fill_t      fill;                
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (fill_time < H5D_FILL_TIME_ALLOC || fill_time > H5D_FILL_TIME_IFSET)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fill time setting");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

    
    fill.fill_time = fill_time;

    
    if (H5P_poke(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set fill value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_fill_time(hid_t plist_id, H5D_fill_time_t *fill_time )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (fill_time) {
        H5P_genplist_t *plist; 
        H5O_fill_t      fill;  

        
        if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_peek(plist, H5D_CRT_FILL_VALUE_NAME, &fill) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get fill value");

        
        *fill_time = fill.fill_time;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_dset_no_attrs_hint(hid_t dcpl_id, bool *minimize )
{
    bool            setting   = false;
    H5P_genplist_t *plist     = NULL;
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    if (NULL == minimize)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "receiving pointer cannot be NULL");

    plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, true);
    if (NULL == plist)
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (H5P_peek(plist, H5D_CRT_MIN_DSET_HDR_SIZE_NAME, &setting) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get dset oh minimize flag value");

    *minimize = setting;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_dset_no_attrs_hint(hid_t dcpl_id, bool minimize)
{
    H5P_genplist_t *plist     = NULL;
    bool            prev_set  = false;
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    plist = H5P_object_verify(dcpl_id, H5P_DATASET_CREATE, false);
    if (NULL == plist)
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (H5P_peek(plist, H5D_CRT_MIN_DSET_HDR_SIZE_NAME, &prev_set) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get extant dset oh minimize flag value");

    if (H5P_poke(plist, H5D_CRT_MIN_DSET_HDR_SIZE_NAME, &minimize) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't get dset oh minimize flag value");

done:
    FUNC_LEAVE_API(ret_value)
} 
