/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

hid_t
H5Tarray_create2(hid_t base_id, unsigned ndims, const hsize_t dim[])
{
    H5T_t   *base;      
    H5T_t   *dt = NULL; 
    unsigned u;         
    hid_t    ret_value; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (ndims < 1 || ndims > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid dimensionality");
    if (!dim)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "no dimensions specified");
    for (u = 0; u < ndims; u++)
        if (!(dim[u] > 0))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "zero-sized dimension specified");
    if (NULL == (base = (H5T_t *)H5I_object_verify(base_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not an valid base datatype");

    
    if (NULL == (dt = H5T__array_create(base, ndims, dim)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to create datatype");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register datatype");

done:
    if (ret_value < 0)
        if (dt && H5T_close_real(dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, H5I_INVALID_HID, "can't release datatype");

    FUNC_LEAVE_API(ret_value)
} 

H5T_t *
H5T__array_create(H5T_t *base, unsigned ndims, const hsize_t dim[])
{
    unsigned u;                
    H5T_t   *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(base);
    assert(ndims <= H5S_MAX_RANK);
    assert(dim);

    
    if (NULL == (ret_value = H5T__alloc()))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    ret_value->shared->type = H5T_ARRAY;

    
    if (NULL == (ret_value->shared->parent = H5T_copy(base, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "unable to copy base datatype");

    
    ret_value->shared->u.array.ndims = ndims;

    
    for (u = 0, ret_value->shared->u.array.nelem = 1; u < ndims; u++) {
        H5_CHECKED_ASSIGN(ret_value->shared->u.array.dim[u], size_t, dim[u], hsize_t);
        ret_value->shared->u.array.nelem *= (size_t)dim[u];
    } 

    
    ret_value->shared->size = ret_value->shared->parent->shared->size * ret_value->shared->u.array.nelem;

    
    if (base->shared->force_conv == true)
        ret_value->shared->force_conv = true;

    
    ret_value->shared->version = MAX(base->shared->version, H5O_DTYPE_VERSION_2);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5Tget_array_ndims(hid_t type_id)
{
    H5T_t *dt;        
    int    ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (dt->shared->type != H5T_ARRAY)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an array datatype");

    
    ret_value = H5T__get_array_ndims(dt);

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5T__get_array_ndims(const H5T_t *dt)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(dt);
    assert(dt->shared->type == H5T_ARRAY);

    
    FUNC_LEAVE_NOAPI((int)dt->shared->u.array.ndims)
} 

int
H5Tget_array_dims2(hid_t type_id, hsize_t dims[] )
{
    H5T_t *dt;        
    int    ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (dt->shared->type != H5T_ARRAY)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an array datatype");

    
    if ((ret_value = H5T__get_array_dims(dt, dims)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unable to get dimension sizes");
done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5T__get_array_dims(const H5T_t *dt, hsize_t dims[])
{
    unsigned u; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(dt);
    assert(dt->shared->type == H5T_ARRAY);

    
    if (dims)
        for (u = 0; u < dt->shared->u.array.ndims; u++)
            dims[u] = dt->shared->u.array.dim[u];

    
    FUNC_LEAVE_NOAPI((int)dt->shared->u.array.ndims)
} 

#ifndef H5_NO_DEPRECATED_SYMBOLS

hid_t
H5Tarray_create1(hid_t base_id, int ndims, const hsize_t dim[],
                 const int H5_ATTR_UNUSED perm[])
{
    H5T_t   *base;      
    H5T_t   *dt = NULL; 
    unsigned u;         
    hid_t    ret_value; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (ndims < 1 || ndims > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid dimensionality");
    if (!dim)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "no dimensions specified");
    for (u = 0; u < (unsigned)ndims; u++)
        if (!(dim[u] > 0))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "zero-sized dimension specified");
    if (NULL == (base = (H5T_t *)H5I_object_verify(base_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not an valid base datatype");

    
    if (NULL == (dt = H5T__array_create(base, (unsigned)ndims, dim)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to create datatype");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register datatype");

done:
    if (ret_value < 0)
        if (dt && H5T_close_real(dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, H5I_INVALID_HID, "can't release datatype");

    FUNC_LEAVE_API(ret_value)
} 

int
H5Tget_array_dims1(hid_t type_id, hsize_t dims[] , int H5_ATTR_UNUSED perm[] )
{
    H5T_t *dt;        
    int    ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (dt->shared->type != H5T_ARRAY)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an array datatype");

    
    if ((ret_value = H5T__get_array_dims(dt, dims)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unable to get dimension sizes");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 
