/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Amodule.h" 

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5SMprivate.h" 

typedef struct H5A_fh_ud_cmp_t {
    
    H5F_t                          *f;             
    const char                     *name;          
    const H5A_dense_bt2_name_rec_t *record;        
    H5A_bt2_found_t                 found_op;      
    void                           *found_op_data; 

    
    int cmp; 
} H5A_fh_ud_cmp_t;

static herr_t H5A__dense_btree2_corder_store(void *native, const void *udata);
static herr_t H5A__dense_btree2_corder_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5A__dense_btree2_corder_encode(uint8_t *raw, const void *native, void *ctx);
static herr_t H5A__dense_btree2_corder_decode(const uint8_t *raw, void *native, void *ctx);
static herr_t H5A__dense_btree2_corder_debug(FILE *stream, int indent, int fwidth, const void *record,
                                             const void *_udata);

static herr_t H5A__dense_btree2_name_store(void *native, const void *udata);
static herr_t H5A__dense_btree2_name_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5A__dense_btree2_name_encode(uint8_t *raw, const void *native, void *ctx);
static herr_t H5A__dense_btree2_name_decode(const uint8_t *raw, void *native, void *ctx);
static herr_t H5A__dense_btree2_name_debug(FILE *stream, int indent, int fwidth, const void *record,
                                           const void *_udata);

static herr_t H5A__dense_fh_name_cmp(const void *obj, size_t obj_len, void *op_data);

const H5B2_class_t H5A_BT2_NAME[1] = {{
    
    H5B2_ATTR_DENSE_NAME_ID,          
    "H5B2_ATTR_DENSE_NAME_ID",        
    sizeof(H5A_dense_bt2_name_rec_t), 
    NULL,                             
    NULL,                             
    H5A__dense_btree2_name_store,     
    H5A__dense_btree2_name_compare,   
    H5A__dense_btree2_name_encode,    
    H5A__dense_btree2_name_decode,    
    H5A__dense_btree2_name_debug      
}};

const H5B2_class_t H5A_BT2_CORDER[1] = {{
    
    H5B2_ATTR_DENSE_CORDER_ID,          
    "H5B2_ATTR_DENSE_CORDER_ID",        
    sizeof(H5A_dense_bt2_corder_rec_t), 
    NULL,                               
    NULL,                               
    H5A__dense_btree2_corder_store,     
    H5A__dense_btree2_corder_compare,   
    H5A__dense_btree2_corder_encode,    
    H5A__dense_btree2_corder_decode,    
    H5A__dense_btree2_corder_debug      
}};

static herr_t
H5A__dense_fh_name_cmp(const void *obj, size_t obj_len, void *_udata)
{
    H5A_fh_ud_cmp_t *udata = (H5A_fh_ud_cmp_t *)_udata; 
    H5A_t           *attr  = NULL;                      
    bool   took_ownership  = false;   
    herr_t ret_value       = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL ==
        (attr = (H5A_t *)H5O_msg_decode(udata->f, NULL, H5O_ATTR_ID, obj_len, (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, FAIL, "can't decode attribute");

    
    udata->cmp = strcmp(udata->name, attr->shared->name);

    
    if (udata->cmp == 0 && udata->found_op) {
        
        if (udata->record->flags & H5O_MSG_FLAG_SHARED)
            H5SM_reconstitute(&(attr->sh_loc), udata->f, H5O_ATTR_ID, udata->record->id);

        
        attr->shared->crt_idx = udata->record->corder;

        
        if ((udata->found_op)(attr, &took_ownership, udata->found_op_data) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTOPERATE, FAIL, "attribute found callback failed");
    } 

done:
    
    if (attr && !took_ownership)
        H5O_msg_free(H5O_ATTR_ID, attr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_btree2_name_store(void *_nrecord, const void *_udata)
{
    const H5A_bt2_ud_ins_t   *udata   = (const H5A_bt2_ud_ins_t *)_udata;
    H5A_dense_bt2_name_rec_t *nrecord = (H5A_dense_bt2_name_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    nrecord->id     = udata->id;
    nrecord->flags  = udata->common.flags;
    nrecord->corder = udata->common.corder;
    nrecord->hash   = udata->common.name_hash;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_name_compare(const void *_bt2_udata, const void *_bt2_rec, int *result)
{
    const H5A_bt2_ud_common_t      *bt2_udata = (const H5A_bt2_ud_common_t *)_bt2_udata;
    const H5A_dense_bt2_name_rec_t *bt2_rec   = (const H5A_dense_bt2_name_rec_t *)_bt2_rec;
    herr_t                          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(bt2_udata);
    assert(bt2_rec);

    
    if (bt2_udata->name_hash < bt2_rec->hash)
        *result = (-1);
    else if (bt2_udata->name_hash > bt2_rec->hash)
        *result = 1;
    else {
        H5A_fh_ud_cmp_t fh_udata; 
        H5HF_t         *fheap;    

        
        assert(bt2_udata->name_hash == bt2_rec->hash);

        
        
        fh_udata.f             = bt2_udata->f;
        fh_udata.name          = bt2_udata->name;
        fh_udata.record        = bt2_rec;
        fh_udata.found_op      = bt2_udata->found_op;
        fh_udata.found_op_data = bt2_udata->found_op_data;

        
        fh_udata.cmp = 0;

        
        if (bt2_rec->flags & H5O_MSG_FLAG_SHARED)
            fheap = bt2_udata->shared_fheap;
        else
            fheap = bt2_udata->fheap;
        assert(fheap);

        
        if (H5HF_op(fheap, &bt2_rec->id, H5A__dense_fh_name_cmp, &fh_udata) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");

        
        *result = fh_udata.cmp;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_btree2_name_encode(uint8_t *raw, const void *_nrecord, void H5_ATTR_UNUSED *ctx)
{
    const H5A_dense_bt2_name_rec_t *nrecord = (const H5A_dense_bt2_name_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(raw, nrecord->id.id, (size_t)H5O_FHEAP_ID_LEN);
    raw += H5O_FHEAP_ID_LEN;
    *raw++ = nrecord->flags;
    UINT32ENCODE(raw, nrecord->corder);
    UINT32ENCODE(raw, nrecord->hash);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_name_decode(const uint8_t *raw, void *_nrecord, void H5_ATTR_UNUSED *ctx)
{
    H5A_dense_bt2_name_rec_t *nrecord = (H5A_dense_bt2_name_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(nrecord->id.id, raw, (size_t)H5O_FHEAP_ID_LEN);
    raw += H5O_FHEAP_ID_LEN;
    nrecord->flags = *raw++;
    UINT32DECODE(raw, nrecord->corder);
    UINT32DECODE(raw, nrecord->hash);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_name_debug(FILE *stream, int indent, int fwidth, const void *_nrecord,
                             const void H5_ATTR_UNUSED *_udata)
{
    const H5A_dense_bt2_name_rec_t *nrecord = (const H5A_dense_bt2_name_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(stream, "%*s%-*s {%016" PRIx64 ", %02" PRIx8 ", %u, %08" PRIx32 "}\n", indent, "", fwidth,
            "Record:", nrecord->id.val, nrecord->flags, (unsigned)nrecord->corder, nrecord->hash);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_corder_store(void *_nrecord, const void *_udata)
{
    const H5A_bt2_ud_ins_t     *udata   = (const H5A_bt2_ud_ins_t *)_udata;
    H5A_dense_bt2_corder_rec_t *nrecord = (H5A_dense_bt2_corder_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    nrecord->id     = udata->id;
    nrecord->flags  = udata->common.flags;
    nrecord->corder = udata->common.corder;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_corder_compare(const void *_bt2_udata, const void *_bt2_rec, int *result)
{
    const H5A_bt2_ud_common_t        *bt2_udata = (const H5A_bt2_ud_common_t *)_bt2_udata;
    const H5A_dense_bt2_corder_rec_t *bt2_rec   = (const H5A_dense_bt2_corder_rec_t *)_bt2_rec;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(bt2_udata);
    assert(bt2_rec);

    
    if (bt2_udata->corder < bt2_rec->corder)
        *result = -1;
    else if (bt2_udata->corder > bt2_rec->corder)
        *result = 1;
    else
        *result = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_corder_encode(uint8_t *raw, const void *_nrecord, void H5_ATTR_UNUSED *ctx)
{
    const H5A_dense_bt2_corder_rec_t *nrecord = (const H5A_dense_bt2_corder_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(raw, nrecord->id.id, (size_t)H5O_FHEAP_ID_LEN);
    raw += H5O_FHEAP_ID_LEN;
    *raw++ = nrecord->flags;
    UINT32ENCODE(raw, nrecord->corder);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_corder_decode(const uint8_t *raw, void *_nrecord, void H5_ATTR_UNUSED *ctx)
{
    H5A_dense_bt2_corder_rec_t *nrecord = (H5A_dense_bt2_corder_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(nrecord->id.id, raw, (size_t)H5O_FHEAP_ID_LEN);
    raw += H5O_FHEAP_ID_LEN;
    nrecord->flags = *raw++;
    UINT32DECODE(raw, nrecord->corder);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_btree2_corder_debug(FILE *stream, int indent, int fwidth, const void *_nrecord,
                               const void H5_ATTR_UNUSED *_udata)
{
    const H5A_dense_bt2_corder_rec_t *nrecord = (const H5A_dense_bt2_corder_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(stream, "%*s%-*s {%016" PRIx64 ", %02" PRIx8 ", %u}\n", indent, "", fwidth,
            "Record:", nrecord->id.val, nrecord->flags, (unsigned)nrecord->corder);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
