\name{ccc}
\alias{ccc}
\title{Lin's concordance correlation coefficient}
\description{
  Calculates Lin's concordance correlation coefficient for evaluating the degree of 
  agreement between measurements generated by two different methods.
}
\usage{
ccc(x, data, method = "z-transform", level = 0.95, equal.means = FALSE, 
    ustat = TRUE, subset, na.action)
}
\arguments{
  \item{x}{ a formula or a numeric matrix or an object that can be coerced to a numeric matrix.}
  \item{data}{ an optional data frame (or similar: see \code{\link{model.frame}}), used only if
    \code{x} is a formula. By default the variables are taken from \code{environment(formula)}.}
  \item{method}{ a character string, indicating the method for the computation of the required 
    confidence interval. Options available are \code{"z-transform"} or \code{"asymp"} (see details 
    in Lin, 1989, 2000).}
  \item{level}{ the confidence level required, must be a single number between 0 and 1 
    (by default 95\%).}
  \item{equal.means}{ logical, should the means of the measuring devices be considered equal? 
    In which case the restricted estimation is carried out under this assumption.}
  \item{ustat}{ logical, should the concordance correlation coefficient be estimated using 
    \emph{U}-statistics?}
  \item{subset}{ an optional expression indicating the subset of the rows of data that should 
    be used in the fitting process.}
  \item{na.action}{ a function that indicates what should happen when the data contain NAs.}
}
\value{
  A list with class \code{'ccc'} containing the following named components:
  \item{call}{ a list containing an image of the \code{ccc} call that produced the object.}
  \item{x}{ \code{data.frame} used in the estimation process.}
  \item{ccc}{ estimate of the concordance correlation coefficient.}
  \item{var.ccc}{ asymptotic variance of the concordance correlation coefficient estimate.}
  \item{accuracy}{ estimate of the accuracy (or bias) coefficient that measures how far the 
    best-fit line deviates from a line at 45 degrees. No deviation from the 45 degree line 
    occurs when \code{accuracy} = 1.}
  \item{precision}{ estimate of the precision (or Pearson correlation) coefficient.}
  \item{shifts}{ list with the location and scale shifts.}
  \item{z}{ \emph{Z}-transformation parameter estimate.}
  \item{var.z}{ asymptotic variance of the \emph{Z}-transformation parameter estimate.}
  \item{confint}{ confidence interval for the Lin's concordance correlation coefficient.}
  \item{bland}{ a data frame with two columns containing the \code{average} of each pair of measurements, 
    and \code{difference} between the measurements.}
  \item{center}{ the estimated mean vector.}
  \item{cov}{ the estimated covariance matrix.}
  \item{ustat}{ available only if \code{ustat = TRUE}, in which case this element corresponds 
    to a list containing the following elements \code{rhoc}, \code{var.rhoc}, \code{ustat}, 
    and \code{cov}.}
  \item{Restricted}{ available only if \code{equal.means = TRUE}, in which case this element 
  corresponds to a list containing the following elements \code{ccc}, \code{accuracy}, \code{precision}, 
    \code{shifts}, \code{center}, and \code{cov}.}
}
\references{
  Bland, J., Altman, D. (1986). 
  Statistical methods for assessing agreement between two methods of clinical measurement. 
  \emph{The Lancet} \bold{327}, 307-310.

  King, T.S., Chinchilli, V.M. (2001).
  A generalized concordance correlation coefficient for continuous and categorical data.
  \emph{Statistics in Medicine} \bold{20}, 2131-2147.

  King, T.S., Chinchilli, V.M. (2001).
  Robust estimators of the concordance correlation coefficient.
  \emph{Journal of Biopharmaceutical Statistics} \bold{11}, 83-105.

  Lin, L. (1989). 
  A concordance correlation coefficient to evaluate reproducibility. 
  \emph{Biometrics} \bold{45}, 255-268.

  Lin, L. (2000). 
  A note on the concordance correlation coefficient. 
  \emph{Biometrics} \bold{56}, 324-325.

  Vallejos, R., Osorio, F., Ferrer, C. (2025+).
  A new coefficient to measure agreement between two continuous variables.
  Working paper.
}
\examples{
## data in Fig.1 from Bland and Altman (1986).
x <- list(Large = c(494,395,516,434,476,557,413,442,650,433,
          417,656,267,478,178,423,427),
          Mini  = c(512,430,520,428,500,600,364,380,658,445,
          432,626,260,477,259,350,451))
x <- as.data.frame(x)

plot(Mini ~ Large, data = x, xlim = c(100,800), ylim = c(100,800),
     xlab = "PERF by Large meter", ylab = "PERF by Mini meter")
abline(c(0,1), col = "gray", lwd = 2)

## estimating CCC
z <- ccc(~ Mini + Large, data = x, method = "asymp")
z
## output:
# Call:
# ccc(x = ~ Mini + Large, data = x, method = "asymp")
#
# Coefficients:
#  estimate  variance  accuracy precision 
#   0.9427    0.0008    0.9994    0.9433 
#
# Asymptotic 95% confidence interval:
#     CCC     SE  lower  upper 
#  0.9427 0.0286 0.8867 0.9988 
}

\keyword{univar}
