% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coef_simu.R, R/confint.cirls.R, R/vcov.cirls.R
\name{coef_simu}
\alias{coef_simu}
\alias{confint.cirls}
\alias{vcov.cirls}
\title{Simulate coefficients, calculate Confidence Intervals and Variance-Covariance Matrix for a \code{cirls} object.}
\usage{
coef_simu(object, nsim = 1000)

\method{confint}{cirls}(object, parm, level = 0.95, nsim = 1000, ...)

\method{vcov}{cirls}(object, nsim = 1000, ...)
}
\arguments{
\item{object}{A fitted \code{cirls} object.}

\item{nsim}{The number of simulations to consider. Corresponds to \code{n} in \link[TruncatedNormal:tmvnorm]{rtmvnorm}. See details().}

\item{parm}{A specification of which parameters to compute the confidence intervals for. Either a vector of numbers or a vector of names. If missing, all parameters are considered.}

\item{level}{The confidence level required.}

\item{...}{Further arguments passed to or from other methods. Currently ignored.}
}
\value{
For \code{confint}, a two-column matrix with columns giving lower and upper confidence limits for each parameter.

For \code{vcov}, a matrix of the estimated covariances between the parameter estimates of the model.

For \code{coef_simu}, a matrix with \code{nsim} rows containing simulated coefficients.
}
\description{
\code{confint} computes confidence intervals for one of more parameters in a GLM fitted via \link[=cirls.fit]{cirls.fit}. \code{vcov} compute the variance-covariance matrix of the parameters. Both methods are based on \code{coef_simu} that simulates coefficients from a Truncated Multivariate Normal distribution. These methods supersede the default \link[stats:confint]{confint} and \link[stats:vcov]{vcov} methods for \code{cirls} objects.
}
\details{
These functions are custom methods for \link[=cirls.fit]{cirls} objects to supersede the default methods used for \link[stats:glm]{glm} objects.

Both methods rely on the fact that \eqn{C\hat{\beta}} (with \eqn{C} the constraint matrix) follows a \emph{Truncated Multivariate Normal} distribution
\deqn{C\hat{\beta} \sim TMVN(C\beta, CVC^T), l, u}
where TMVN represents a truncated Multivariate Normal distribution. \eqn{C} is the constraint matrix (\code{object$control$Cmat}) with bound \eqn{l} and \eqn{u}, while \eqn{V} is the unconstrained variance-covariance matrix (such as returned by \code{vcov.glm}).

\code{coef_simu} simulates from the TMVN above and transforms back the realisations into the coefficients space. These realisations are then used by the \code{confint} and \code{vcov} methods which compute empirical quantiles and variance-covariance matrix, respectively. \code{coef_simu} is called internally by \code{confint} and \code{vcov} and doesn't need to be used directly, but it can be used to check other summaries of the coefficients distribution.
}
\note{
These methods only work when \code{Cmat} is of full row rank. If not the case, \code{Cmat} can be inspected through \code{\link[=check_cmat]{check_cmat()}}.
}
\examples{
####################################################
# Isotonic regression

#----- Perform isotonic regression

# Generate data
set.seed(222)
p1 <- 5; p2 <- 3
x1 <- matrix(rnorm(100 * p1), 100, p1)
x2 <- matrix(rnorm(100 * p2), 100, p2)
b1 <- runif(p1) |> sort()
b2 <- runif(p2)
y <- x1 \%*\% b1 + x2 \%*\% b2 + rnorm(100, sd = 2)

# Fit model
Ciso <- diff(diag(p1))
resiso <- glm(y ~ x1 + x2, method = cirls.fit, Cmat = list(x1 = Ciso))

#----- Extract uncertainty

# Extract variance covariance
vcov(resiso)

# Extract confidence intervals
confint(resiso)

# We can extract the usual unconstrained vcov
summary(resiso)$cov.scaled
all.equal(vcov(resiso), summary(resiso)$cov.scaled)

# Simulate from the distribution of coefficients
sims <- coef_simu(resiso, nsim = 10)

# Check that all simulated coefficient vectors are feasible
apply(resiso$Cmat \%*\% t(sims) >= resiso$lb, 2, all)
}
\references{
Geweke, J.F., 1996. Bayesian Inference for Linear Models Subject to Linear Inequality Constraints, in: Lee, J.C., Johnson, W.O., Zellner, A. (Eds.), Modelling and Prediction Honoring Seymour Geisser. \emph{Springer, New York, NY}, pp. 248–263. \doi{10.1007/978-1-4612-2414-3_15}

Botev, Z.I., 2017, The normal law under linear restrictions: simulation and estimation via minimax tilting, \emph{Journal of the Royal Statistical Society, Series B}, \strong{79} (\strong{1}), pp. 1–24.
}
\seealso{
\link[TruncatedNormal:tmvnorm]{rtmvnorm} for the underlying routine to simulate from a TMVN. \code{\link[=check_cmat]{check_cmat()}} to check if the contraint matrix can be reduced.
}
