% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/S3Extract.R
\name{extract}
\alias{extract}
\alias{extract.CDM}
\alias{extract.validation}
\alias{extract.sim.data}
\alias{extract.fit}
\title{Extract Components from Qval Package Objects}
\usage{
extract(object, what, ...)

\method{extract}{CDM}(object, what, ...)

\method{extract}{validation}(object, what, ...)

\method{extract}{sim.data}(object, what, ...)

\method{extract}{fit}(object, what, ...)
}
\arguments{
\item{object}{An object of class \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, \code{\link[Qval]{sim.data}}, or \code{\link[Qval]{fit}}.}

\item{what}{A character string specifying the name of the component to extract.}

\item{...}{Additional arguments (currently ignored).}
}
\value{
The requested component. The return type depends on the specified \code{what} and the class of the \code{object}.
}
\description{
A unified extractor function for retrieving internal components from objects produced by the \pkg{Qval} package. 
This method allows users to access key elements such as model results, validation logs, and simulation settings 
in a structured and object-oriented manner.
}
\details{
This generic extractor supports three core object classes: \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, 
\code{\link[Qval]{sim.data}} and \code{\link[Qval]{fit}}. 
It is intended to streamline access to commonly used internal components without manually referencing object slots.
The available components for each class are listed below:

\describe{
  \item{\strong{\code{CDM}}}{Cognitive Diagnosis Model fitting results. Available components:
    \describe{
      \item{analysis.obj}{The internal model fitting object (e.g., GDINA or Baseline Model).}
      \item{alpha}{Estimated attribute profiles (EAP estimates) for each respondent.}
      \item{P.alpha.Xi}{Posterior distribution of latent attribute patterns.}
      \item{alpha.P}{Marginal attribute mastery probabilities (estimated).}
      \item{P.alpha}{Prior attribute probabilities at convergence.}
      \item{Deviance}{Negative twice the marginal log-likelihood (model deviance).}
      \item{npar}{Number of free parameters estimated in the model.}
      \item{AIC}{Akaike Information Criterion.}
      \item{BIC}{Bayesian Information Criterion.}
      \item{call}{The original model-fitting function call.}
      \item{...}{Can \code{\link[GDINA]{extract}} corresponding value from the \code{\link[GDINA]{GDINA}} object.}
    }}

  \item{\strong{\code{validation}}}{Q-matrix validation results. Available components:
    \describe{
      \item{Q.orig}{The original Q-matrix submitted for validation.}
      \item{Q.sug}{The suggested (revised) Q-matrix after validation.}
      \item{time.cost}{Total computation time for the validation procedure.}
      \item{process}{Log of Q-matrix modifications across iterations.}
      \item{iter}{Number of iterations performed during validation.}
      \item{priority}{Attribute priority matrix (available for PAA-based methods only).}
      \item{Hull.fit}{Data required to plot the Hull method results (for Hull-based validation only).}
      \item{call}{The original function call used for validation.}
    }}

  \item{\strong{\code{sim.data}}}{Simulated data and parameters used in cognitive diagnosis simulation studies:
    \describe{
      \item{dat}{Simulated dichotomous response matrix (\eqn{N \times I}).}
      \item{Q}{Q-matrix used to generate the data.}
      \item{attribute}{True latent attribute profiles (\eqn{N \times K}).}
      \item{catprob.parm}{Item-category conditional success probabilities (list format).}
      \item{delta.parm}{Item-level delta parameters (list format).}
      \item{higher.order.parm}{Higher-order model parameters (if used).}
      \item{mvnorm.parm}{Parameters for the multivariate normal attribute distribution (if used).}
      \item{LCprob.parm}{Latent class-based success probability matrix.}
      \item{call}{The original function call that generated the simulated data.}
    }}
    \item{\strong{\code{fit}}}{Relative fit indices (-2LL, AIC, BIC, CAIC, SABIC) and absolute fit indices (\eqn{M_2} test, \eqn{RMSEA_2}, SRMSR):
    \describe{
      \item{npar}{The number of parameters.}
      \item{-2LL}{The Deviance.}
      \item{AIC}{The Akaike information criterion.}
      \item{BIC}{The Bayesian information criterion.}
      \item{CAIC}{The consistent Akaike information criterion.}
      \item{SABIC}{The Sample-size Adjusted BIC.}
      \item{M2}{A vector consisting of \eqn{M_2} statistic, degrees of freedom, significance level, and \eqn{RMSEA_2} (Liu, Tian, & Xin, 2016).}
      \item{SRMSR}{The standardized root mean squared residual (SRMSR; Ravand & Robitzsch, 2018).}
    }}
}
}
\section{Methods (by class)}{
\itemize{
\item \code{extract(CDM)}: Extract fields from a CDM object

\item \code{extract(validation)}: Extract fields from a validation object

\item \code{extract(sim.data)}: Extract fields from a sim.data object

\item \code{extract(fit)}: Extract fields from a fit object

}}
\examples{
library(Qval)
set.seed(123)

\donttest{
################################################################
# Example 1: sim.data extraction                               #
################################################################
Q <- sim.Q(3, 10)
data.obj <- sim.data(Q, N = 200)
extract(data.obj, "dat")


################################################################
# Example 2: CDM extraction                                    #
################################################################
CDM.obj <- CDM(data.obj$dat, Q)
extract(CDM.obj, "alpha")
extract(CDM.obj, "AIC")


################################################################
# Example 3: validation extraction                             #
################################################################
validation.obj <- validation(data.obj$dat, Q, CDM.obj)
Q.sug <- extract(validation.obj, "Q.sug")
print(Q.sug)

################################################################
# Example 4: fit extraction                                    #
################################################################
fit.obj <- fit(data.obj$dat, Q.sug, model="GDINA")
extract(fit.obj, "M2")
}

}
\references{
Khaldi, R., Chiheb, R., & Afa, A.E. (2018). Feed-forward and Recurrent Neural Networks for Time Series Forecasting: Comparative Study. In: Proceedings of the International Conference on Learning and Optimization Algorithms: Theory and Applications (LOPAL 18). Association for Computing Machinery, New York, NY, USA, Article 18, 1–6. DOI: 10.1145/3230905.3230946.

Liu, Y., Tian, W., & Xin, T. (2016). An application of M2 statistic to evaluate the fit of cognitive diagnostic models. Journal of Educational and Behavioral Statistics, 41, 3–26. DOI: 10.3102/1076998615621293.

Ravand, H., & Robitzsch, A. (2018). Cognitive diagnostic model of best choice: a study of reading comprehension. Educational Psychology, 38, 1255–1277. DOI: 10.1080/01443410.2018.1489524.
}
