#' Cross-validation for penalized Lorenz regression
#'
#' \code{PLR.CV} performs k-fold cross-validation to select the grid and penalization parameters of the penalized Lorenz regression.
#'
#' @param object An object of class \code{"PLR"}, i.e., the result of a call to \code{\link{Lorenz.Reg}} where \code{penalty} is either \code{"SCAD"} or \code{"LASSO"}.
#' @param k An integer specifying the number of folds in the k-fold cross-validation.
#' @param seed.CV An optional integer specifying a seed for reproducibility in the creation of the folds. Default is \code{NULL}, in which case no seed is imposed.
#' @param parallel A logical or numeric value controlling parallel computation. If \code{TRUE}, parallel processing is enabled using the maximum available cores minus one. If a numeric value is provided, it specifies the number of cores to use. Default is \code{FALSE} (no parallelization).
#' @param ... Additional arguments passed to either the cross-validation function \code{\link[rsample]{vfold_cv}} from the \pkg{rsample} package or the underlying fit functions (\code{\link{Lorenz.GA}}, \code{\link{Lorenz.FABS}}, or \code{\link{Lorenz.SCADFABS}}). By default, the fit function uses the same parameters as in the original call to \code{Lorenz.Reg}, but these can be overridden by explicitly passing them in \code{...}.
#'
#' @return An object of class \code{c("PLR_cv", "PLR")}.
#' The returned list contains the following components:
#' \describe{
#'    \item{\code{path}}{See \code{\link{Lorenz.Reg}} for the original path. The cross-validation score is added.}
#'    \item{\code{lambda.idx}}{A vector indicating the index of the optimal lambda obtained by each selection method.}
#'    \item{\code{grid.idx}}{A vector indicating the index of the optimal grid parameter obtained by each selection method.}
#'    \item{\code{splits}}{A list storing the data splits used for cross-validation, as generated by \code{\link[rsample]{vfold_cv}}.}
#' }
#' Note: The returned object may have additional classes such as \code{"PLR_boot"} if bootstrap was performed.
#'
#' @details The parameter \code{seed.CV} allows for local seed setting to control randomness in the generation of the folds.
#' The specified seed is applied to the respective part of the computation, and the seed is reverted to its previous state after the operation.
#' This ensures that the seed settings do not interfere with the global random state or other parts of the code.
#'
#' @seealso \code{\link{Lorenz.Reg}}, \code{\link{Lorenz.SCADFABS}}, \code{\link{Lorenz.FABS}}, \code{\link{Lorenz.boot}}
#'
#' @section References:
#' Jacquemain, A., C. Heuchenne, and E. Pircalabelu (2024). A penalised bootstrap estimation procedure for the explained Gini coefficient. \emph{Electronic Journal of Statistics 18(1) 247-300}.
#'
#' @examples
#' \dontshow{
#' utils::example(Lorenz.Reg, echo = FALSE)
#' }
#' # Continuing the Lorenz.Reg(.) example:
#' PLR_CV <- PLR.CV(PLR, k = 5, seed.CV = 123)
#' # The object now inherits from the class "PLR_cv".
#' # Hence the methods (also) display the results obtained by cross-validation.
#' print(PLR_CV)
#' summary(PLR_CV)
#' coef(PLR_CV, pars.idx = "CV")
#' predict(PLR_CV, pars.idx = "CV")
#' plot(PLR_CV)
#' plot(PLR_CV, type = "diagnostic") # Plot of the scores depending on the grid and penalty parameters
#'
#' @importFrom rsample vfold_cv analysis
#' @importFrom doParallel registerDoParallel stopImplicitCluster
#' @importFrom parallel detectCores
#' @importFrom foreach foreach '%do%' '%dopar%'
#' @importFrom stats setNames runif
#' @importFrom utils modifyList
#'
#' @export

PLR.CV<-function(object,
                 k,
                 seed.CV=NULL,
                 parallel=FALSE,
                 ...
){

  # 0. Checks ----
  if(!inherits(object,"PLR")) stop("object must be the output of a penalized Lorenz regression.")

  args <- list(...)

  # 1. Arguments of the cv ----
  data <- cbind(object$y, object$x)
  cv_args <- args[names(args) %in% names(formals(vfold_cv))]
  PLR_args <- list("penalty"=object$penalty, "grid.arg"=object$grid.arg,
                   "grid.value"=object$grid.value, "lambda.list"=object$lambda.list)
  fit_formals <- switch(object$penalty,
                        "LASSO" = names(formals(Lorenz.FABS)),
                        "SCAD" = names(formals(Lorenz.SCADFABS)))
  fit_args <- modifyList(object$fit_args, args[names(args) %in% fit_formals])
  fit_args <- c(fit_args, PLR_args)

  # 1. statistic for cv ----
  cv.f <- function(split) {
    indices <- split$in_id
    x.train <- object$x[indices,,drop=FALSE]
    y.train <- object$y[indices]
    x.test <- object$x[-indices,,drop=FALSE]
    y.test <- object$y[-indices]
    if(!is.null(object$weights)){
      w.train <- object$weights[indices]
      w.test <- object$weights[-indices]
    }else{
      w.test <- w.train <- NULL
    }
    train.LR <- do.call(PLR.fit, c(list(y = y.train, x = x.train, weights = w.train), fit_args))
    # With penalized reg, the algorithm may stop sooner than in the original sample.
    # Therefore the paths would be shorter and the objects would not have the same size
    compare.paths <- function(path.long,path.short){
      lth.diff <- ncol(path.long) - ncol(path.short)
      if(lth.diff > 0) path.short <- cbind(path.short,replicate(lth.diff,path.short[,ncol(path.short)]))
      return(path.short)
    }
    train.LR$path <- lapply(1:length(object$path),function(i)compare.paths(object$path[[i]],train.LR$path[[i]]))
    # Computation of the CV score
    theta.train <- lapply(train.LR$path,function(x)x[(nrow(x)-ncol(object$x)+1):nrow(x),])
    cv.score <- PLR.scores(y.test,x.test,w.test,theta.train)
    cv.vec <- unlist(cv.score)

    return(cv.vec)

  }

  # 2. cv folds ----

  data <- as.data.frame(cbind("y" = object$y, object$x))

  if (!is.null(seed.CV)) {
    if(exists(".Random.seed")){
      old <- .Random.seed
    }else{
      runif(1)
      old <- .Random.seed
    }
    on.exit( { .Random.seed <<- old } )
    set.seed(seed.CV)
  }
  cv_folds <- vfold_cv(data, v = k, ...)

  # 3. cv computations ----

  j <- NULL

  if(parallel){
    if(is.numeric(parallel)){
      registerDoParallel(parallel)
    }else{
      numCores <- detectCores()
      registerDoParallel(numCores-1)
    }
    cv.j <- foreach(j=1:k) %dopar% {
      cv.f(cv_folds$splits[[j]])
    }
    stopImplicitCluster()
  }else{
    cv.j <- foreach(j=1:k) %do% {
      cv.f(cv_folds$splits[[j]])
    }
  }

  cv_out <- t(sapply(1:k,function(j)cv.j[[j]]))

  # 4. Adding to the PLR object ----
  path.sizes <- sapply(object$path,ncol)
  path.size <- sum(path.sizes)
  lth.path <- length(path.sizes)
  # the cv score is the mean of the cv scores across the folds
  cv_total <- colMeans(cv_out)
  # Adding cv score to the path
  idx <- lapply(1:lth.path,function(i)(cumsum(path.sizes)-path.sizes+1)[i]:cumsum(path.sizes)[i])
  val.cv <- lapply(idx,function(i)cv_total[i])
  lth.theta <- ncol(object$x)
  lth <- nrow(object$path[[1]]) # Same for all anyway (what changes is ncol)
  for (i in 1:lth.path){
    path.tmp <- rbind(object$path[[i]][1:(lth-lth.theta),],
                      "CV score" = val.cv[[i]])
    object$path[[i]] <- rbind(path.tmp,
                              object$path[[i]][(lth-lth.theta+1):lth,])
  }
  lth <- lth + 1
  # Best pair (grid,lambda) in terms of CV score
  path.wl <- unlist(sapply(path.sizes,function(x)1:x))
  path.wt <- rep(1:lth.path,times=path.sizes)
  wl <- path.wl[which.max(cv_total)]
  wt <- path.wt[which.max(cv_total)]
  object$grid.idx <- c(object$grid.idx,"CV"=wt)
  object$lambda.idx <- c(object$lambda.idx,"CV"=wl)
  object$splits <- cv_folds$splits

  class(object) <- c("PLR_cv",class(object))

  return(object)

}
