% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MaxContribution.R
\name{MaxContribution}
\alias{MaxContribution}
\title{Find major contributions to aggregates}
\usage{
MaxContribution(
  x,
  y,
  n = 1,
  decreasing = TRUE,
  index = FALSE,
  groups = NULL,
  return2 = FALSE
)
}
\arguments{
\item{x}{A (sparse) dummy matrix}

\item{y}{Vector of input values (contributors)}

\item{n}{Number of contributors to be found}

\item{decreasing}{Ordering parameter. Smallest contributors found when \code{FALSE}.}

\item{index}{Indices to \code{y} returned when TRUE}

\item{groups}{When non-NULL, major contributions after aggregation within groups.
Cannot be combined with \code{index = TRUE}.
The missing group category is excluded.}

\item{return2}{When \code{TRUE}, two matrices are returned, \code{value} and \code{id}.
The latter contains indices when \code{group} is \code{NULL} and otherwise a character matrix of groups.}
}
\value{
Matrix with lagest contributions in first column, second largest in second column and so on.
Alternative output when using parameters \code{index} or \code{return2}.
}
\description{
Assuming aggregates are calculated via a dummy matrix by
\code{z = t(x) \%*\% y},
the \code{n} largest contributions are found (value or index) for each aggregate.
}
\examples{
library(SSBtools)

z <- SSBtoolsData("sprt_emp_withEU")
z$age[z$age == "Y15-29"] <- "young"
z$age[z$age == "Y30-64"] <- "old"

a <- ModelMatrix(z, formula = ~age + geo, crossTable = TRUE)

cbind(as.data.frame(a$crossTable), MaxContribution(a$modelMatrix, z$ths_per, 1))
cbind(a$crossTable, MaxContribution(a$modelMatrix, z$ths_per, 10))
cbind(a$crossTable, MaxContribution(a$modelMatrix, z$ths_per, 10, index = TRUE))

# Both types of output can be achieved with return2 = TRUE)
identical(MaxContribution(a$modelMatrix, z$ths_per, 10, return2 = TRUE),
          list(value =  MaxContribution(a$modelMatrix, z$ths_per, 10), 
               id =  MaxContribution(a$modelMatrix, z$ths_per, 10, index = TRUE)))

b <- ModelMatrix(z[, -4], crossTable = TRUE, inputInOutput = c(TRUE, FALSE, TRUE))

k <- cbind(b$crossTable, MaxContribution(b$modelMatrix, z$ths_per, 10))

gr18 <- paste0("g", 1:18)                          # Each row is a group
k18 <- cbind(b$crossTable, MaxContribution(b$modelMatrix, z$ths_per, 10, groups = gr18))
identical(k, k18) # TRUE

gr9 <- paste0("g", as.integer(10 * z$ths_per)\%\%10) # 9 groups from decimal
k9 <- cbind(b$crossTable, MaxContribution(b$modelMatrix, z$ths_per, 10, groups = gr9))

k18[c(4, 13, 17, 33), ]
k9[c(4, 13, 17, 33), ]

# Group info obtained with return2 = TRUE
k9_id <- cbind(b$crossTable, MaxContribution(b$modelMatrix, z$ths_per, 10, groups = gr9, 
                                             return2 = TRUE)$id)
k9_id[c(4, 13, 17, 33), ]


# Verify similarity
z$y <- z$ths_per + (1:nrow(z))/100  # to avoid equal values
id1 <- MaxContribution(b$modelMatrix, z$y, 10, index = TRUE)
id1[!is.na(id1)] <- paste0("g", id1[!is.na(id1)])
mc2 <- MaxContribution(b$modelMatrix, z$y, 10, groups = gr18, return2 = TRUE)
id2 <- mc2$id
identical(id1, id2)

}
\seealso{
\code{\link[SSBtools]{ModelMatrix}}
}
\author{
Øyvind Langsrud
}
