% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes2S_v6_seq.r
\name{bayes.2S_seq}
\alias{bayes.2S_seq}
\title{bayes.2S_seq: Bayesian Prevalence-Incidence Mixture Model (sequential processing)}
\usage{
bayes.2S_seq(
  Vobs,
  Z.X = NULL,
  Z.W = NULL,
  r = NULL,
  dist.X = "weibull",
  kappa = 0.5,
  update.kappa = FALSE,
  kappa.prior = NULL,
  ndraws = 1000,
  prop.sd.X = NULL,
  chains = 3,
  thining = 1,
  parallel = TRUE,
  update.till.converge = FALSE,
  maxit = Inf,
  conv.crit = "upper",
  min_effss = chains * 10,
  beta.prior = "norm",
  beta.prior.X = 1,
  sig.prior.X = 1,
  tau.w = 1,
  fix.sigma.X = FALSE,
  prev.run = NULL,
  update.burnin = TRUE,
  ndraws.update = NULL,
  prev = TRUE,
  vanilla = FALSE,
  ndraws.naive = 5000,
  naive.run.prop.sd.X = prop.sd.X,
  par.exp = FALSE,
  collapsed.g = TRUE,
  k.prior = 1,
  fix.k = FALSE
)
}
\arguments{
\item{Vobs}{A list of length \eqn{n} of numeric vectors representing screening times. The first element of each vector should always be \code{0} and the last element \code{Inf} in the case of right censoring.}

\item{Z.X}{A numeric matrix of dimension \eqn{n \times p_x} containing covariates for the AFT incidence model. Missing values are not allowed.}

\item{Z.W}{A numeric matrix of dimension \eqn{n \times p_w} containing covariates for the probit prevalence model. Missing values are not allowed.}

\item{r}{A binary vector of length \eqn{n} indicating whether a baseline test was conducted (\code{1} for yes, \code{0} for no / missing baseline test).}

\item{dist.X}{Character. Specifies the distribution for the time-to-incidence variable; choices are \code{'weibull'}, \code{'lognormal'}, or \code{'loglog'} (log-logistic).}

\item{kappa}{Numeric. The test sensitivity value to be used if \code{update.kappa = FALSE}; otherwise, the starting value for estimating \eqn{\kappa}.}

\item{update.kappa}{Logical. If \code{TRUE}, the test sensitivity (\eqn{\kappa}) is updated during the Gibbs sampler.}

\item{kappa.prior}{A numeric vector of length 2. When specified, a Beta distribution prior is used for \eqn{\kappa}, centered at \code{kappa.prior[1]} with standard deviation \code{kappa.prior[2]}. If \code{NULL}, a uniform prior (Beta(1,1)) is used.}

\item{ndraws}{Integer. The total number of MCMC draws for the main Gibbs sampler.}

\item{prop.sd.X}{Numeric. The standard deviation for the proposal (jumping) distribution in the Metropolis sampler used for updating \eqn{\beta_{xj}}. Can be searched for automatically using \link{search.prop.sd}.}

\item{chains}{Integer. The number of MCMC chains to run.}

\item{thining}{Integer. The thinning interval for the MCMC sampler.}

\item{parallel}{Logical. If \code{TRUE}, parallel processing is enabled for the Gibbs sampler. Each chain is assigned to one core (via the \code{foreach} package). Alternatively, use \link{bayes.2S_seq} which employs a \code{for} loop over the chains.}

\item{update.till.converge}{Logical. If \code{TRUE}, the model is updated iteratively until convergence criteria are met. Convergence is assessed using the Gelman-Rubin diagnostic (\eqn{R<1.1}) and a minimum effective sample size (\code{min_effss}) for each parameter, respectively.}

\item{maxit}{Numeric. The maximum number of MCMC draws allowed before interrupting the update process when \code{update.till.converge} is enabled. Default is \code{Inf} (i.e., no automatic interruption).}

\item{conv.crit}{Character. Specifies whether the convergence check uses the point estimate (\code{'point'}) or the upper bound (\code{'upper'}) of the Gelman-Rubin diagnostic estimate (\eqn{\hat{R}}).}

\item{min_effss}{Integer. The minimum effective sample size required for each parameter before convergence is accepted during iterative updating.}

\item{beta.prior}{Character. Specifies the type of prior for the incidence regression coefficients (\eqn{\beta_{xj}}); options are \code{'norm'} for normal and \code{'t'} for student-t.}

\item{beta.prior.X}{Numeric. The hyperparameter for the prior distribution of the regression coefficients (\eqn{\beta_{xj}}) in the AFT incidence model. For a normal prior, this is the standard deviation; for a student-t prior, it represents the degrees of freedom. The default produces a standard-normal prior.}

\item{sig.prior.X}{Numeric. The hyperparameter (standard deviation) for a half-normal prior on the scale parameter (\eqn{\sigma}) of the AFT incidence model.}

\item{tau.w}{Numeric. The hyperparameter (standard deviation) for the normal prior distribution of the regression coefficients (\eqn{\beta_{wj}}) in the probit prevalence model. The default produces a standard-normal prior.}

\item{fix.sigma.X}{Logical. If \code{TRUE}, the scale parameter (\eqn{\sigma}) in the AFT incidence model is fixed at the value provided in \code{sig.prior.X}; if \code{FALSE}, it is updated.}

\item{prev.run}{Optional. An object of class \code{BayesPIM} containing results from a previous run. When provided, data and prior settings are adopted from the previous run, and the MCMC chain continues from the last draw.}

\item{update.burnin}{Logical. If \code{TRUE} (default) and \code{prev.run} is provided, the burn-in period is updated to half of the total draws (sum of previous and current runs); otherwise, the burn-in is maintained as half of the draws from the initial run.}

\item{ndraws.update}{Integer. The number of MCMC draws for updating a previous run or for convergence updates. If unspecified, \code{ndraws} is used.}

\item{prev}{Logical. If \code{TRUE}, prevalence adjustment is applied; if \code{FALSE}, prevalence is assumed to be zero.}

\item{vanilla}{Logical. If \code{TRUE}, a vanilla run is performed that assumes no prevalence adjustment and fixes \eqn{\kappa = 1}, equivalent to a standard Bayesian interval-censored survival regression.}

\item{ndraws.naive}{Integer. The number of MCMC draws for a preliminary vanilla run used to obtain starting values. Increase if initial values lead to issues (e.g., an infinite posterior).}

\item{naive.run.prop.sd.X}{Numeric. The standard deviation for the proposal distribution used in the vanilla run. Adjust only if the acceptance rate is significantly off target, as indicated by an interruption message.}

\item{par.exp}{Logical. If \code{TRUE}, the parameter expansion technique of Liu & Wiu (1999) with a Haar prior is employed for updating the regression coefficients (\eqn{\beta_{wj}}) in the prevalence model. Experimental: tests suggest that it does not improve convergence or reduce autocorrelation.}

\item{collapsed.g}{Logical. If \code{TRUE}, the latent prevalence class membership update in the Gibbs sampler is integrated (collapsed) over the latent incidence time variable. This setting is recommended to improve convergence.}

\item{k.prior}{Experimental prior parameter for generalized gamma; currently not used.}

\item{fix.k}{Experimental fixing of prior parameter for generalized gamma; currently not used.}
}
\value{
A list containing the following elements:

\item{par.X.all}{An \code{mcmc.list} of MCMC samples for the incidence and prevalence model parameters.}
\item{par.X.bi}{An \code{mcmc.list} of MCMC samples for the incidence and prevalence model parameters after burn-in removal.}
\item{X}{A matrix of posterior draws for the latent event times \eqn{x_i}, with one column per chain.}
\item{C}{A matrix of posterior draws for prevalence class membership \eqn{g_i}, with one column per chain.}
\item{ac.X}{A matrix with MCMC draws in rows and chains in columns, where each row indicates whether the Metropolis sampler accepted (1) or rejected (0) a sample.}
\item{ac.X.cur}{Same as \code{ac.X}, but only for the last update.}
\item{dat}{A data frame containing the last observed interval.}
\item{priors}{A list of prior specifications for the model parameters, including \code{beta.prior.X} (incidence regression coefficients) and \code{sig.prior.X} (scale parameter for the AFT model).}
\item{runtime}{The total runtime of the MCMC sampler.}

Additionally, most input arguments are returned as part of the output for reference.
}
\description{
Estimates the Pattern Mixture model of Klausch et al. (2025) using a Bayesian Gibbs sampler. Identical to \link{bayes.2S} but uses a \code{for} loop over MCMC chains instead of \code{foreach}.
}
\details{
This function is dentical to \link{bayes.2S} with the only difference being that the \code{chains} MCMC chains are run in sequence using a \code{for} loop instead of parallel processing. This can be useful if operating systems do not support \code{foreach} or for simulation studies that parallize replication of experiments using \code{foreach} and/or need a worker that does not apply \code{foreach} internally.
}
\examples{
\donttest{
library(BayesPIM)

# Generate data according to the Klausch et al. (2025) PIM
set.seed(2025)
dat <- gen.dat(kappa = 0.7, n = 1e3, theta = 0.2,
               p = 1, p.discrete = 1,
               beta.X = c(0.2, 0.2), beta.W = c(0.2, 0.2),
               v.min = 20, v.max = 30, mean.rc = 80,
               sigma.X = 0.2, mu.X = 5, dist.X = "weibull",
               prob.r = 1)

# Initial model fit with fixed test sensitivity kappa (approx. 4-12 minutes runtime)
mod <- bayes.2S_seq(Vobs = dat$Vobs,
                Z.X = dat$Z,
                Z.W = dat$Z,
                r = dat$r,
                kappa = 0.7,
                update.kappa = FALSE,
                ndraws = 1e4,
                chains = 4,
                prop.sd.X = 0.008,
                parallel = TRUE,
                dist.X = "weibull")

# Inspect results
mod$runtime  # Runtime of the Gibbs sampler
plot(trim.mcmc(mod$par.X.all, thining = 10))  # MCMC chains including burn-in, also see ?trim.mcmc
plot(trim.mcmc(mod$par.X.bi, thining = 10))   # MCMC chains excluding burn-in
apply(mod$ac.X, 2, mean)  # Acceptance rates per chain
gelman.diag(mod$par.X.bi)  # Gelman convergence diagnostics

# Model updating
mod_update <- bayes.2S(prev.run = mod)      # Adds ndraws additional MCMC draws
mod_update <- bayes.2S(prev.run = mod, 
                       ndraws.update = 1e3) # Adds ndraws.update additional MCMC draws

# Example with kappa estimated/updated
mod2 <- bayes.2S_seq(Vobs = dat$Vobs,
                 Z.X = dat$Z,
                 Z.W = dat$Z,
                 r = dat$r,
                 kappa = 0.7,
                 update.kappa = TRUE,
                 kappa.prior = c(0.7, 0.1), # Beta prior, mean = 0.7, s.d. = 0.1
                 ndraws = 1e4,
                 chains = 4,
                 prop.sd.X = 0.008,
                 parallel = TRUE,
                 dist.X = "weibull")

# Inspect results
mod2$runtime # runtime of Gibbs sampler
plot( trim.mcmc( mod2$par.X.all, thining = 10) ) # kappa returned as part of the mcmc.list
}

}
\references{
T. Klausch, B. I. Lissenberg-Witte, and V. M. Coupe (2024). "A Bayesian prevalence-incidence mixture model for screening outcomes with misclassification.", doi:10.48550/arXiv.2412.16065.

J. S. Liu and Y. N. Wu, “Parameter Expansion for Data Augmentation,” Journal of the American Statistical Association, vol. 94, no. 448, pp. 1264–1274, 1999, doi: 10.2307/2669940.
}
