use super::bitmask::BitMask;
use super::EMPTY;
use core::{mem, ptr};





cfg_if! {
    if #[cfg(any(
        target_pointer_width = "64",
        target_arch = "aarch64",
        target_arch = "x86_64",
        target_arch = "wasm32",
    ))] {
        type GroupWord = u64;
        type NonZeroGroupWord = core::num::NonZeroU64;
    } else {
        type GroupWord = u32;
        type NonZeroGroupWord = core::num::NonZeroU32;
    }
}

pub(crate) type BitMaskWord = GroupWord;
pub(crate) type NonZeroBitMaskWord = NonZeroGroupWord;
pub(crate) const BITMASK_STRIDE: usize = 8;

#[allow(clippy::cast_possible_truncation, clippy::unnecessary_cast)]
pub(crate) const BITMASK_MASK: BitMaskWord = 0x8080_8080_8080_8080_u64 as GroupWord;
pub(crate) const BITMASK_ITER_MASK: BitMaskWord = !0;


#[inline]
fn repeat(byte: u8) -> GroupWord {
    GroupWord::from_ne_bytes([byte; Group::WIDTH])
}





#[derive(Copy, Clone)]
pub(crate) struct Group(GroupWord);





#[allow(clippy::use_self)]
impl Group {

    pub(crate) const WIDTH: usize = mem::size_of::<Self>();





    #[inline]
    pub(crate) const fn static_empty() -> &'static [u8; Group::WIDTH] {
        #[repr(C)]
        struct AlignedBytes {
            _align: [Group; 0],
            bytes: [u8; Group::WIDTH],
        }
        const ALIGNED_BYTES: AlignedBytes = AlignedBytes {
            _align: [],
            bytes: [EMPTY; Group::WIDTH],
        };
        &ALIGNED_BYTES.bytes
    }


    #[inline]
    #[allow(clippy::cast_ptr_alignment)] // unaligned load
    pub(crate) unsafe fn load(ptr: *const u8) -> Self {
        Group(ptr::read_unaligned(ptr.cast()))
    }



    #[inline]
    #[allow(clippy::cast_ptr_alignment)]
    pub(crate) unsafe fn load_aligned(ptr: *const u8) -> Self {

        debug_assert_eq!(ptr as usize & (mem::align_of::<Self>() - 1), 0);
        Group(ptr::read(ptr.cast()))
    }



    #[inline]
    #[allow(clippy::cast_ptr_alignment)]
    pub(crate) unsafe fn store_aligned(self, ptr: *mut u8) {

        debug_assert_eq!(ptr as usize & (mem::align_of::<Self>() - 1), 0);
        ptr::write(ptr.cast(), self.0);
    }











    #[inline]
    pub(crate) fn match_byte(self, byte: u8) -> BitMask {


        let cmp = self.0 ^ repeat(byte);
        BitMask((cmp.wrapping_sub(repeat(0x01)) & !cmp & repeat(0x80)).to_le())
    }



    #[inline]
    pub(crate) fn match_empty(self) -> BitMask {



        BitMask((self.0 & (self.0 << 1) & repeat(0x80)).to_le())
    }



    #[inline]
    pub(crate) fn match_empty_or_deleted(self) -> BitMask {

        BitMask((self.0 & repeat(0x80)).to_le())
    }


    #[inline]
    pub(crate) fn match_full(self) -> BitMask {
        self.match_empty_or_deleted().invert()
    }





    #[inline]
    pub(crate) fn convert_special_to_empty_and_full_to_deleted(self) -> Self {







        let full = !self.0 & repeat(0x80);
        Group(!full + (full >> 7))
    }
}
