% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/verify_precision.R
\name{verify_precision}
\alias{verify_precision}
\title{Precision Verification Against Manufacturer Claims}
\usage{
verify_precision(
  x,
  claimed_cv = NULL,
  claimed_sd = NULL,
  mean_value = NULL,
  alpha = 0.05,
  alternative = c("less", "two.sided", "greater"),
  conf_level = 0.95,
  value = "value",
  day = "day",
  run = NULL,
  ...
)
}
\arguments{
\item{x}{Either a numeric vector of measurements, a \code{precision_study} object,
or a data frame containing precision study data.}

\item{claimed_cv}{Manufacturer's claimed coefficient of variation (as percent).
Either \code{claimed_cv} or \code{claimed_sd} must be provided.}

\item{claimed_sd}{Manufacturer's claimed standard deviation. Either \code{claimed_cv}
or \code{claimed_sd} must be provided.}

\item{mean_value}{Mean concentration of the sample. Required when \code{x} is a
numeric vector and \code{claimed_cv} is used, or when \code{claimed_sd} is used and
CV-based comparison is desired.}

\item{alpha}{Significance level for the hypothesis test (default: 0.05).}

\item{alternative}{Type of alternative hypothesis:
\code{"less"} (default) tests if observed is not worse than claimed,
\code{"two.sided"} tests for any difference,
\code{"greater"} tests if observed is worse than claimed.}

\item{conf_level}{Confidence level for intervals (default: 0.95).}

\item{value}{Character string specifying the column name containing
measurement values when \code{x} is a data frame. Default is \code{"value"}.}

\item{day}{Character string specifying the column name for day identifier
when \code{x} is a data frame. Default is \code{"day"}.}

\item{run}{Character string specifying the column name for run identifier
when \code{x} is a data frame. Default is \code{NULL}.}

\item{...}{Additional arguments passed to \code{precision_study()} when \code{x} is
a data frame.}
}
\value{
An object of class \code{c("verify_precision", "valytics_precision", "valytics_result")},
which is a list containing:

\describe{
\item{input}{List with input data and metadata:
\itemize{
\item \code{n}: Number of observations
\item \code{df}: Degrees of freedom for the test
\item \code{mean_value}: Mean of measurements
\item \code{source}: Description of input source
}
}
\item{observed}{List with observed precision:
\itemize{
\item \code{sd}: Observed standard deviation
\item \code{cv_pct}: Observed CV (percent)
\item \code{variance}: Observed variance
}
}
\item{claimed}{List with manufacturer's claimed precision:
\itemize{
\item \code{sd}: Claimed SD
\item \code{cv_pct}: Claimed CV (percent)
\item \code{variance}: Claimed variance
}
}
\item{test}{List with hypothesis test results:
\itemize{
\item \code{statistic}: Chi-square test statistic
\item \code{df}: Degrees of freedom
\item \code{p_value}: P-value
\item \code{alternative}: Alternative hypothesis used
\item \code{method}: Test method description
}
}
\item{verification}{List with verification outcome:
\itemize{
\item \code{verified}: Logical; TRUE if precision is verified
\item \code{ratio}: Ratio of observed variance to claimed variance
\item \code{cv_ratio}: Ratio of observed CV to claimed CV
\item \code{upper_verification_limit}: Upper limit for verification
}
}
\item{ci}{List with confidence intervals:
\itemize{
\item \code{sd_ci}: CI for standard deviation
\item \code{cv_ci}: CI for CV (percent)
\item \code{variance_ci}: CI for variance
}
}
\item{settings}{List with analysis settings}
\item{call}{The matched function call}
}
}
\description{
Compares observed precision (CV or SD) to manufacturer's claimed performance
using statistical hypothesis testing. This function implements verification
protocols for validating that an analytical method meets specified precision
goals.
}
\details{
\strong{Statistical Test:}

The verification uses a chi-square test comparing observed variance to
claimed variance:

\deqn{\chi^2 = \frac{(n-1) \cdot s^2}{\sigma^2_{claimed}}}

where \eqn{s^2} is the observed sample variance and \eqn{\sigma^2_{claimed}}
is the manufacturer's claimed variance.

\strong{Hypothesis Testing:}

For \code{alternative = "less"} (default, recommended for verification):
\itemize{
\item H0: True precision is worse than or equal to claimed
\item H1: True precision is better than or equal to claimed
\item Verification passes if observed precision is not significantly worse
}

For typical verification studies, the observed CV should not exceed the
manufacturer's claimed CV by more than expected from sampling variability.

\strong{Verification Limit:}

The upper verification limit (UVL) represents the maximum observed CV that
would still be consistent with the claimed CV at the given significance level:

\deqn{UVL = CV_{claimed} \cdot \sqrt{\frac{\chi^2_{1-\alpha, df}}{df}}}

If observed CV <= UVL, precision is verified.
}
\section{Input Options}{

The function accepts three types of input:
\itemize{
\item \strong{Numeric vector}: Raw measurements (simplest case)
\item \strong{precision_study object}: Uses within-laboratory precision from
a previous analysis
\item \strong{Data frame}: Runs \code{precision_study()} internally with specified
factors
}
}

\examples{
# Example 1: Verify precision from raw measurements
set.seed(42)
measurements <- rnorm(25, mean = 100, sd = 3.5)

# Manufacturer claims CV = 4\%
result <- verify_precision(measurements, claimed_cv = 4, mean_value = 100)
print(result)

# Example 2: Verify precision from a precision_study object
prec_data <- data.frame(
  day = rep(1:5, each = 5),
  value = rnorm(25, mean = 100, sd = 3)
)
prec_data$value <- prec_data$value + rep(rnorm(5, 0, 1.5), each = 5)

prec <- precision_study(prec_data, value = "value", day = "day")
result <- verify_precision(prec, claimed_cv = 5)
print(result)

# Example 3: Verify precision directly from data frame
result <- verify_precision(
  prec_data,
  claimed_cv = 5,
  value = "value",
  day = "day"
)
print(result)

}
\references{
Chesher D (2008). Evaluating assay precision. \emph{Clinical Biochemist
Reviews}, 29(Suppl 1):S23-S26.

ISO 5725-6:1994. Accuracy (trueness and precision) of measurement methods
and results - Part 6: Use in practice of accuracy values.
}
\seealso{
\code{\link[=precision_study]{precision_study()}} for full precision analysis,
\code{\link[=ate_assessment]{ate_assessment()}} for total error assessment
}
