% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/enhance_caim.R
\name{enhance_caim}
\alias{enhance_caim}
\title{Enhance canopy image}
\usage{
enhance_caim(
  caim,
  m = NULL,
  sky_blue = NULL,
  sigma = NULL,
  w_red = 0,
  thr = NULL,
  fuzziness = NULL,
  gamma = NULL
)
}
\arguments{
\item{caim}{\linkS4class{SpatRaster}. The return of a call to \code{\link[=read_caim]{read_caim()}} or
\code{\link[=read_caim_raw]{read_caim_raw()}}.}

\item{m}{\linkS4class{SpatRaster}. A mask. For hemispherical photographs, check
\code{\link[=mask_hs]{mask_hs()}}.}

\item{sky_blue}{\linkS4class{color}. Is the \code{target_color} argument to be passed to
\code{\link[=membership_to_color]{membership_to_color()}}. Default (\code{NULL}) is the equivalent to enter
\code{sRGB(0.1, 0.4, 0.8)}--the HEX color code is #1A66CC, it can be entered
into a search engine (such as Mozilla Firefox) to see a color swatch.}

\item{sigma}{Numeric vector of length one. Use \code{NULL} (default) to estimate
it automatically as the euclidean distance between \code{target_color} and grey
in the \emph{CIE LAB} color space.}

\item{w_red}{Numeric vector of length one. Weight of the red channel. A
single layer image is calculated as a weighted average of the blue and red
channels. This layer is used as lightness information. The weight of the
blue is the complement of \code{w_red}.}

\item{thr}{Numeric vector of length one. Location parameter of the logistic
membership function. Use \code{NULL} to estimate it automatically with
\code{\link[=thr_isodata]{thr_isodata()}}.}

\item{fuzziness}{Numeric vector of length one. This number is a constant
value that scales \code{mem}. Use \code{NULL} to estimate it automatically as the
midpoint between the maximum and minimum values of \code{lightness}.}

\item{gamma}{Numeric vector of length one. This is for applying a gamma back
correction to the lightness information (see Details and argument \code{w_red}).}
}
\value{
An object of class \linkS4class{SpatRaster} (with same pixel dimensions
than \code{caim}) that should show more contrast between the sky and plant
pixels than any of the individual bands from \code{caim}; if not, different
parameters should be tested.
}
\description{
This function was first proposed in \insertCite{Diaz2015;textual}{rcaiman}.
It uses the color perceptual attributes (hue, lightness, and chroma) to
enhance the contrast between the sky and plants through fuzzy classification.
It applies the next classification rules (here expressed in natural
language): clear sky is blue and clouds decrease its chroma; if clouds are
highly dense, then the sky is achromatic, and, in such cases, it can be light
or dark; everything that does not match this description is not sky. These
linguistic rules were translated to math language by means of fuzzy logic.
This translation was thoughtfully explained in the aforementioned article.
}
\details{
This is a pixel-wise methodology that evaluates the possibility for a pixel
to be member of the class \emph{Gap}. High score could mean either high membership
to \code{sky_blue} or, in the case of achromatic pixels, a high membership to
values above \code{thr}. The algorithm internally uses \code{\link[=membership_to_color]{membership_to_color()}} and
\code{\link[=local_fuzzy_thresholding]{local_fuzzy_thresholding()}}. The argument \code{sky_blue} is the \code{target_color}
of the former function and its output is the argument \code{mem} of the latter
function.

The argument \code{sky_blue} can be obtained from a photograph that clearly shows
the sky. Then, it can be used to process all the others photograph taken with
the same equipment, configuration, and protocol.

Via the \code{gamma} argument, \code{\link[=gbc]{gbc()}} can be internally used to back-correct the
values passed to \code{\link[=local_fuzzy_thresholding]{local_fuzzy_thresholding()}}.
}
\note{
If you use this function in your research, please cite
\insertCite{Diaz2015;textual}{rcaiman} in addition to this package
(\verb{citation("rcaiman"}).

The default value of argument \code{m} is the equivalent to enter
\code{!is.na(caim$Red)}. See the Details section in \code{\link[=local_fuzzy_thresholding]{local_fuzzy_thresholding()}}
to understand how this argument can modify the output.
}
\examples{
\dontrun{
caim <- read_caim()
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)
m <- !is.na(z)

mn <- quantile(caim$Blue[m], 0.01)
mx <- quantile(caim$Blue[m], 0.99)
r <- normalize(caim$Blue, mn, mx, TRUE)

bin <- find_sky_pixels(r, z, a)
mblt <- ootb_mblt(r, z, a, bin)

mx <- optim_normalize(caim, mblt$bin)
mn <- min(caim[m])

sky_blue_sample <- crop_caim(caim, c(327, 239), 41, 89)
plotRGB(normalize(sky_blue_sample, mn, mx, TRUE)*255)
sky_blue <- apply(sky_blue_sample[], 2, median) \%>\%
  normalize(., mn, mx) \%>\%
  as.numeric() \%>\%
  matrix(., ncol = 3) \%>\%
  sRGB()
hex(sky_blue)
# Use hex() to obtain the HEX color code. To see a color swatch, enter the
# HEX code into a search engine (such as Mozilla Firefox).
# NOTE: see extract_dn() for an alternative method to obtain sky_blue

as(sky_blue, "HSV") #saturatio (S) is low
# To obtain same hue (H) but greater saturation
sky_blue <- HSV(239, 0.85, 0.5) \%>\% as(., "sRGB") \%>\% as(., "LAB")
hex(sky_blue)

caim <- normalize(caim, mx = mx, force_range = TRUE)
ecaim <- enhance_caim(caim, m, sky_blue = sky_blue)
plot(ecaim)
plot(caim$Blue)

## to compare
plot(apply_thr(ecaim, thr_isodata(ecaim[m])))
plot(apply_thr(caim$Blue, thr_isodata(caim$Blue[m])))
}
}
\references{
\insertAllCited{}
}
\seealso{
Other Pre-processing Functions: 
\code{\link{gbc}()},
\code{\link{local_fuzzy_thresholding}()},
\code{\link{membership_to_color}()},
\code{\link{normalize}()}
}
\concept{Pre-processing Functions}
