% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_contrasts.R
\name{estimate_contrasts}
\alias{estimate_contrasts}
\alias{estimate_contrasts.default}
\title{Estimate Marginal Contrasts}
\usage{
estimate_contrasts(model, ...)

\method{estimate_contrasts}{default}(
  model,
  contrast = NULL,
  by = NULL,
  predict = NULL,
  ci = 0.95,
  comparison = "pairwise",
  estimate = NULL,
  p_adjust = "none",
  transform = NULL,
  keep_iterations = FALSE,
  effectsize = NULL,
  iterations = 200,
  es_type = "cohens.d",
  backend = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{A statistical model.}

\item{...}{Other arguments passed, for instance, to \code{\link[insight:get_datagrid]{insight::get_datagrid()}},
to functions from the \strong{emmeans} or \strong{marginaleffects} package, or to process
Bayesian models via \code{\link[bayestestR:describe_posterior]{bayestestR::describe_posterior()}}. Examples:
\itemize{
\item \code{insight::get_datagrid()}: Argument such as \code{length}, \code{digits} or \code{range}
can be used to control the (number of) representative values. For integer
variables, \code{protect_integers} modulates whether these should also be
treated as numerics, i.e. values can have fractions or not.
\item \strong{marginaleffects}: Internally used functions are \code{avg_predictions()} for
means and contrasts, and \code{avg_slope()} for slopes. Therefore, arguments for
instance like \code{vcov}, \code{equivalence}, \code{df}, \code{slope}, \code{hypothesis} or even
\code{newdata} can be passed to those functions. A \code{weights} argument is passed
to the \code{wts} argument in \code{avg_predictions()} or \code{avg_slopes()}, however,
weights can only be applied when \code{estimate} is \code{"average"} or
\code{"population"} (i.e. for those marginalization options that do not use data
grids). Other arguments, such as \code{re.form} or \code{allow.new.levels}, may be
passed to \code{predict()} (which is internally used by \emph{marginaleffects}) if
supported by that model class.
\item \strong{emmeans}: Internally used functions are \code{emmeans()} and \code{emtrends()}.
Additional arguments can be passed to these functions.
\item Bayesian models: For Bayesian models, parameters are cleaned using
\code{describe_posterior()}, thus, arguments like, for example, \code{centrality},
\code{rope_range}, or \code{test} are passed to that function.
\item Especially for \code{estimate_contrasts()} with integer focal predictors, for
which contrasts should be calculated, use argument \code{integer_as_numeric} to
set the maximum number of unique values in an integer predictor to treat
that predictor as "discrete integer" or as numeric. For the first case,
contrasts are calculated between values of the predictor, for the latter,
contrasts of slopes are calculated. If the integer has more than
\code{integer_as_numeric} unique values, it is treated as numeric. Defaults to
\code{5}.
\item For count regression models that use an offset term, use \verb{offset = <value>}
to fix the offset at a specific value. Or use \code{estimate = "average"}, to
average predictions over the distribution of the offset (if appropriate).
}}

\item{contrast}{A character vector indicating the name of the variable(s) for
which to compute the contrasts, optionally including representative values or
levels at which contrasts are evaluated (e.g., \code{contrast="x=c('a','b')"}).}

\item{by}{The (focal) predictor variable(s) at which to evaluate the desired
effect / mean / contrasts. Other predictors of the model that are not
included here will be collapsed and "averaged" over (the effect will be
estimated across them). \code{by} can be a character (vector) naming the focal
predictors, optionally including representative values or levels at which
focal predictors are evaluated (e.g., \code{by = "x = c(1, 2)"}). When \code{estimate}
is \emph{not} \code{"average"}, the \code{by} argument is used to create a "reference grid"
or "data grid" with representative values for the focal predictors. In this
case, \code{by} can also be list of named elements. See details in
\code{\link[insight:get_datagrid]{insight::get_datagrid()}} to learn more about how to create data grids for
predictors of interest.}

\item{predict}{Is passed to the \code{type} argument in \code{emmeans::emmeans()} (when
\code{backend = "emmeans"}) or in \code{marginaleffects::avg_predictions()} (when
\code{backend = "marginaleffects"}). Valid options for \code{predict} are:
\itemize{
\item \code{backend = "marginaleffects"}: \code{predict} can be \code{"response"}, \code{"link"},
\code{"inverse_link"} or any valid \code{type} option supported by model's class
\code{predict()} method (e.g., for zero-inflation models from package
\strong{glmmTMB}, you can choose \code{predict = "zprob"} or \code{predict = "conditional"}
etc., see \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB}). By default, when \code{predict = NULL},
the most appropriate transformation is selected, which usually returns
predictions or contrasts on the response-scale. The \code{"inverse_link"} is a
special option, comparable to \emph{marginaleffects}' \code{invlink(link)} option. It
will calculate predictions on the link scale and then back-transform to the
response scale.
\item \code{backend = "emmeans"}: \code{predict} can be \code{"response"}, \code{"link"}, \code{"mu"},
\code{"unlink"}, or \code{"log"}. If \code{predict = NULL} (default), the most appropriate
transformation is selected (which usually is \code{"response"}). See also
\href{https://CRAN.R-project.org/package=emmeans/vignettes/transformations.html}{this vignette}.
}

See also section \emph{Predictions on different scales}.}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}

\item{comparison}{Specify the type of contrasts or tests that should be
carried out.
\itemize{
\item When \code{backend = "emmeans"}, can be one of \code{"pairwise"}, \code{"poly"},
\code{"consec"}, \code{"eff"}, \code{"del.eff"}, \code{"mean_chg"}, \code{"trt.vs.ctrl"},
\code{"dunnett"}, \code{"wtcon"} and some more. To test multiple hypotheses jointly
(usually used for factorial designs), \code{comparison} can also be \code{"joint"}.
See also \code{method} argument in \link[emmeans:contrast]{emmeans::contrast} and the
\code{?emmeans::emmc-functions}.
\item For \code{backend = "marginaleffects"}, can be a numeric value, vector, or
matrix, a string equation specifying the hypothesis to test, a string
naming the comparison method, a formula, or a function. For options not
described below, see documentation of \link[marginaleffects:comparisons]{marginaleffects::comparisons},
\href{https://marginaleffects.com/bonus/hypothesis.html}{this website} and
section \emph{Comparison options} below.
\itemize{
\item String: One of \code{"pairwise"}, \code{"reference"}, \code{"sequential"}, \code{"meandev"}
\code{"meanotherdev"}, \code{"poly"}, \code{"helmert"}, or \code{"trt_vs_ctrl"}. To test
multiple hypotheses jointly (usually used for factorial designs),
\code{comparison} can also be \code{"joint"}. In this case, use the \code{test} argument
to specify which test should be conducted: \code{"F"} (default) or \code{"Chi2"}.
\item String: Two special string options are \code{"inequality"} and
\code{"inequality_pairwise"}. \code{comparison = "inequality"} computes the
marginal effect inequality summary of categorical predictors' overall
effects, respectively, the comprehensive effect of an independent
variable across all outcome categories of a nominal or ordinal dependent
variable (total marginal effect, see \emph{Mize and Han, 2025}).
\code{comparison = "inequality_pairwise"} computes the difference (pairwise
comparisons) between marginal effects inequality measures.
\item String equation: To identify parameters from the output, either specify
the term name, or \code{"b1"}, \code{"b2"} etc. to indicate rows, e.g.:\code{"hp = drat"},
\code{"b1 = b2"}, or \code{"b1 + b2 + b3 = 0"}.
\item Formula: A formula like \code{comparison ~ pairs | group}, where the left-hand
side indicates the type of comparison (\code{difference} or \code{ratio}), the
right-hand side determines the pairs of estimates to compare (\code{reference},
\code{sequential}, \code{meandev}, etc., see string-options). Optionally, comparisons
can be carried out within subsets by indicating the grouping variable
after a vertical bar ( \code{|}).
\item A custom function, e.g. \code{comparison = myfun}, or
\code{comparison ~ I(my_fun(x)) | groups}.
\item If contrasts should be calculated (or grouped by) factors, \code{comparison}
can also be a matrix that specifies factor contrasts (see 'Examples').
}
}}

\item{estimate}{The \code{estimate} argument determines how predictions are
averaged ("marginalized") over variables not specified in \code{by} or \code{contrast}
(non-focal predictors). It controls whether predictions represent a "typical"
individual, an "average" individual from the sample, or an "average"
individual from a broader population.
\itemize{
\item \code{"typical"} (Default): Calculates predictions for a balanced data grid
representing all combinations of focal predictor levels (specified in \code{by}).
For non-focal numeric predictors, it uses the mean; for non-focal
categorical predictors, it marginalizes (averages) over the levels. This
represents a "typical" observation based on the data grid and is useful for
comparing groups. It answers: "What would the average outcome be for a
'typical' observation?". This is the default approach when estimating
marginal means using the \emph{emmeans} package.
\item \code{"average"}: Calculates predictions for each observation in the sample and
then averages these predictions within each group defined by the focal
predictors. This reflects the sample's actual distribution of non-focal
predictors, not a balanced grid. It answers: "What is the predicted value
for an average observation in my data?"
\item \code{"population"}: "Clones" each observation, creating copies with all
possible combinations of focal predictor levels. It then averages the
predictions across these "counterfactual" observations (non-observed
permutations) within each group. This extrapolates to a hypothetical
broader population, considering "what if" scenarios. It answers: "What is
the predicted response for the 'average' observation in a broader possible
target population?" This approach entails more assumptions about the
likelihood of different combinations, but can be more apt to generalize.
This is also the option that should be used for \strong{G-computation}
(\emph{Chatton and Rohrer 2024}).
}

You can set a default option for the \code{estimate} argument via \code{options()},
e.g. \code{options(modelbased_estimate = "average")}}

\item{p_adjust}{The p-values adjustment method for frequentist multiple
comparisons. Can be one of \code{"none"} (default), \code{"hochberg"}, \code{"hommel"},
\code{"bonferroni"}, \code{"BH"}, \code{"BY"}, \code{"fdr"}, \code{"tukey"}, \code{"sidak"}, \code{"sup-t"},
\code{"esarey"} or \code{"holm"}. The \code{"esarey"} option is specifically for the case of
Johnson-Neyman intervals, i.e. when calling \code{estimate_slopes()} with two
numeric predictors in an interaction term. \code{"sup-t"} computes simultaneous
confidence bands, also called sup-t confidence band (Montiel Olea &
Plagborg-Møller, 2019). Details for the other options can be found in the
p-value adjustment section of the \code{emmeans::test} documentation or
\code{?stats::p.adjust}. Note that certain options provided by the \strong{emmeans}
package are only available if you set \code{backend = "emmeans"}.}

\item{transform}{A function applied to predictions and confidence intervals
to (back-) transform results, which can be useful in case the regression
model has a transformed response variable (e.g., \code{lm(log(y) ~ x)}). For
Bayesian models, this function is applied to individual draws from the
posterior distribution, before computing summaries. Can also be \code{TRUE}, in
which case \code{insight::get_transformation()} is called to determine the
appropriate transformation-function. Note that no standard errors are returned
when transformations are applied.}

\item{keep_iterations}{If \code{TRUE}, will keep all iterations (draws) of
bootstrapped or Bayesian models. They will be added as additional columns
named \code{iter_1}, \code{iter_2}, and so on. If \code{keep_iterations} is a positive
number, only as many columns as indicated in \code{keep_iterations} will be added
to the output. You can reshape them to a long format by running
\code{\link[bayestestR:reshape_iterations]{bayestestR::reshape_iterations()}}.}

\item{effectsize}{Desired measure of standardized effect size, one of
\code{"emmeans"}, \code{"marginal"}, or \code{"boot"}. Default is \code{NULL}, i.e. no effect
size will be computed.}

\item{iterations}{The number of bootstrap resamples to perform.}

\item{es_type}{Specifies the type of effect-size measure to estimate when
using \code{effectsize = "boot"}. One of \code{"unstandardized"}, \code{"cohens.d"},
\code{"hedges.g"}, \code{"cohens.d.sigma"}, \code{"r"}, or \code{"akp.robust.d"}. See\code{ effect.type} argument of \link[bootES:bootES]{bootES::bootES} for details.}

\item{backend}{Whether to use \code{"marginaleffects"} (default) or \code{"emmeans"} as
a backend. Results are usually very similar. The major difference will be
found for mixed models, where \code{backend = "marginaleffects"} will also average
across random effects levels, producing "marginal predictions" (instead of
"conditional predictions", see Heiss 2022).

Another difference is that \code{backend = "marginaleffects"} will be slower than
\code{backend = "emmeans"}. For most models, this difference is negligible. However,
in particular complex models or large data sets fitted with \emph{glmmTMB} can be
significantly slower.

You can set a default backend via \code{options()}, e.g. use
\code{options(modelbased_backend = "emmeans")} to use the \strong{emmeans} package or
\code{options(modelbased_backend = "marginaleffects")} to set \strong{marginaleffects} as
default backend.}

\item{verbose}{Use \code{FALSE} to silence messages and warnings.}
}
\value{
A data frame of estimated contrasts.
}
\description{
Run a contrast analysis by estimating the differences between each level of a
factor. See also other related functions such as \code{\link[=estimate_means]{estimate_means()}}
and \code{\link[=estimate_slopes]{estimate_slopes()}}.
}
\details{
The \code{\link[=estimate_slopes]{estimate_slopes()}}, \code{\link[=estimate_means]{estimate_means()}} and \code{\link[=estimate_contrasts]{estimate_contrasts()}}
functions are forming a group, as they are all based on \emph{marginal}
estimations (estimations based on a model). All three are built on the
\strong{emmeans} or \strong{marginaleffects} package (depending on the \code{backend}
argument), so reading its documentation (for instance \code{\link[emmeans:emmeans]{emmeans::emmeans()}},
\code{\link[emmeans:emtrends]{emmeans::emtrends()}} or this \href{https://marginaleffects.com/}{website}) is
recommended to understand the idea behind these types of procedures.
\itemize{
\item Model-based \strong{predictions} is the basis for all that follows. Indeed,
the first thing to understand is how models can be used to make predictions
(see \code{\link[=estimate_link]{estimate_link()}}). This corresponds to the predicted response (or
"outcome variable") given specific predictor values of the predictors (i.e.,
given a specific data configuration). This is why the concept of \code{\link[insight:get_datagrid]{reference grid()}} is so important for direct predictions.
\item \strong{Marginal "means"}, obtained via \code{\link[=estimate_means]{estimate_means()}}, are an extension
of such predictions, allowing to "average" (collapse) some of the predictors,
to obtain the average response value at a specific predictors configuration.
This is typically used when some of the predictors of interest are factors.
Indeed, the parameters of the model will usually give you the intercept value
and then the "effect" of each factor level (how different it is from the
intercept). Marginal means can be used to directly give you the mean value of
the response variable at all the levels of a factor. Moreover, it can also be
used to control, or average over predictors, which is useful in the case of
multiple predictors with or without interactions.
\item \strong{Marginal contrasts}, obtained via \code{\link[=estimate_contrasts]{estimate_contrasts()}}, are
themselves at extension of marginal means, in that they allow to investigate
the difference (i.e., the contrast) between the marginal means. This is,
again, often used to get all pairwise differences between all levels of a
factor. It works also for continuous predictors, for instance one could also
be interested in whether the difference at two extremes of a continuous
predictor is significant.
\item Finally, \strong{marginal effects}, obtained via \code{\link[=estimate_slopes]{estimate_slopes()}}, are
different in that their focus is not values on the response variable, but the
model's parameters. The idea is to assess the effect of a predictor at a
specific configuration of the other predictors. This is relevant in the case
of interactions or non-linear relationships, when the effect of a predictor
variable changes depending on the other predictors. Moreover, these effects
can also be "averaged" over other predictors, to get for instance the
"general trend" of a predictor over different factor levels.
}

\strong{Example:} Let's imagine the following model \code{lm(y ~ condition * x)} where
\code{condition} is a factor with 3 levels A, B and C and \code{x} a continuous
variable (like age for example). One idea is to see how this model performs,
and compare the actual response y to the one predicted by the model (using
\code{\link[=estimate_expectation]{estimate_expectation()}}). Another idea is evaluate the average mean at each of
the condition's levels (using \code{\link[=estimate_means]{estimate_means()}}), which can be useful to
visualize them. Another possibility is to evaluate the difference between
these levels (using \code{\link[=estimate_contrasts]{estimate_contrasts()}}). Finally, one could also estimate
the effect of x averaged over all conditions, or instead within each
condition (using \code{\link[=estimate_slopes]{estimate_slopes()}}).
}
\section{Comparison options}{

\itemize{
\item \code{comparison = "pairwise"}: This method computes all possible unique
differences between pairs of levels of the focal predictor. For example, if
a factor has levels A, B, and C, it would compute A-B, A-C, and B-C.
\item \code{comparison = "reference"}: This compares each level of the focal predictor
to a specified reference level (by default, the first level). For example,
if levels are A, B, C, and A is the reference, it computes B-A and C-A.
\item \code{comparison = "sequential"}: This compares each level to the one
immediately following it in the factor's order. For levels A, B, C, it
would compute B-A and C-B.
\item \code{comparison = "meandev"}: This contrasts each level's estimate against the
grand mean of all estimates for the focal predictor.
\item \code{comparison = "meanotherdev"}: Similar to \code{meandev}, but each level's
estimate is compared against the mean of all \emph{other} levels, excluding
itself.
\item \code{comparison = "poly"}: These are used for ordered categorical variables to
test for linear, quadratic, cubic, etc., trends across the levels. They
assume equal spacing between levels.
\item \code{comparison = "helmert"}: Contrast 2nd level to the first, 3rd to the
average of the first two, and so on. Each level (except the first) is
compared to the mean of the preceding levels. For levels A, B, C, it would
compute B-A and C-(A+B)/2.
\item \code{comparison = "trt_vs_ctrl"}: This compares all levels (excluding the
first, which is typically the control) against the first level. It's often
used when comparing multiple treatment groups to a single control group.
\item To test multiple hypotheses jointly (usually used for factorial designs),
\code{comparison} can also be \code{"joint"}. In this case, use the \code{test} argument
to specify which test should be conducted: \code{"F"} (default) or \code{"Chi2"}.
\item \code{comparison = "inequality"} computes the marginal effect inequality summary
of categorical predictors' overall effects, respectively, the comprehensive
effect of an independent variable across all outcome categories of a
nominal or ordinal dependent variable (total marginal effect, see \emph{Mize and
Han, 2025}). The marginal effect inequality focuses on the heterogeneity of
the effects of a categorical \emph{independent} variable. It helps understand
how the effect of the variable differs across its categories or levels.
When the \emph{dependent} variable is categorical (e.g., logistic, ordinal or
multinomial regression), marginal effect inequality provides a holistic
view of how an independent variable affects a nominal or ordinal
\emph{dependent} variable. It summarizes the overall impact (total marginal
effects) across all possible outcome categories.
\item \code{comparison = "inequality_pairwise"} computes the difference (pairwise
comparisons) between marginal effects inequality measures. Depending on the
sign, this measure indicates which of the predictors has a stronger impact
on the dependent variable in terms of inequalities.
}
}

\section{Effect Size}{


By default, \code{estimate_contrasts()} reports no standardized effect size on
purpose. Should one request one, some things are to keep in mind. As the
authors of \emph{emmeans} write, "There is substantial disagreement among
practitioners on what is the appropriate sigma to use in computing effect
sizes; or, indeed, whether any effect-size measure is appropriate for some
situations. The user is completely responsible for specifying appropriate
parameters (or for failing to do so)."

In particular, effect size method \code{"boot"} does not correct for covariates
in the model, so should probably only be used when there is just one
categorical predictor (with however many levels). Some believe that if there
are multiple predictors or any covariates, it is important to re-compute
sigma adding back in the response variance associated with the variables that
aren't part of the contrast.

\code{effectsize = "emmeans"} uses \link[emmeans:eff_size]{emmeans::eff_size} with
\code{sigma = stats::sigma(model)}, \code{edf = stats::df.residual(model)} and
\code{method = "identity"}. This standardizes using the MSE (sigma). Some believe
this works when the contrasts are the only predictors in the model, but not
when there are covariates. The response variance accounted for by the
covariates should not be removed from the SD used to standardize. Otherwise,
\emph{d} will be overestimated.

\code{effectsize = "marginal"} uses the following formula to compute effect
size: \code{d_adj <- difference * (1- R2)/ sigma}. This standardizes
using the response SD with only the between-groups variance on the focal
factor/contrast removed. This allows for groups to be equated on their
covariates, but creates an appropriate scale for standardizing the response.

\code{effectsize = "boot"} uses bootstrapping (defaults to a low value of
200) through \link[bootES:bootES]{bootES::bootES}. Adjusts for contrasts, but not for covariates.
}

\section{Predictions and contrasts at meaningful values (data grids)}{


To define representative values for focal predictors (specified in \code{by},
\code{contrast}, and \code{trend}), you can use several methods. These values are
internally generated by \code{insight::get_datagrid()}, so consult its
documentation for more details.
\itemize{
\item You can directly specify values as strings or lists for \code{by}, \code{contrast},
and \code{trend}.
\itemize{
\item For numeric focal predictors, use examples like \code{by = "gear = c(4, 8)"},
\code{by = list(gear = c(4, 8))} or \code{by = "gear = 5:10"}
\item For factor or character predictors, use \code{by = "Species = c('setosa', 'virginica')"}
or \code{by = list(Species = c('setosa', 'virginica'))}
}
\item You can use "shortcuts" within square brackets, such as \code{by = "Sepal.Width = [sd]"}
or \code{by = "Sepal.Width = [fivenum]"}
\item For numeric focal predictors, if no representative values are specified,
\code{length} and \code{range} control the number and type of representative values:
\itemize{
\item \code{length} determines how many equally spaced values are generated.
\item \code{range} specifies the type of values, like \code{"range"} or \code{"sd"}.
\item \code{length} and \code{range} apply to all numeric focal predictors.
\item If you have multiple numeric predictors, \code{length} and \code{range} can accept
multiple elements, one for each predictor.
}
\item For integer variables, only values that appear in the data will be included
in the data grid, independent from the \code{length} argument. This behaviour
can be changed by setting \code{protect_integers = FALSE}, which will then treat
integer variables as numerics (and possibly produce fractions).
}

See also \href{https://easystats.github.io/modelbased/articles/visualisation_matrix.html}{this vignette}
for some examples.
}

\section{Predictions on different scales}{


The \code{predict} argument allows to generate predictions on different scales of
the response variable. The \code{"link"} option does not apply to all models, and
usually not to Gaussian models. \code{"link"} will leave the values on scale of
the linear predictors. \code{"response"} (or \code{NULL}) will transform them on scale
of the response variable. Thus for a logistic model, \code{"link"} will give
estimations expressed in log-odds (probabilities on logit scale) and
\code{"response"} in terms of probabilities.

To predict distributional parameters (called "dpar" in other packages), for
instance when using complex formulae in \code{brms} models, the \code{predict} argument
can take the value of the parameter you want to estimate, for instance
\code{"sigma"}, \code{"kappa"}, etc.

\code{"response"} and \code{"inverse_link"} both return predictions on the response
scale, however, \code{"response"} first calculates predictions on the response
scale for each observation and \emph{then} aggregates them by groups or levels
defined in \code{by}. \code{"inverse_link"} first calculates predictions on the link
scale for each observation, then aggregates them by groups or levels defined
in \code{by}, and finally back-transforms the predictions to the response scale.
Both approaches have advantages and disadvantages. \code{"response"} usually
produces less biased predictions, but confidence intervals might be outside
reasonable bounds (i.e., for instance can be negative for count data). The
\code{"inverse_link"} approach is more robust in terms of confidence intervals,
but might produce biased predictions. However, you can try to set
\code{bias_correction = TRUE}, to adjust for this bias.

In particular for mixed models, using \code{"response"} is recommended, because
averaging across random effects groups is then more accurate.
}

\examples{
\dontshow{if (all(insight::check_if_installed(c("lme4", "marginaleffects", "rstanarm"), quietly = TRUE))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontrun{
# Basic usage
model <- lm(Sepal.Width ~ Species, data = iris)
estimate_contrasts(model)

# Dealing with interactions
model <- lm(Sepal.Width ~ Species * Petal.Width, data = iris)

# By default: selects first factor
estimate_contrasts(model)

# Can also run contrasts between points of numeric, stratified by "Species"
estimate_contrasts(model, contrast = "Petal.Width", by = "Species")

# Or both
estimate_contrasts(model, contrast = c("Species", "Petal.Width"), length = 2)

# Or with custom specifications
estimate_contrasts(model, contrast = c("Species", "Petal.Width = c(1, 2)"))

# Or modulate it
estimate_contrasts(model, by = "Petal.Width", length = 4)

# Standardized differences
estimated <- estimate_contrasts(lm(Sepal.Width ~ Species, data = iris))
standardize(estimated)

# custom factor contrasts - contrasts the average effects of two levels
# against the remaining third level
data(puppy_love, package = "modelbased")
cond_tx <- cbind("no treatment" = c(1, 0, 0), "treatment" = c(0, 0.5, 0.5))
model <- lm(happiness ~ puppy_love * dose, data = puppy_love)
estimate_slopes(model, "puppy_love", by = "dose", comparison = cond_tx)

# Other models (mixed, Bayesian, ...)
data <- iris
data$Petal.Length_factor <- ifelse(data$Petal.Length < 4.2, "A", "B")

model <- lme4::lmer(Sepal.Width ~ Species + (1 | Petal.Length_factor), data = data)
estimate_contrasts(model)

data <- mtcars
data$cyl <- as.factor(data$cyl)
data$am <- as.factor(data$am)

model <- rstanarm::stan_glm(mpg ~ cyl * wt, data = data, refresh = 0)
estimate_contrasts(model)
estimate_contrasts(model, by = "wt", length = 4)

model <- rstanarm::stan_glm(
  Sepal.Width ~ Species + Petal.Width + Petal.Length,
  data = iris,
  refresh = 0
)
estimate_contrasts(model, by = "Petal.Length = [sd]", test = "bf")
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Mize, T., & Han, B. (2025). Inequality and Total Effect Summary Measures for
Nominal and Ordinal Variables. Sociological Science, 12, 115–157.
\doi{10.15195/v12.a7}
\item Montiel Olea, J. L., and Plagborg-Møller, M. (2019). Simultaneous
confidence bands: Theory, implementation, and an application to SVARs.
Journal of Applied Econometrics, 34(1), 1–17. \doi{10.1002/jae.2656}
}
}
