/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5LTprivate.h"
#include "H5TBprivate.h"

static bool H5TB_find_field(const char *field, const char *field_list);

static herr_t H5TB_attach_attributes(const char *table_title, hid_t loc_id, const char *dset_name,
                                     hsize_t nfields, hid_t tid);

static hid_t H5TB_create_type(hid_t loc_id, const char *dset_name, size_t type_size,
                              const size_t *field_offset, const size_t *field_sizes, hid_t ftype_id);

herr_t
H5TBmake_table(const char *table_title, hid_t loc_id, const char *dset_name, hsize_t nfields,
               hsize_t nrecords, size_t type_size, const char *field_names[], const size_t *field_offset,
               const hid_t *field_types, hsize_t chunk_size, void *fill_data, int compress, const void *buf)
{
    hid_t          did         = H5I_INVALID_HID;
    hid_t          sid         = H5I_INVALID_HID;
    hid_t          mem_type_id = H5I_INVALID_HID;
    hid_t          plist_id    = H5I_INVALID_HID;
    hid_t          attr_id     = H5I_INVALID_HID;
    hsize_t        dims[1];
    hsize_t        dims_chunk[1];
    hsize_t        maxdims[1] = {H5S_UNLIMITED};
    char           attr_name[255];
    char          *member_name = NULL;
    unsigned char *tmp_buf     = NULL;
    hsize_t        i;
    herr_t         ret_val = -1;

    
    if (table_title == NULL) {
        goto out;
    }
    if (dset_name == NULL) {
        goto out;
    }
    if (field_names == NULL) {
        goto out;
    }

    dims[0]       = nrecords;
    dims_chunk[0] = chunk_size;

    
    if ((mem_type_id = H5Tcreate(H5T_COMPOUND, type_size)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++)
        if (H5Tinsert(mem_type_id, field_names[i], field_offset[i], field_types[i]) < 0)
            goto out;

    
    if ((sid = H5Screate_simple(1, dims, maxdims)) < 0)
        goto out;

    
    if ((plist_id = H5Pcreate(H5P_DATASET_CREATE)) < 0)
        goto out;
    if (H5Pset_chunk(plist_id, 1, dims_chunk) < 0)
        goto out;

    
    if (fill_data)
        if (H5Pset_fill_value(plist_id, mem_type_id, fill_data) < 0)
            goto out;

    
    if (compress)
        if (H5Pset_deflate(plist_id, 6) < 0)
            goto out;

    
    if ((did = H5Dcreate2(loc_id, dset_name, mem_type_id, sid, H5P_DEFAULT, plist_id, H5P_DEFAULT)) < 0)
        goto out;

    
    if (buf)
        if (H5Dwrite(did, mem_type_id, H5S_ALL, H5S_ALL, H5P_DEFAULT, buf) < 0)
            goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;
    sid = H5I_INVALID_HID;

    
    if (H5Dclose(did) < 0)
        goto out;
    did = H5I_INVALID_HID;

    
    if (H5Pclose(plist_id) < 0)
        goto out;
    plist_id = H5I_INVALID_HID;

    

    
    if (H5LTset_attribute_string(loc_id, dset_name, "CLASS", TABLE_CLASS) < 0)
        goto out;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "VERSION", TABLE_VERSION) < 0)
        goto out;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "TITLE", table_title) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(mem_type_id, (unsigned)i)))
            goto out;

        snprintf(attr_name, sizeof(attr_name), "FIELD_%d_NAME", (int)i);

        
        if (H5LTset_attribute_string(loc_id, dset_name, attr_name, member_name) < 0)
            goto out;

        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if (fill_data) {
        tmp_buf = (unsigned char *)fill_data;

        
        if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
            goto out;

        if ((sid = H5Screate(H5S_SCALAR)) < 0)
            goto out;

        for (i = 0; i < nfields; i++) {
            snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)i);

            if ((attr_id = H5Acreate2(did, attr_name, field_types[i], sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                goto out;

            if (H5Awrite(attr_id, field_types[i], tmp_buf + field_offset[i]) < 0)
                goto out;

            if (H5Aclose(attr_id) < 0)
                goto out;
            attr_id = H5I_INVALID_HID;
        } 

        
        if (H5Sclose(sid) < 0)
            goto out;
        sid = H5I_INVALID_HID;

        
        if (H5Dclose(did) < 0)
            goto out;
        did = H5I_INVALID_HID;
    } 

    
    if (H5Tclose(mem_type_id) < 0)
        goto out;
    mem_type_id = H5I_INVALID_HID;

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (attr_id > 0)
        if (H5Aclose(attr_id) < 0)
            ret_val = -1;
    if (plist_id > 0)
        if (H5Pclose(plist_id) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBappend_records(hid_t loc_id, const char *dset_name, hsize_t nrecords, size_t type_size,
                   const size_t *field_offset, const size_t *field_sizes, const void *buf)
{
    hid_t   did         = H5I_INVALID_HID;
    hid_t   tid         = H5I_INVALID_HID;
    hid_t   mem_type_id = H5I_INVALID_HID;
    hsize_t nrecords_orig;
    hsize_t nfields;
    herr_t  ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords_orig) < 0)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    if ((mem_type_id = H5TB_create_type(loc_id, dset_name, type_size, field_offset, field_sizes, tid)) < 0)
        goto out;

    
    if ((H5TB_common_append_records(did, mem_type_id, (size_t)nrecords, nrecords_orig, buf)) < 0)
        goto out;

    ret_val = 0;

out:
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBwrite_records(hid_t loc_id, const char *dset_name, hsize_t start, hsize_t nrecords, size_t type_size,
                  const size_t *field_offset, const size_t *field_sizes, const void *buf)
{
    hid_t   did         = H5I_INVALID_HID;
    hid_t   tid         = H5I_INVALID_HID;
    hid_t   sid         = H5I_INVALID_HID;
    hid_t   m_sid       = H5I_INVALID_HID;
    hid_t   mem_type_id = H5I_INVALID_HID;
    hsize_t count[1];
    hsize_t offset[1];
    hsize_t mem_size[1];
    hsize_t dims[1];
    herr_t  ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    if ((mem_type_id = H5TB_create_type(loc_id, dset_name, type_size, field_offset, field_sizes, tid)) < 0)
        goto out;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    if (H5Sget_simple_extent_dims(sid, dims, NULL) < 0)
        goto out;

    if (start + nrecords > dims[0])
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    if (H5Dwrite(did, mem_type_id, m_sid, sid, H5P_DEFAULT, buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBwrite_fields_name(hid_t loc_id, const char *dset_name, const char *field_names, hsize_t start,
                      hsize_t nrecords, size_t type_size, const size_t *field_offset,
                      const size_t *field_sizes, const void *buf)
{
    hid_t    did            = H5I_INVALID_HID;
    hid_t    tid            = H5I_INVALID_HID;
    hid_t    write_type_id  = H5I_INVALID_HID;
    hid_t    member_type_id = H5I_INVALID_HID;
    hid_t    nmtype_id      = H5I_INVALID_HID;
    hid_t    m_sid          = H5I_INVALID_HID;
    hid_t    file_space_id  = H5I_INVALID_HID;
    hid_t    preserve_id    = H5I_INVALID_HID;
    hssize_t nfields;
    hssize_t i, j;
    hsize_t  count[1];
    hsize_t  offset[1];
    char    *member_name = NULL;
    size_t   size_native;
    herr_t   ret_val = -1;

    
    if (dset_name == NULL)
        goto out;
    if (field_names == NULL)
        goto out;

    
    if ((preserve_id = H5Pcreate(H5P_DATASET_XFER)) < 0)
        goto out;
    if (H5Pset_preserve(preserve_id, 1) < 0)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if ((nfields = H5Tget_nmembers(tid)) < 0)
        goto out;

    
    if ((write_type_id = H5Tcreate(H5T_COMPOUND, type_size)) < 0)
        goto out;

    j = 0;

    
    for (i = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(tid, (unsigned)i)))
            goto out;

        if (H5TB_find_field(member_name, field_names)) {
            
            if ((member_type_id = H5Tget_member_type(tid, (unsigned)i)) < 0)
                goto out;

            
            if ((nmtype_id = H5Tget_native_type(member_type_id, H5T_DIR_DEFAULT)) < 0)
                goto out;

            if (0 == (size_native = H5Tget_size(nmtype_id)))
                goto out;

            
            if (field_sizes[j] != size_native)
                if (H5Tset_size(nmtype_id, field_sizes[j]) < 0)
                    goto out;

            
            if (field_offset) {
                if (H5Tinsert(write_type_id, member_name, field_offset[j], nmtype_id) < 0)
                    goto out;
            } 
            
            else {
                if (H5Tinsert(write_type_id, member_name, (size_t)0, nmtype_id) < 0)
                    goto out;
            } 

            j++;

            
            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;
            if (H5Tclose(nmtype_id) < 0)
                goto out;
            nmtype_id = H5I_INVALID_HID;
        } 

        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if ((file_space_id = H5Dget_space(did)) < 0)
        goto out;
    if ((m_sid = H5Screate_simple(1, &nrecords, NULL)) < 0)
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(file_space_id, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    if (H5Dwrite(did, write_type_id, m_sid, file_space_id, preserve_id, buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (preserve_id > 0)
        if (H5Pclose(preserve_id) < 0)
            ret_val = -1;
    if (write_type_id > 0)
        if (H5Tclose(write_type_id) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (file_space_id > 0)
        if (H5Sclose(file_space_id) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBwrite_fields_index(hid_t loc_id, const char *dset_name, hsize_t nfields, const int *field_index,
                       hsize_t start, hsize_t nrecords, size_t type_size, const size_t *field_offset,
                       const size_t *field_sizes, const void *buf)
{
    hid_t   did            = H5I_INVALID_HID;
    hid_t   tid            = H5I_INVALID_HID;
    hid_t   write_type_id  = H5I_INVALID_HID;
    hid_t   member_type_id = H5I_INVALID_HID;
    hid_t   nmtype_id      = H5I_INVALID_HID;
    hid_t   m_sid          = H5I_INVALID_HID;
    hid_t   file_space_id  = H5I_INVALID_HID;
    hid_t   preserve_id    = H5I_INVALID_HID;
    hsize_t count[1];
    hsize_t offset[1];
    hsize_t i;
    size_t  size_native;
    char   *member_name = NULL;
    herr_t  ret_val     = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if ((preserve_id = H5Pcreate(H5P_DATASET_XFER)) < 0)
        goto out;
    if (H5Pset_preserve(preserve_id, 1) < 0)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if ((write_type_id = H5Tcreate(H5T_COMPOUND, type_size)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        unsigned j;

        
        if (field_index[i] < 0)
            goto out;

        j = (unsigned)field_index[i];

        
        if (NULL == (member_name = H5Tget_member_name(tid, j)))
            goto out;

        
        if ((member_type_id = H5Tget_member_type(tid, j)) < 0)
            goto out;

        
        if ((nmtype_id = H5Tget_native_type(member_type_id, H5T_DIR_DEFAULT)) < 0)
            goto out;

        if (0 == (size_native = H5Tget_size(nmtype_id)))
            goto out;

        if (field_sizes[i] != size_native)
            if (H5Tset_size(nmtype_id, field_sizes[i]) < 0)
                goto out;

        
        if (field_offset) {
            if (H5Tinsert(write_type_id, member_name, field_offset[i], nmtype_id) < 0)
                goto out;
        } 
        
        else {
            if (H5Tinsert(write_type_id, member_name, (size_t)0, nmtype_id) < 0)
                goto out;
        } 

        
        if (H5Tclose(member_type_id) < 0)
            goto out;
        member_type_id = H5I_INVALID_HID;
        if (H5Tclose(nmtype_id) < 0)
            goto out;
        nmtype_id = H5I_INVALID_HID;

        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if ((file_space_id = H5Dget_space(did)) < 0)
        goto out;
    if ((m_sid = H5Screate_simple(1, &nrecords, NULL)) < 0)
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(file_space_id, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    if (H5Dwrite(did, write_type_id, m_sid, file_space_id, preserve_id, buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (preserve_id > 0)
        if (H5Pclose(preserve_id) < 0)
            ret_val = -1;
    if (write_type_id > 0)
        if (H5Tclose(write_type_id) < 0)
            ret_val = -1;
    if (member_type_id > 0)
        if (H5Tclose(member_type_id) < 0)
            ret_val = -1;
    if (nmtype_id > 0)
        if (H5Tclose(nmtype_id) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (file_space_id > 0)
        if (H5Sclose(file_space_id) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBread_table(hid_t loc_id, const char *dset_name, size_t type_size, const size_t *field_offset,
               const size_t *field_sizes, void *dst_buf)
{
    hid_t   did         = H5I_INVALID_HID;
    hid_t   ftype_id    = H5I_INVALID_HID;
    hid_t   mem_type_id = H5I_INVALID_HID;
    hid_t   sid         = H5I_INVALID_HID;
    hsize_t dims[1];
    herr_t  ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    if (H5Sget_simple_extent_dims(sid, dims, NULL) < 0)
        goto out;

    
    if ((ftype_id = H5Dget_type(did)) < 0)
        goto out;

    if ((mem_type_id = H5TB_create_type(loc_id, dset_name, type_size, field_offset, field_sizes, ftype_id)) <
        0)
        goto out;

    
    if (H5Dread(did, mem_type_id, H5S_ALL, H5S_ALL, H5P_DEFAULT, dst_buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (ftype_id > 0)
        if (H5Tclose(ftype_id) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBread_records(hid_t loc_id, const char *dset_name, hsize_t start, hsize_t nrecords, size_t type_size,
                 const size_t *field_offset, const size_t *field_sizes, void *buf)
{
    hid_t   did         = H5I_INVALID_HID;
    hid_t   ftype_id    = H5I_INVALID_HID;
    hid_t   mem_type_id = H5I_INVALID_HID;
    hsize_t nrecords_orig;
    hsize_t nfields;
    herr_t  ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords_orig) < 0)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((ftype_id = H5Dget_type(did)) < 0)
        goto out;

    if ((mem_type_id = H5TB_create_type(loc_id, dset_name, type_size, field_offset, field_sizes, ftype_id)) <
        0)
        goto out;

    
    if ((H5TB_common_read_records(did, mem_type_id, start, (size_t)nrecords, nrecords_orig, buf)) < 0)
        goto out;

    ret_val = 0;

out:
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (ftype_id > 0)
        if (H5Tclose(ftype_id) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBread_fields_name(hid_t loc_id, const char *dset_name, const char *field_names, hsize_t start,
                     hsize_t nrecords, size_t type_size, const size_t *field_offset,
                     const size_t *field_sizes, void *buf)
{
    hid_t    did         = H5I_INVALID_HID;
    hid_t    ftype_id    = H5I_INVALID_HID;
    hid_t    mem_type_id = H5I_INVALID_HID;
    hid_t    mtype_id    = H5I_INVALID_HID;
    hid_t    nmtype_id   = H5I_INVALID_HID;
    hid_t    sid         = H5I_INVALID_HID;
    hid_t    m_sid       = H5I_INVALID_HID;
    hssize_t nfields;
    hsize_t  count[1];
    hsize_t  offset[1];
    hsize_t  mem_size[1];
    size_t   size_native;
    char    *member_name = NULL;
    hssize_t i, j;
    herr_t   ret_val = -1;

    
    if (dset_name == NULL)
        goto out;
    if (field_names == NULL)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((ftype_id = H5Dget_type(did)) < 0)
        goto out;

    
    if ((nfields = H5Tget_nmembers(ftype_id)) < 0)
        goto out;

    
    if ((mem_type_id = H5Tcreate(H5T_COMPOUND, type_size)) < 0)
        goto out;

    
    for (i = 0, j = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(ftype_id, (unsigned)i)))
            goto out;

        if (H5TB_find_field(member_name, field_names)) {
            
            if ((mtype_id = H5Tget_member_type(ftype_id, (unsigned)i)) < 0)
                goto out;

            
            if ((nmtype_id = H5Tget_native_type(mtype_id, H5T_DIR_DEFAULT)) < 0)
                goto out;

            if (0 == (size_native = H5Tget_size(nmtype_id)))
                goto out;

            if (field_sizes[j] != size_native)
                if (H5Tset_size(nmtype_id, field_sizes[j]) < 0)
                    goto out;

            
            if (field_offset) {
                if (H5Tinsert(mem_type_id, member_name, field_offset[j], nmtype_id) < 0)
                    goto out;
            } 
            else {
                if (H5Tinsert(mem_type_id, member_name, (size_t)0, nmtype_id) < 0)
                    goto out;
            } 

            
            if (H5Tclose(mtype_id) < 0)
                goto out;
            mtype_id = H5I_INVALID_HID;
            if (H5Tclose(nmtype_id) < 0)
                goto out;
            nmtype_id = H5I_INVALID_HID;
            j++;
        } 

        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if (j == 0)
        goto out;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    
    if (H5Dread(did, mem_type_id, m_sid, sid, H5P_DEFAULT, buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (mtype_id > 0)
        if (H5Tclose(mtype_id) < 0)
            ret_val = -1;
    if (nmtype_id > 0)
        if (H5Tclose(nmtype_id) < 0)
            ret_val = -1;
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (ftype_id > 0)
        if (H5Tclose(ftype_id) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBread_fields_index(hid_t loc_id, const char *dset_name, hsize_t nfields, const int *field_index,
                      hsize_t start, hsize_t nrecords, size_t type_size, const size_t *field_offset,
                      const size_t *field_sizes, void *buf)
{
    hid_t   did            = H5I_INVALID_HID;
    hid_t   tid            = H5I_INVALID_HID;
    hid_t   read_type_id   = H5I_INVALID_HID;
    hid_t   member_type_id = H5I_INVALID_HID;
    hid_t   nmtype_id      = H5I_INVALID_HID;
    hid_t   sid            = H5I_INVALID_HID;
    hid_t   m_sid          = H5I_INVALID_HID;
    hsize_t count[1];
    hsize_t offset[1];
    hsize_t mem_size[1];
    hsize_t i;
    size_t  size_native;
    char   *member_name = NULL;
    herr_t  ret_val     = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if ((read_type_id = H5Tcreate(H5T_COMPOUND, type_size)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        unsigned j;

        
        if (field_index[i] < 0)
            goto out;

        j = (unsigned)field_index[i];

        
        if (NULL == (member_name = H5Tget_member_name(tid, (unsigned)j)))
            goto out;

        
        if ((member_type_id = H5Tget_member_type(tid, (unsigned)j)) < 0)
            goto out;

        
        if ((nmtype_id = H5Tget_native_type(member_type_id, H5T_DIR_DEFAULT)) < 0)
            goto out;

        if (0 == (size_native = H5Tget_size(nmtype_id)))
            goto out;

        if (field_sizes[i] != size_native)
            if (H5Tset_size(nmtype_id, field_sizes[i]) < 0)
                goto out;

        
        if (field_offset) {
            if (H5Tinsert(read_type_id, member_name, field_offset[i], nmtype_id) < 0)
                goto out;
        } 
        else {
            if (H5Tinsert(read_type_id, member_name, (size_t)0, nmtype_id) < 0)
                goto out;
        } 

        
        if (H5Tclose(member_type_id) < 0)
            goto out;
        member_type_id = H5I_INVALID_HID;
        if (H5Tclose(nmtype_id) < 0)
            goto out;
        nmtype_id = H5I_INVALID_HID;

        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    
    if (H5Dread(did, read_type_id, m_sid, sid, H5P_DEFAULT, buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (read_type_id > 0)
        if (H5Tclose(read_type_id) < 0)
            ret_val = -1;
    if (member_type_id > 0)
        if (H5Tclose(member_type_id) < 0)
            ret_val = -1;
    if (nmtype_id > 0)
        if (H5Tclose(nmtype_id) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBdelete_record(hid_t loc_id, const char *dset_name, hsize_t start, hsize_t nrecords)
{
    hid_t          did         = H5I_INVALID_HID;
    hid_t          tid         = H5I_INVALID_HID;
    hid_t          sid         = H5I_INVALID_HID;
    hid_t          m_sid       = H5I_INVALID_HID;
    hid_t          mem_type_id = H5I_INVALID_HID;
    hsize_t        nfields;
    hsize_t        ntotal_records;
    hsize_t        read_start;
    hsize_t        read_nrecords;
    hsize_t        count[1];
    hsize_t        offset[1];
    hsize_t        mem_size[1];
    hsize_t        dims[1];
    size_t         src_size;
    size_t        *src_offset = NULL;
    size_t        *src_sizes  = NULL;
    unsigned char *tmp_buf    = NULL;
    herr_t         ret_val    = -1;

    
    if (dset_name == NULL)
        goto out;

    

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &ntotal_records) < 0)
        goto out;

    if (NULL == (src_offset = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;
    if (NULL == (src_sizes = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;

    
    if (H5TBget_field_info(loc_id, dset_name, NULL, src_sizes, src_offset, &src_size) < 0)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    

    read_start    = start + nrecords;
    read_nrecords = ntotal_records - read_start;

    if (read_nrecords) {
        if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)read_nrecords, src_size)))
            goto out;

        
        if (H5TBread_records(loc_id, dset_name, read_start, read_nrecords, src_size, src_offset, src_sizes,
                             tmp_buf) < 0)
            goto out;

        

        
        if ((tid = H5Dget_type(did)) < 0)
            goto out;

        
        if ((sid = H5Dget_space(did)) < 0)
            goto out;

        
        if ((mem_type_id = H5TB_create_type(loc_id, dset_name, src_size, src_offset, src_sizes, tid)) < 0)
            goto out;

        
        offset[0] = start;
        count[0]  = read_nrecords;
        if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
            goto out;

        
        mem_size[0] = count[0];
        if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
            goto out;

        if (H5Dwrite(did, mem_type_id, m_sid, sid, H5P_DEFAULT, tmp_buf) < 0)
            goto out;

        
        if (H5Sclose(m_sid) < 0)
            goto out;
        m_sid = H5I_INVALID_HID;
        if (H5Tclose(mem_type_id) < 0)
            goto out;
        mem_type_id = H5I_INVALID_HID;
        if (H5Sclose(sid) < 0)
            goto out;
        sid = H5I_INVALID_HID;
        if (H5Tclose(tid) < 0)
            goto out;
        tid = H5I_INVALID_HID;
    } 

    
    dims[0] = ntotal_records - nrecords;
    if (H5Dset_extent(did, dims) < 0)
        goto out;

    ret_val = 0;

out:
    if (tmp_buf)
        free(tmp_buf);
    if (src_offset)
        free(src_offset);
    if (src_sizes)
        free(src_sizes);
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBinsert_record(hid_t loc_id, const char *dset_name, hsize_t start, hsize_t nrecords, size_t type_size,
                  const size_t *field_offset, const size_t *field_sizes, void *buf)
{
    hid_t          did         = H5I_INVALID_HID;
    hid_t          tid         = H5I_INVALID_HID;
    hid_t          mem_type_id = H5I_INVALID_HID;
    hid_t          sid         = H5I_INVALID_HID;
    hid_t          m_sid       = H5I_INVALID_HID;
    hsize_t        nfields;
    hsize_t        ntotal_records;
    hsize_t        read_nrecords;
    hsize_t        count[1];
    hsize_t        offset[1];
    hsize_t        dims[1];
    hsize_t        mem_dims[1];
    unsigned char *tmp_buf = NULL;
    herr_t         ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &ntotal_records) < 0)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if ((mem_type_id = H5TB_create_type(loc_id, dset_name, type_size, field_offset, field_sizes, tid)) < 0)
        goto out;

    read_nrecords = ntotal_records - start;
    if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)read_nrecords, type_size)))
        goto out;

    
    if (H5TBread_records(loc_id, dset_name, start, read_nrecords, type_size, field_offset, field_sizes,
                         tmp_buf) < 0)
        goto out;

    
    dims[0] = ntotal_records + nrecords;

    if (H5Dset_extent(did, dims) < 0)
        goto out;

    

    
    mem_dims[0] = nrecords;
    if ((m_sid = H5Screate_simple(1, mem_dims, NULL)) < 0)
        goto out;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    if (H5Dwrite(did, mem_type_id, m_sid, sid, H5P_DEFAULT, buf) < 0)
        goto out;

    
    if (H5Sclose(m_sid) < 0)
        goto out;
    m_sid = H5I_INVALID_HID;
    if (H5Sclose(sid) < 0)
        goto out;
    sid = H5I_INVALID_HID;

    

    
    mem_dims[0] = read_nrecords;
    if ((m_sid = H5Screate_simple(1, mem_dims, NULL)) < 0)
        goto out;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    offset[0] = start + nrecords;
    count[0]  = read_nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    if (H5Dwrite(did, mem_type_id, m_sid, sid, H5P_DEFAULT, tmp_buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (tmp_buf)
        free(tmp_buf);
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (mem_type_id > 0)
        if (H5Tclose(mem_type_id) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBadd_records_from(hid_t loc_id, const char *dset_name1, hsize_t start1, hsize_t nrecords,
                     const char *dset_name2, hsize_t start2)
{
    hid_t          did   = H5I_INVALID_HID;
    hid_t          tid   = H5I_INVALID_HID;
    hid_t          sid   = H5I_INVALID_HID;
    hid_t          m_sid = H5I_INVALID_HID;
    hsize_t        count[1];
    hsize_t        offset[1];
    hsize_t        mem_size[1];
    hsize_t        nfields;
    hsize_t        ntotal_records;
    size_t         type_size1;
    size_t         src_size;
    size_t        *src_offset = NULL;
    size_t        *src_sizes  = NULL;
    unsigned char *tmp_buf    = NULL;
    herr_t         ret_val    = -1;

    
    if (dset_name1 == NULL)
        goto out;
    if (dset_name2 == NULL)
        goto out;

    

    
    if (H5TBget_table_info(loc_id, dset_name1, &nfields, &ntotal_records) < 0)
        goto out;

    if (NULL == (src_offset = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;
    if (NULL == (src_sizes = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;

    
    if (H5TBget_field_info(loc_id, dset_name1, NULL, src_sizes, src_offset, &src_size) < 0)
        goto out;

    

    
    if ((did = H5Dopen2(loc_id, dset_name1, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    if (0 == (type_size1 = H5Tget_size(tid)))
        goto out;

    if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)nrecords, type_size1)))
        goto out;

    
    offset[0] = start1;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    if (H5Dread(did, tid, m_sid, sid, H5P_DEFAULT, tmp_buf) < 0)
        goto out;

    
    if (H5TBinsert_record(loc_id, dset_name2, start2, nrecords, src_size, src_offset, src_sizes, tmp_buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (tmp_buf)
        free(tmp_buf);
    if (src_offset)
        free(src_offset);
    if (src_sizes)
        free(src_sizes);
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBcombine_tables(hid_t loc_id1, const char *dset_name1, hid_t loc_id2, const char *dset_name2,
                   const char *dset_name3)
{
    
    hid_t did_1 = H5I_INVALID_HID;
    hid_t tid_1 = H5I_INVALID_HID;
    hid_t sid_1 = H5I_INVALID_HID;
    hid_t pid_1 = H5I_INVALID_HID;
    
    hid_t did_2 = H5I_INVALID_HID;
    hid_t tid_2 = H5I_INVALID_HID;
    hid_t sid_2 = H5I_INVALID_HID;
    hid_t pid_2 = H5I_INVALID_HID;
    
    hid_t          did_3          = H5I_INVALID_HID;
    hid_t          tid_3          = H5I_INVALID_HID;
    hid_t          sid_3          = H5I_INVALID_HID;
    hid_t          pid_3          = H5I_INVALID_HID;
    hid_t          sid            = H5I_INVALID_HID;
    hid_t          m_sid          = H5I_INVALID_HID;
    hid_t          member_type_id = H5I_INVALID_HID;
    hid_t          attr_id        = H5I_INVALID_HID;
    hsize_t        count[1];
    hsize_t        offset[1];
    hsize_t        mem_size[1];
    hsize_t        nfields;
    hsize_t        nrecords;
    hsize_t        dims[1];
    hsize_t        maxdims[1] = {H5S_UNLIMITED};
    hsize_t        i;
    size_t         type_size;
    size_t         member_offset;
    size_t         src_size;
    size_t        *src_offset = NULL;
    size_t        *src_sizes  = NULL;
    char           attr_name[255];
    unsigned char *tmp_buf      = NULL;
    unsigned char *tmp_fill_buf = NULL;
    htri_t         has_fill;
    herr_t         ret_val = -1;

    
    if (dset_name1 == NULL)
        goto out;
    if (dset_name2 == NULL)
        goto out;
    if (dset_name3 == NULL)
        goto out;

    

    
    if (H5TBget_table_info(loc_id1, dset_name1, &nfields, &nrecords) < 0)
        goto out;

    if (NULL == (src_offset = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;
    if (NULL == (src_sizes = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;

    
    if (H5TBget_field_info(loc_id1, dset_name1, NULL, src_sizes, src_offset, &src_size) < 0)
        goto out;

    

    
    if ((did_1 = H5Dopen2(loc_id1, dset_name1, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid_1 = H5Dget_type(did_1)) < 0)
        goto out;

    
    if ((sid_1 = H5Dget_space(did_1)) < 0)
        goto out;

    
    if ((pid_1 = H5Dget_create_plist(did_1)) < 0)
        goto out;

    
    if (H5TBget_table_info(loc_id1, dset_name1, &nfields, &nrecords) < 0)
        goto out;

    

    
    if ((pid_3 = H5Pcopy(pid_1)) < 0)
        goto out;

    
    if ((tid_3 = H5Tcopy(tid_1)) < 0)
        goto out;

    
    dims[0] = 0;

    
    if ((sid_3 = H5Screate_simple(1, dims, maxdims)) < 0)
        goto out;

    
    if ((did_3 = H5Dcreate2(loc_id1, dset_name3, tid_3, sid_3, H5P_DEFAULT, pid_3, H5P_DEFAULT)) < 0)
        goto out;

    
    if (H5TB_attach_attributes("Merge table", loc_id1, dset_name3, nfields, tid_3) < 0)
        goto out;

    
    if (0 == (type_size = H5Tget_size(tid_3)))
        goto out;

    
    if (NULL == (tmp_fill_buf = (unsigned char *)malloc(type_size)))
        goto out;

    
    if ((has_fill = H5TBAget_fill(loc_id1, dset_name1, did_1, tmp_fill_buf)) < 0)
        goto out;

    
    if (has_fill) {
        if ((sid = H5Screate(H5S_SCALAR)) < 0)
            goto out;

        for (i = 0; i < nfields; i++) {
            
            if ((member_type_id = H5Tget_member_type(tid_3, (unsigned)i)) < 0)
                goto out;

            
            member_offset = src_offset[i];

            snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)i);

            if ((attr_id = H5Acreate2(did_3, attr_name, member_type_id, sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                goto out;

            if (H5Awrite(attr_id, member_type_id, tmp_fill_buf + member_offset) < 0)
                goto out;

            if (H5Aclose(attr_id) < 0)
                goto out;
            attr_id = H5I_INVALID_HID;

            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;
        } 

        
        if (H5Sclose(sid) < 0)
            goto out;
        sid = H5I_INVALID_HID;
    } 

    
    if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)nrecords, type_size)))
        goto out;

    
    offset[0] = 0;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid_1, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    if (H5Dread(did_1, tid_1, m_sid, sid_1, H5P_DEFAULT, tmp_buf) < 0)
        goto out;

    

    
    if (H5TBappend_records(loc_id1, dset_name3, nrecords, src_size, src_offset, src_sizes, tmp_buf) < 0)
        goto out;

    
    if (H5Sclose(m_sid) < 0)
        goto out;
    m_sid = H5I_INVALID_HID;
    free(tmp_buf);
    tmp_buf = NULL;

    

    
    if ((did_2 = H5Dopen2(loc_id2, dset_name2, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid_2 = H5Dget_type(did_2)) < 0)
        goto out;

    
    if ((sid_2 = H5Dget_space(did_2)) < 0)
        goto out;

    
    if ((pid_2 = H5Dget_create_plist(did_2)) < 0)
        goto out;

    
    if (H5TBget_table_info(loc_id2, dset_name2, &nfields, &nrecords) < 0)
        goto out;

    

    if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)nrecords, type_size)))
        goto out;

    
    offset[0] = 0;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid_2, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    if (H5Dread(did_2, tid_2, m_sid, sid_2, H5P_DEFAULT, tmp_buf) < 0)
        goto out;

    

    
    if (H5TBappend_records(loc_id1, dset_name3, nrecords, src_size, src_offset, src_sizes, tmp_buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (tmp_buf)
        free(tmp_buf);
    if (tmp_fill_buf)
        free(tmp_fill_buf);
    if (src_offset)
        free(src_offset);
    if (src_sizes)
        free(src_sizes);
    if (member_type_id > 0)
        if (H5Tclose(member_type_id) < 0)
            ret_val = -1;
    if (attr_id > 0)
        if (H5Aclose(attr_id) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid_1 > 0)
        if (H5Sclose(sid_1) < 0)
            ret_val = -1;
    if (tid_1 > 0)
        if (H5Tclose(tid_1) < 0)
            ret_val = -1;
    if (pid_1 > 0)
        if (H5Pclose(pid_1) < 0)
            ret_val = -1;
    if (did_1 > 0)
        if (H5Dclose(did_1) < 0)
            ret_val = -1;
    if (sid_2 > 0)
        if (H5Sclose(sid_2) < 0)
            ret_val = -1;
    if (tid_2 > 0)
        if (H5Tclose(tid_2) < 0)
            ret_val = -1;
    if (pid_2 > 0)
        if (H5Pclose(pid_2) < 0)
            ret_val = -1;
    if (did_2 > 0)
        if (H5Dclose(did_2) < 0)
            ret_val = -1;
    if (sid_3 > 0)
        if (H5Sclose(sid_3) < 0)
            ret_val = -1;
    if (tid_3 > 0)
        if (H5Tclose(tid_3) < 0)
            ret_val = -1;
    if (pid_3 > 0)
        if (H5Pclose(pid_3) < 0)
            ret_val = -1;
    if (did_3 > 0)
        if (H5Dclose(did_3) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBinsert_field(hid_t loc_id, const char *dset_name, const char *field_name, hid_t field_type,
                 hsize_t position, const void *fill_data, const void *buf)
{
    
    hid_t did_1  = H5I_INVALID_HID;
    hid_t tid_1  = H5I_INVALID_HID;
    hid_t sid_1  = H5I_INVALID_HID;
    hid_t pid_1  = H5I_INVALID_HID;
    hid_t msid_1 = H5I_INVALID_HID;
    
    hid_t did_2  = H5I_INVALID_HID;
    hid_t tid_2  = H5I_INVALID_HID;
    hid_t sid_2  = H5I_INVALID_HID;
    hid_t pid_2  = H5I_INVALID_HID;
    hid_t msid_2 = H5I_INVALID_HID;
    
    hid_t          did_3          = H5I_INVALID_HID;
    hid_t          tid_3          = H5I_INVALID_HID;
    hid_t          sid_3          = H5I_INVALID_HID;
    hid_t          member_type_id = H5I_INVALID_HID;
    hid_t          write_type_id  = H5I_INVALID_HID;
    hid_t          preserve_id    = H5I_INVALID_HID;
    hid_t          attr_id        = H5I_INVALID_HID;
    size_t         member_size;
    size_t         new_member_size = 0;
    size_t         total_size;
    size_t         curr_offset;
    hsize_t        nfields;
    hsize_t        nrecords;
    hsize_t        dims_chunk[1];
    hsize_t        dims[1];
    hsize_t        maxdims[1] = {H5S_UNLIMITED};
    hsize_t        count[1];
    hsize_t        offset[1];
    hsize_t        mem_size[1];
    hsize_t        i;
    char           table_title[255];
    char           attr_name[255];
    char          *member_name  = NULL;
    unsigned char *tmp_buf      = NULL;
    unsigned char *tmp_fill_buf = NULL;
    size_t        *src_offset   = NULL;
    bool           inserted;
    herr_t         ret_val = -1;

    
    if (dset_name == NULL)
        goto out;
    if (field_name == NULL)
        goto out;

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords) < 0)
        goto out;

    

    
    if ((did_1 = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((pid_1 = H5Dget_create_plist(did_1)) < 0)
        goto out;

    
    if ((tid_1 = H5Dget_type(did_1)) < 0)
        goto out;

    
    if (0 == (total_size = H5Tget_size(tid_1)))
        goto out;

    
    if ((sid_1 = H5Dget_space(did_1)) < 0)
        goto out;

    
    if (H5Sget_simple_extent_dims(sid_1, dims, NULL) < 0)
        goto out;

    

    
    if ((H5TBAget_title(did_1, table_title)) < 0)
        goto out;

    
    if (NULL == (tmp_fill_buf = (unsigned char *)calloc(1, total_size)))
        goto out;

    
    if ((H5TBAget_fill(loc_id, dset_name, did_1, tmp_fill_buf)) < 0)
        goto out;

    if (NULL == (src_offset = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;

    
    if (H5TBget_field_info(loc_id, dset_name, NULL, NULL, src_offset, NULL) < 0)
        goto out;

    

    
    if (0 == (member_size = H5Tget_size(field_type)))
        goto out;

    
    if ((tid_2 = H5Tcreate(H5T_COMPOUND, (size_t)(total_size + member_size))) < 0)
        goto out;

    curr_offset = 0;
    inserted    = false;

    
    for (i = 0; i < nfields + 1; i++) {
        hsize_t idx;

        idx = i;
        if (inserted)
            idx = i - 1;

        if (i == position) {
            
            if (0 == (new_member_size = H5Tget_size(field_type)))
                goto out;

            
            if (H5Tinsert(tid_2, field_name, curr_offset, field_type) < 0)
                goto out;

            curr_offset += new_member_size;

            inserted = true;
        } 
        else {
            
            if (NULL == (member_name = H5Tget_member_name(tid_1, (unsigned)idx)))
                goto out;

            
            if ((member_type_id = H5Tget_member_type(tid_1, (unsigned)idx)) < 0)
                goto out;

            
            if (0 == (member_size = H5Tget_size(member_type_id)))
                goto out;

            
            if (H5Tinsert(tid_2, member_name, curr_offset, member_type_id) < 0)
                goto out;

            curr_offset += member_size;

            H5free_memory(member_name);
            member_name = NULL;

            
            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;
        } 
    }     

    

    
    if (H5Pget_chunk(pid_1, 1, dims_chunk) < 0)
        goto out;

    
    if ((sid_2 = H5Screate_simple(1, dims, maxdims)) < 0)
        goto out;

    
    if ((pid_2 = H5Pcreate(H5P_DATASET_CREATE)) < 0)
        goto out;
    if (H5Pset_chunk(pid_2, 1, dims_chunk) < 0)
        goto out;

    
    if ((did_2 = H5Dcreate2(loc_id, "new", tid_2, sid_2, H5P_DEFAULT, pid_2, H5P_DEFAULT)) < 0)
        goto out;

    

    if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)nrecords, (size_t)total_size)))
        goto out;

    
    offset[0] = 0;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid_1, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((msid_1 = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;

    if (H5Dread(did_1, tid_1, msid_1, H5S_ALL, H5P_DEFAULT, tmp_buf) < 0)
        goto out;

    

    
    if (H5Dwrite(did_2, tid_1, msid_1, H5S_ALL, H5P_DEFAULT, tmp_buf) < 0)
        goto out;

    

    
    if ((write_type_id = H5Tcreate(H5T_COMPOUND, (size_t)new_member_size)) < 0)
        goto out;

    
    if (H5Tinsert(write_type_id, field_name, (size_t)0, field_type) < 0)
        goto out;

    
    if ((preserve_id = H5Pcreate(H5P_DATASET_XFER)) < 0)
        goto out;
    if (H5Pset_preserve(preserve_id, 1) < 0)
        goto out;

    
    if (buf) {
        
        if ((msid_2 = H5Screate_simple(1, mem_size, NULL)) < 0)
            goto out;

        
        if (H5Dwrite(did_2, write_type_id, msid_2, sid_2, preserve_id, buf) < 0)
            goto out;
    } 

    
    if (H5Ldelete(loc_id, dset_name, H5P_DEFAULT) < 0)
        goto out;

    
    if (H5Lmove(loc_id, "new", H5L_SAME_LOC, dset_name, H5P_DEFAULT, H5P_DEFAULT) < 0)
        goto out;

    

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords) < 0)
        goto out;

    
    if ((did_3 = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid_3 = H5Dget_type(did_3)) < 0)
        goto out;

    
    if (H5TB_attach_attributes(table_title, loc_id, dset_name, (hsize_t)nfields, tid_3) < 0)
        goto out;

    
    if ((sid_3 = H5Screate(H5S_SCALAR)) < 0)
        goto out;

    for (i = 0; i < (nfields - 1); i++) {
        size_t member_offset;

        
        if ((member_type_id = H5Tget_member_type(tid_3, (unsigned)i)) < 0)
            goto out;

        
        member_offset = src_offset[i];
        snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)i);

        if ((attr_id = H5Acreate2(did_3, attr_name, member_type_id, sid_3, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        if (H5Awrite(attr_id, member_type_id, tmp_fill_buf + member_offset) < 0)
            goto out;

        if (H5Aclose(attr_id) < 0)
            goto out;
        attr_id = H5I_INVALID_HID;

        
        if (H5Tclose(member_type_id) < 0)
            goto out;
        member_type_id = H5I_INVALID_HID;
    } 

    
    if (fill_data) {
        snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)(nfields - 1));

        
        if ((member_type_id = H5Tget_member_type(tid_3, (unsigned)nfields - 1)) < 0)
            goto out;

        if ((attr_id = H5Acreate2(did_3, attr_name, member_type_id, sid_3, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        if (H5Awrite(attr_id, member_type_id, fill_data) < 0)
            goto out;

        if (H5Aclose(attr_id) < 0)
            goto out;
        attr_id = H5I_INVALID_HID;

        if (H5Tclose(member_type_id) < 0)
            goto out;
        member_type_id = H5I_INVALID_HID;
    } 

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (src_offset)
        free(src_offset);
    if (tmp_buf)
        free(tmp_buf);
    if (tmp_fill_buf)
        free(tmp_fill_buf);
    if (preserve_id > 0)
        if (H5Pclose(preserve_id) < 0)
            ret_val = -1;
    if (msid_1 > 0)
        if (H5Sclose(msid_1) < 0)
            ret_val = -1;
    if (tid_1 > 0)
        if (H5Tclose(tid_1) < 0)
            ret_val = -1;
    if (pid_1 > 0)
        if (H5Pclose(pid_1) < 0)
            ret_val = -1;
    if (sid_1 > 0)
        if (H5Sclose(sid_1) < 0)
            ret_val = -1;
    if (did_1 > 0)
        if (H5Dclose(did_1) < 0)
            ret_val = -1;
    if (msid_2 > 0)
        if (H5Sclose(msid_2) < 0)
            ret_val = -1;
    if (sid_2 > 0)
        if (H5Sclose(sid_2) < 0)
            ret_val = -1;
    if (tid_2 > 0)
        if (H5Tclose(tid_2) < 0)
            ret_val = -1;
    if (pid_2 > 0)
        if (H5Pclose(pid_2) < 0)
            ret_val = -1;
    if (did_2 > 0)
        if (H5Dclose(did_2) < 0)
            ret_val = -1;
    if (sid_3 > 0)
        if (H5Sclose(sid_3) < 0)
            ret_val = -1;
    if (tid_3 > 0)
        if (H5Tclose(tid_3) < 0)
            ret_val = -1;
    if (did_3 > 0)
        if (H5Dclose(did_3) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBdelete_field(hid_t loc_id, const char *dset_name, const char *field_name)
{
    
    hid_t did_1 = H5I_INVALID_HID;
    hid_t tid_1 = H5I_INVALID_HID;
    hid_t sid_1 = H5I_INVALID_HID;
    hid_t pid_1 = H5I_INVALID_HID;
    
    hid_t did_2 = H5I_INVALID_HID;
    hid_t tid_2 = H5I_INVALID_HID;
    hid_t sid_2 = H5I_INVALID_HID;
    hid_t pid_2 = H5I_INVALID_HID;
    
    hid_t          did_3          = H5I_INVALID_HID;
    hid_t          tid_3          = H5I_INVALID_HID;
    hid_t          member_type_id = H5I_INVALID_HID;
    hid_t          preserve_id    = H5I_INVALID_HID;
    hid_t          read_type_id   = H5I_INVALID_HID;
    hid_t          write_type_id  = H5I_INVALID_HID;
    hid_t          attr_id        = H5I_INVALID_HID;
    size_t         member_size;
    size_t         type_size1;
    size_t         type_size2;
    size_t         curr_offset;
    size_t         delete_member_size = 0;
    size_t         member_offset;
    hsize_t        nfields;
    hsize_t        nrecords;
    hsize_t        dims_chunk[1];
    hsize_t        dims[1];
    hsize_t        maxdims[1] = {H5S_UNLIMITED};
    hsize_t        i;
    char           attr_name[255];
    char           table_title[255];
    char          *member_name  = NULL;
    unsigned char *tmp_buf      = NULL;
    unsigned char *tmp_fill_buf = NULL;
    size_t        *src_offset   = NULL;
    htri_t         has_fill     = false;
    herr_t         ret_val      = -1;

    
    if (dset_name == NULL)
        goto out;
    if (field_name == NULL)
        goto out;

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords) < 0)
        goto out;

    

    
    if ((did_1 = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((pid_1 = H5Dget_create_plist(did_1)) < 0)
        goto out;

    
    if ((tid_1 = H5Dget_type(did_1)) < 0)
        goto out;

    
    if (0 == (type_size1 = H5Tget_size(tid_1)))
        goto out;

    
    if ((sid_1 = H5Dget_space(did_1)) < 0)
        goto out;

    
    if (H5Sget_simple_extent_dims(sid_1, dims, NULL) < 0)
        goto out;

    

    
    for (i = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(tid_1, (unsigned)i)))
            goto out;

        
        if (H5TB_find_field(member_name, field_name)) {
            
            if ((member_type_id = H5Tget_member_type(tid_1, (unsigned)i)) < 0)
                goto out;

            
            if (0 == (delete_member_size = H5Tget_size(member_type_id)))
                goto out;

            
            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;

            H5free_memory(member_name);
            member_name = NULL;

            break;
        } 

        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if (delete_member_size == 0)
        goto out;

    

    type_size2 = type_size1 - delete_member_size;

    
    if ((tid_2 = H5Tcreate(H5T_COMPOUND, type_size2)) < 0)
        goto out;

    curr_offset = 0;

    
    if (NULL == (tmp_fill_buf = (unsigned char *)malloc((size_t)type_size2)))
        goto out;

    

    
    if ((H5TBAget_title(did_1, table_title)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(tid_1, (unsigned)i)))
            goto out;

        
        if (!H5TB_find_field(member_name, field_name)) {
            
            if ((member_type_id = H5Tget_member_type(tid_1, (unsigned)i)) < 0)
                goto out;

            
            if (0 == (member_size = H5Tget_size(member_type_id)))
                goto out;

            
            if (H5Tinsert(tid_2, member_name, curr_offset, member_type_id) < 0)
                goto out;

            

            snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)i);

            
            if ((has_fill = H5Aexists(did_1, attr_name)) < 0)
                goto out;

            
            if (has_fill > 0)
                if (H5LT_get_attribute_disk(did_1, attr_name, tmp_fill_buf + curr_offset) < 0)
                    goto out;

            curr_offset += member_size;

            
            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;
        } 

        H5free_memory(member_name);
        member_name = NULL;
    } 

    

    
    if (H5Pget_chunk(pid_1, 1, dims_chunk) < 0)
        goto out;

    
    if ((sid_2 = H5Screate_simple(1, dims, maxdims)) < 0)
        goto out;

    
    pid_2 = H5Pcreate(H5P_DATASET_CREATE);
    if (H5Pset_chunk(pid_2, 1, dims_chunk) < 0)
        goto out;

    
    if ((did_2 = H5Dcreate2(loc_id, "new", tid_2, sid_2, H5P_DEFAULT, pid_2, H5P_DEFAULT)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(tid_1, (unsigned)i)))
            goto out;

        
        if (!H5TB_find_field(member_name, field_name)) {
            
            if ((member_type_id = H5Tget_member_type(tid_1, (unsigned)i)) < 0)
                goto out;

            
            if (0 == (member_size = H5Tget_size(member_type_id)))
                goto out;

            
            if ((read_type_id = H5Tcreate(H5T_COMPOUND, member_size)) < 0)
                goto out;

            
            if (H5Tinsert(read_type_id, member_name, (size_t)0, member_type_id) < 0)
                goto out;

            if (NULL == (tmp_buf = (unsigned char *)calloc((size_t)nrecords, member_size)))
                goto out;

            
            if (H5Dread(did_1, read_type_id, H5S_ALL, H5S_ALL, H5P_DEFAULT, tmp_buf) < 0)
                goto out;

            
            if ((write_type_id = H5Tcreate(H5T_COMPOUND, member_size)) < 0)
                goto out;

            
            if (H5Tinsert(write_type_id, member_name, (size_t)0, member_type_id) < 0)
                goto out;

            
            if ((preserve_id = H5Pcreate(H5P_DATASET_XFER)) < 0)
                goto out;
            if (H5Pset_preserve(preserve_id, 1) < 0)
                goto out;

            
            if (H5Dwrite(did_2, write_type_id, H5S_ALL, H5S_ALL, preserve_id, tmp_buf) < 0)
                goto out;

            
            if (H5Pclose(preserve_id) < 0)
                goto out;
            preserve_id = H5I_INVALID_HID;

            
            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;

            
            if (H5Tclose(read_type_id) < 0)
                goto out;
            read_type_id = H5I_INVALID_HID;

            
            if (H5Tclose(write_type_id) < 0)
                goto out;
            write_type_id = H5I_INVALID_HID;

            free(tmp_buf);
            tmp_buf = NULL;
        } 

        
        H5free_memory(member_name);
        member_name = NULL;
    } 

    
    if (H5Ldelete(loc_id, dset_name, H5P_DEFAULT) < 0)
        goto out;

    
    if (H5Lmove(loc_id, "new", H5L_SAME_LOC, dset_name, H5P_DEFAULT, H5P_DEFAULT) < 0)
        goto out;

    

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords) < 0)
        goto out;

    
    if ((did_3 = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid_3 = H5Dget_type(did_3)) < 0)
        goto out;

    
    if (H5TB_attach_attributes(table_title, loc_id, dset_name, nfields, tid_3) < 0)
        goto out;

    if (NULL == (src_offset = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;

    
    if (H5TBget_field_info(loc_id, dset_name, NULL, NULL, src_offset, NULL) < 0)
        goto out;

    
    if (has_fill > 0) {
        if ((sid_1 = H5Screate(H5S_SCALAR)) < 0)
            goto out;

        for (i = 0; i < nfields; i++) {
            
            if ((member_type_id = H5Tget_member_type(tid_3, (unsigned)i)) < 0)
                goto out;

            
            member_offset = src_offset[i];

            snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)i);

            if ((attr_id = H5Acreate2(did_3, attr_name, member_type_id, sid_1, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                goto out;

            if (H5Awrite(attr_id, member_type_id, tmp_fill_buf + member_offset) < 0)
                goto out;

            if (H5Aclose(attr_id) < 0)
                goto out;
            attr_id = H5I_INVALID_HID;

            
            if (H5Tclose(member_type_id) < 0)
                goto out;
            member_type_id = H5I_INVALID_HID;
        } 

        
        if (H5Sclose(sid_1) < 0)
            goto out;
        sid_1 = H5I_INVALID_HID;
    } 

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);
    if (src_offset)
        free(src_offset);
    if (tmp_fill_buf)
        free(tmp_fill_buf);
    if (tmp_buf)
        free(tmp_buf);
    if (attr_id > 0)
        if (H5Aclose(attr_id) < 0)
            ret_val = -1;
    if (preserve_id > 0)
        if (H5Pclose(preserve_id) < 0)
            ret_val = -1;
    if (member_type_id > 0)
        if (H5Tclose(member_type_id) < 0)
            ret_val = -1;
    if (read_type_id > 0)
        if (H5Tclose(read_type_id) < 0)
            ret_val = -1;
    if (write_type_id > 0)
        if (H5Tclose(write_type_id) < 0)
            ret_val = -1;
    if (tid_1 > 0)
        if (H5Tclose(tid_1) < 0)
            ret_val = -1;
    if (pid_1 > 0)
        if (H5Pclose(pid_1) < 0)
            ret_val = -1;
    if (sid_1 > 0)
        if (H5Sclose(sid_1) < 0)
            ret_val = -1;
    if (did_1 > 0)
        if (H5Dclose(did_1) < 0)
            ret_val = -1;
    if (sid_2 > 0)
        if (H5Sclose(sid_2) < 0)
            ret_val = -1;
    if (tid_2 > 0)
        if (H5Tclose(tid_2) < 0)
            ret_val = -1;
    if (pid_2 > 0)
        if (H5Pclose(pid_2) < 0)
            ret_val = -1;
    if (did_2 > 0)
        if (H5Dclose(did_2) < 0)
            ret_val = -1;
    if (tid_3 > 0)
        if (H5Tclose(tid_3) < 0)
            ret_val = -1;
    if (did_3 > 0)
        if (H5Dclose(did_3) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBAget_title(hid_t loc_id, char *table_title)
{

    
    if (H5LT_get_attribute_disk(loc_id, "TITLE", table_title) < 0)
        return -1;

    return 0;
}

htri_t
H5TBAget_fill(hid_t loc_id, const char *dset_name, hid_t dset_id, unsigned char *dst_buf)
{
    hsize_t nfields;
    hsize_t nrecords;
    hsize_t i;
    size_t *src_offset = NULL;
    char    attr_name[255];
    htri_t  has_fill = false;
    htri_t  ret_val  = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, &nrecords) < 0)
        goto out;

    if (NULL == (src_offset = (size_t *)malloc((size_t)nfields * sizeof(size_t))))
        goto out;

    
    if (H5TBget_field_info(loc_id, dset_name, NULL, NULL, src_offset, NULL) < 0)
        goto out;

    for (i = 0; i < nfields; i++) {
        snprintf(attr_name, sizeof(attr_name), "FIELD_%d_FILL", (int)i);

        
        if ((has_fill = H5Aexists(dset_id, attr_name)) < 0)
            goto out;

        
        if (has_fill > 0)
            if (H5LT_get_attribute_disk(dset_id, attr_name, dst_buf + src_offset[i]) < 0)
                goto out;
    }

    ret_val = has_fill;

out:
    if (src_offset)
        free(src_offset);

    return ret_val;
} 

herr_t
H5TBget_table_info(hid_t loc_id, const char *dset_name, hsize_t *nfields, hsize_t *nrecords)
{
    hid_t   tid = H5I_INVALID_HID;
    hid_t   sid = H5I_INVALID_HID;
    hid_t   did = H5I_INVALID_HID;
    hsize_t dims[1];
    int     num_members;
    herr_t  ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if ((num_members = H5Tget_nmembers(tid)) < 0)
        goto out;

    
    if (nfields)
        *nfields = (hsize_t)num_members;

    
    if (nrecords) {
        
        if ((sid = H5Dget_space(did)) < 0)
            goto out;

        
        if (H5Sget_simple_extent_dims(sid, dims, NULL) < 0)
            goto out;

        
        if (H5Sclose(sid) < 0)
            goto out;
        sid = H5I_INVALID_HID;

        *nrecords = dims[0];
    } 

    ret_val = 0;

out:
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TBget_field_info(hid_t loc_id, const char *dset_name, char *field_names[], size_t *field_sizes,
                   size_t *field_offsets, size_t *type_size)
{
    hid_t    did    = H5I_INVALID_HID; 
    hid_t    tid    = H5I_INVALID_HID; 
    hid_t    n_tid  = H5I_INVALID_HID; 
    hid_t    m_tid  = H5I_INVALID_HID; 
    hid_t    nm_tid = H5I_INVALID_HID; 
    hssize_t nfields;
    hssize_t i;
    herr_t   ret_val = -1;

    
    if (dset_name == NULL)
        goto out;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        goto out;

    
    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    if ((n_tid = H5Tget_native_type(tid, H5T_DIR_DEFAULT)) < 0)
        goto out;

    
    if (type_size)
        if (0 == (*type_size = H5Tget_size(n_tid)))
            goto out;

    
    if ((nfields = H5Tget_nmembers(tid)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        
        if (field_names) {
            char *member_name;

            if (NULL == (member_name = H5Tget_member_name(tid, (unsigned)i)))
                goto out;
            strcpy(field_names[i], member_name);
            H5free_memory(member_name);
        } 

        
        if ((m_tid = H5Tget_member_type(tid, (unsigned)i)) < 0)
            goto out;
        if ((nm_tid = H5Tget_native_type(m_tid, H5T_DIR_DEFAULT)) < 0)
            goto out;

        
        if (field_sizes)
            if (0 == (field_sizes[i] = H5Tget_size(nm_tid)))
                goto out;

        
        if (field_offsets)
            field_offsets[i] = H5Tget_member_offset(n_tid, (unsigned)i);

        
        if (H5Tclose(m_tid) < 0)
            goto out;
        m_tid = H5I_INVALID_HID;
        if (H5Tclose(nm_tid) < 0)
            goto out;
        nm_tid = H5I_INVALID_HID;
    } 

    ret_val = 0;

out:
    if (tid > 0)
        if (H5Tclose(tid) < 0)
            ret_val = -1;
    if (n_tid > 0)
        if (H5Tclose(n_tid) < 0)
            ret_val = -1;
    if (m_tid > 0)
        if (H5Tclose(m_tid) < 0)
            ret_val = -1;
    if (nm_tid > 0)
        if (H5Tclose(nm_tid) < 0)
            ret_val = -1;
    if (did > 0)
        if (H5Dclose(did) < 0)
            ret_val = -1;

    return ret_val;
} 

H5_ATTR_PURE static bool
H5TB_find_field(const char *field, const char *field_list)
{
    const char *start = field_list;
    const char *end;

    
    if (field == NULL)
        return false;
    if (field_list == NULL)
        return false;

    while ((end = strstr(start, ",")) != 0) {
        ptrdiff_t count = end - start;

        if (strncmp(start, field, (size_t)count) == 0 && (size_t)count == strlen(field))
            return true;
        start = end + 1;
    } 

    if (strncmp(start, field, strlen(field)) == 0)
        return true;

    return false;
} 

static herr_t
H5TB_attach_attributes(const char *table_title, hid_t loc_id, const char *dset_name, hsize_t nfields,
                       hid_t tid)
{
    char    attr_name[255];
    char   *member_name = NULL;
    hsize_t i;
    herr_t  ret_val = -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "CLASS", TABLE_CLASS) < 0)
        goto out;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "VERSION", TABLE_VERSION) < 0)
        goto out;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "TITLE", table_title) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        
        if (NULL == (member_name = H5Tget_member_name(tid, (unsigned)i)))
            goto out;

        snprintf(attr_name, sizeof(attr_name), "FIELD_%d_NAME", (int)i);

        
        if (H5LTset_attribute_string(loc_id, dset_name, attr_name, member_name) < 0)
            goto out;

        H5free_memory(member_name);
        member_name = NULL;
    } 

    ret_val = 0;

out:
    if (member_name)
        H5free_memory(member_name);

    return ret_val;
} 

static hid_t
H5TB_create_type(hid_t loc_id, const char *dset_name, size_t type_size, const size_t *field_offset,
                 const size_t *field_sizes, hid_t ftype_id)
{
    hid_t    mem_type_id = H5I_INVALID_HID;
    hid_t    mtype_id    = H5I_INVALID_HID;
    hid_t    nmtype_id   = H5I_INVALID_HID;
    size_t   size_native;
    hsize_t  nfields = 0;
    char   **fnames  = NULL;
    unsigned i;
    hid_t    ret_val = -1;

    
    if (H5TBget_table_info(loc_id, dset_name, &nfields, NULL) < 0)
        goto out;

    if (NULL == (fnames = (char **)calloc((size_t)nfields, sizeof(char *))))
        goto out;

    for (i = 0; i < nfields; i++)
        if (NULL == (fnames[i] = (char *)malloc(HLTB_MAX_FIELD_LEN)))
            goto out;

    
    if (H5TBget_field_info(loc_id, dset_name, fnames, NULL, NULL, NULL) < 0)
        goto out;

    
    if ((mem_type_id = H5Tcreate(H5T_COMPOUND, type_size)) < 0)
        goto out;

    
    for (i = 0; i < nfields; i++) {
        if ((mtype_id = H5Tget_member_type(ftype_id, i)) < 0)
            goto out;
        if ((nmtype_id = H5Tget_native_type(mtype_id, H5T_DIR_DEFAULT)) < 0)
            goto out;
        if (0 == (size_native = H5Tget_size(nmtype_id)))
            goto out;
        if (field_sizes[i] != size_native)
            if (H5Tset_size(nmtype_id, field_sizes[i]) < 0)
                goto out;
        if (H5Tinsert(mem_type_id, fnames[i], field_offset[i], nmtype_id) < 0)
            goto out;
        if (H5Tclose(mtype_id) < 0)
            goto out;
        mtype_id = H5I_INVALID_HID;
        if (H5Tclose(nmtype_id) < 0)
            goto out;
        nmtype_id = H5I_INVALID_HID;
    } 

    ret_val = mem_type_id;

out:
    if (fnames) {
        for (i = 0; i < nfields; i++)
            if (fnames[i])
                free(fnames[i]);
        free(fnames);
    } 
    if (mtype_id > 0)
        if (H5Tclose(mtype_id) < 0)
            ret_val = -1;
    if (nmtype_id > 0)
        if (H5Tclose(nmtype_id) < 0)
            ret_val = -1;
    if (ret_val < 0 && mem_type_id > 0)
        H5Tclose(mem_type_id);

    return ret_val;
} 

herr_t
H5TB_common_append_records(hid_t dataset_id, hid_t mem_type_id, size_t nrecords, hsize_t orig_table_size,
                           const void *buf)
{
    hid_t   sid   = H5I_INVALID_HID;
    hid_t   m_sid = H5I_INVALID_HID;
    hsize_t count[1];
    hsize_t offset[1];
    hsize_t dims[1];
    hsize_t mem_dims[1];
    herr_t  ret_val = -1;

    
    dims[0] = nrecords + orig_table_size;
    if (H5Dset_extent(dataset_id, dims) < 0)
        goto out;

    
    mem_dims[0] = nrecords;
    if ((m_sid = H5Screate_simple(1, mem_dims, NULL)) < 0)
        goto out;

    
    if ((sid = H5Dget_space(dataset_id)) < 0)
        goto out;

    
    offset[0] = orig_table_size;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    if (H5Dwrite(dataset_id, mem_type_id, m_sid, sid, H5P_DEFAULT, buf) < 0)
        goto out;

    ret_val = 0;

out:
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;

    return ret_val;
} 

herr_t
H5TB_common_read_records(hid_t dataset_id, hid_t mem_type_id, hsize_t start, size_t nrecords,
                         hsize_t table_size, void *buf)
{
    hid_t   sid   = H5I_INVALID_HID;
    hid_t   m_sid = H5I_INVALID_HID;
    hsize_t count[1];
    hsize_t offset[1];
    hsize_t mem_size[1];
    herr_t  ret_val = -1;

    
    if (start + nrecords > table_size)
        goto out;

    
    if ((sid = H5Dget_space(dataset_id)) < 0)
        goto out;

    
    offset[0] = start;
    count[0]  = nrecords;
    if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, offset, NULL, count, NULL) < 0)
        goto out;

    
    mem_size[0] = count[0];
    if ((m_sid = H5Screate_simple(1, mem_size, NULL)) < 0)
        goto out;
    if ((H5Dread(dataset_id, mem_type_id, m_sid, sid, H5P_DEFAULT, buf)) < 0)
        goto out;

    ret_val = 0;

out:
    if (m_sid > 0)
        if (H5Sclose(m_sid) < 0)
            ret_val = -1;
    if (sid > 0)
        if (H5Sclose(sid) < 0)
            ret_val = -1;

    return ret_val;
} 
