/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Zmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5Zpkg.h"      

#ifdef H5_HAVE_FILTER_DEFLATE

#if defined(H5_HAVE_ZLIB_H) && !defined(H5_ZLIB_HEADER)
#define H5_ZLIB_HEADER "zlib.h"
#endif
#if defined(H5_ZLIB_HEADER)
#include H5_ZLIB_HEADER 
#endif

static size_t H5Z__filter_deflate(unsigned flags, size_t cd_nelmts, const unsigned cd_values[], size_t nbytes,
                                  size_t *buf_size, void **buf);

const H5Z_class2_t H5Z_DEFLATE[1] = {{
    H5Z_CLASS_T_VERS,    
    H5Z_FILTER_DEFLATE,  
    1,                   
    1,                   
    "deflate",           
    NULL,                
    NULL,                
    H5Z__filter_deflate, 
}};

static size_t
H5Z__filter_deflate(unsigned flags, size_t cd_nelmts, const unsigned cd_values[], size_t nbytes,
                    size_t *buf_size, void **buf)
{
    void  *outbuf = NULL; 
    int    status;        
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    assert(*buf_size > 0);
    assert(buf);
    assert(*buf);

    
    if (cd_nelmts != 1 || cd_values[0] > 9)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "invalid deflate aggression level");

    if (flags & H5Z_FLAG_REVERSE) {
        
#if defined(H5_HAVE_ZLIBNG_H)
        zng_stream z_strm; 
#else
        z_stream z_strm; 
#endif
        size_t nalloc = *buf_size; 

        
        if (NULL == (outbuf = H5MM_malloc(nalloc)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0, "memory allocation failed for deflate uncompression");

        
        memset(&z_strm, 0, sizeof(z_strm));
        z_strm.next_in = (Bytef *)*buf;
        H5_CHECKED_ASSIGN(z_strm.avail_in, unsigned, nbytes, size_t);
        z_strm.next_out = (Bytef *)outbuf;
        H5_CHECKED_ASSIGN(z_strm.avail_out, unsigned, nalloc, size_t);

        
#if defined(H5_HAVE_ZLIBNG_H)
        if (Z_OK != zng_inflateInit(&z_strm))
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, 0, "inflateInit() failed");
#else
        if (Z_OK != inflateInit(&z_strm))
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, 0, "inflateInit() failed");
#endif

        
        do {
            
#if defined(H5_HAVE_ZLIBNG_H)
            status = zng_inflate(&z_strm, Z_SYNC_FLUSH);
#else
            status = inflate(&z_strm, Z_SYNC_FLUSH);
#endif

            
            if (Z_STREAM_END == status)
                break; 

            
            if (Z_OK != status) {
#if defined(H5_HAVE_ZLIBNG_H)
                (void)zng_inflateEnd(&z_strm);
#else
                (void)inflateEnd(&z_strm);
#endif
                HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, 0, "inflate() failed");
            }
            else {
                
                if (0 == z_strm.avail_out) {
                    void *new_outbuf; 

                    
                    nalloc *= 2;
                    if (NULL == (new_outbuf = H5MM_realloc(outbuf, nalloc))) {
#if defined(H5_HAVE_ZLIBNG_H)
                        (void)zng_inflateEnd(&z_strm);
#else
                        (void)inflateEnd(&z_strm);
#endif
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0,
                                    "memory allocation failed for deflate uncompression");
                    } 
                    outbuf = new_outbuf;

                    
                    z_strm.next_out  = (unsigned char *)outbuf + z_strm.total_out;
                    z_strm.avail_out = (uInt)(nalloc - z_strm.total_out);
                } 
            }     
        } while (status == Z_OK);

        
        H5MM_xfree(*buf);

        
        *buf      = outbuf;
        outbuf    = NULL;
        *buf_size = nalloc;
        ret_value = z_strm.total_out;

        
#if defined(H5_HAVE_ZLIBNG_H)
        (void)zng_inflateEnd(&z_strm);
#else
        (void)inflateEnd(&z_strm);
#endif
    } 
    else {
        
        const Bytef *z_src = (const Bytef *)(*buf);
        Bytef       *z_dst; 
#if defined(H5_HAVE_ZLIBNG_H)
        uLongf z_dst_nbytes = (uLongf)zng_compressBound(nbytes);
#else
        uLongf z_dst_nbytes = (uLongf)compressBound(nbytes);
#endif
        uLong z_src_nbytes = (uLong)nbytes;
        int   aggression; 

        
        H5_CHECKED_ASSIGN(aggression, int, cd_values[0], unsigned);

        
        if (NULL == (outbuf = H5MM_malloc(z_dst_nbytes)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0, "unable to allocate deflate destination buffer");
        z_dst = (Bytef *)outbuf;

        
#if defined(H5_HAVE_ZLIBNG_H)
        status = zng_compress2(z_dst, &z_dst_nbytes, z_src, z_src_nbytes, aggression);
#else
        status = compress2(z_dst, &z_dst_nbytes, z_src, z_src_nbytes, aggression);
#endif

        
        if (Z_BUF_ERROR == status)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, 0, "overflow");
        else if (Z_MEM_ERROR == status)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, 0, "deflate memory error");
        else if (Z_OK != status)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, 0, "other deflate error");
        
        else {
            
            H5MM_xfree(*buf);

            
            *buf      = outbuf;
            outbuf    = NULL;
            *buf_size = nbytes;
            ret_value = z_dst_nbytes;
        } 
    }     

done:
    if (outbuf)
        H5MM_xfree(outbuf);
    FUNC_LEAVE_NOAPI(ret_value)
}
#endif 
