/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5B2module.h" 

#include "H5private.h"   
#include "H5B2pkg.h"     
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 

static herr_t H5B2__shadow_internal(H5B2_internal_t *internal, H5B2_node_ptr_t *curr_node_ptr);

H5FL_DEFINE(H5B2_internal_t);

herr_t
H5B2__create_internal(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr, uint16_t depth)
{
    H5B2_internal_t *internal  = NULL;    
    bool             inserted  = false;   
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(node_ptr);
    assert(depth > 0);

    
    if (NULL == (internal = H5FL_CALLOC(H5B2_internal_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for B-tree internal info");

    
    if (H5B2__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, FAIL, "can't increment ref. count on B-tree header");

    
    internal->hdr = hdr;

    
    if (NULL == (internal->int_native = (uint8_t *)H5FL_FAC_MALLOC(hdr->node_info[depth].nat_rec_fac)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for B-tree internal native keys");
    memset(internal->int_native, 0, hdr->cls->nrec_size * hdr->node_info[depth].max_nrec);

    
    if (NULL ==
        (internal->node_ptrs = (H5B2_node_ptr_t *)H5FL_FAC_MALLOC(hdr->node_info[depth].node_ptr_fac)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for B-tree internal node pointers");
    memset(internal->node_ptrs, 0, sizeof(H5B2_node_ptr_t) * (hdr->node_info[depth].max_nrec + 1));

    
    internal->depth = depth;

    
    internal->parent = parent;

    
    internal->shadow_epoch = hdr->shadow_epoch;

    
    if (HADDR_UNDEF == (node_ptr->addr = H5MF_alloc(hdr->f, H5FD_MEM_BTREE, (hsize_t)hdr->node_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "file allocation failed for B-tree internal node");

    
    if (H5AC_insert_entry(hdr->f, H5AC_BT2_INT, node_ptr->addr, internal, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL, "can't add B-tree internal node to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, internal) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, FAIL, "unable to add v2 B-tree node as child of proxy");
        internal->top_proxy = hdr->top_proxy;
    } 

done:
    if (ret_value < 0) {
        if (internal) {
            
            if (inserted)
                if (H5AC_remove_entry(internal) < 0)
                    HDONE_ERROR(H5E_BTREE, H5E_CANTREMOVE, FAIL,
                                "unable to remove v2 B-tree internal node from cache");

            
            if (H5_addr_defined(node_ptr->addr) &&
                H5MF_xfree(hdr->f, H5FD_MEM_BTREE, node_ptr->addr, (hsize_t)hdr->node_size) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL,
                            "unable to release file space for v2 B-tree internal node");

            
            if (H5B2__internal_free(internal) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to release v2 B-tree internal node");
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5B2_internal_t *
H5B2__protect_internal(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr, uint16_t depth, bool shadow,
                       unsigned flags)
{
    H5B2_internal_cache_ud_t udata;            
    H5B2_internal_t         *internal  = NULL; 
    H5B2_internal_t         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(node_ptr);
    assert(H5_addr_defined(node_ptr->addr));
    assert(depth > 0);

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.f      = hdr->f;
    udata.hdr    = hdr;
    udata.parent = parent;
    udata.nrec   = node_ptr->node_nrec;
    udata.depth  = depth;

    
    if (NULL ==
        (internal = (H5B2_internal_t *)H5AC_protect(hdr->f, H5AC_BT2_INT, node_ptr->addr, &udata, flags)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, NULL, "unable to protect B-tree internal node");

    
    if (hdr->top_proxy && NULL == internal->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, internal) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, NULL,
                        "unable to add v2 B-tree internal node as child of proxy");
        internal->top_proxy = hdr->top_proxy;
    } 

    
    if (shadow)
        if (H5B2__shadow_internal(internal, node_ptr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, NULL, "unable to shadow internal node");

    
    ret_value = internal;

done:
    
    if (!ret_value) {
        
        if (internal) {
            
            if (internal->top_proxy) {
                if (H5AC_proxy_entry_remove_child(internal->top_proxy, internal) < 0)
                    HDONE_ERROR(
                        H5E_BTREE, H5E_CANTUNDEPEND, NULL,
                        "unable to destroy flush dependency between internal node and v2 B-tree 'top' proxy");
                internal->top_proxy = NULL;
            } 

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, node_ptr->addr, internal, H5AC__NO_FLAGS_SET) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, NULL,
                            "unable to unprotect v2 B-tree internal node, address = %llu",
                            (unsigned long long)node_ptr->addr);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__neighbor_internal(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node_ptr, void *neighbor_loc,
                        H5B2_compare_t comp, void *parent, void *udata, H5B2_found_t op, void *op_data)
{
    H5B2_internal_t *internal;            
    unsigned         idx       = 0;       
    int              cmp       = 0;       
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(depth > 0);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));
    assert(op);

    
    if (NULL ==
        (internal = H5B2__protect_internal(hdr, parent, curr_node_ptr, depth, false, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree internal node");

    
    if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx, &cmp) <
        0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
    if (cmp > 0)
        idx++;

    
    if (comp == H5B2_COMPARE_LESS) {
        if (idx > 0)
            neighbor_loc = H5B2_INT_NREC(internal, hdr, idx - 1);
    } 
    else {
        assert(comp == H5B2_COMPARE_GREATER);

        if (idx < internal->nrec)
            neighbor_loc = H5B2_INT_NREC(internal, hdr, idx);
    } 

    
    if (depth > 1) {
        if (H5B2__neighbor_internal(hdr, (uint16_t)(depth - 1), &internal->node_ptrs[idx], neighbor_loc, comp,
                                    internal, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                        "unable to find neighbor record in B-tree internal node");
    } 
    else {
        if (H5B2__neighbor_leaf(hdr, &internal->node_ptrs[idx], neighbor_loc, comp, internal, udata, op,
                                op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "unable to find neighbor record in B-tree leaf node");
    } 

done:
    
    if (internal &&
        H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr->addr, internal, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release internal B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__insert_internal(H5B2_hdr_t *hdr, uint16_t depth, unsigned *parent_cache_info_flags_ptr,
                      H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos, void *parent, void *udata)
{
    H5B2_internal_t *internal       = NULL; 
    unsigned         internal_flags = H5AC__NO_FLAGS_SET;
    unsigned         idx            = 0;               
    H5B2_nodepos_t   next_pos       = H5B2_POS_MIDDLE; 
    herr_t           ret_value      = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(depth > 0);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL ==
        (internal = H5B2__protect_internal(hdr, parent, curr_node_ptr, depth, false, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree internal node");

    
    assert(internal->nrec == curr_node_ptr->node_nrec);

    
    {
        int      cmp;        
        unsigned retries;    
        size_t   split_nrec; 

        
        if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx,
                                &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        if (cmp == 0)
            HGOTO_ERROR(H5E_BTREE, H5E_EXISTS, FAIL, "record is already in B-tree");
        if (cmp > 0)
            idx++;

        
        
        retries = 2;

        
        split_nrec = hdr->node_info[depth - 1].split_nrec;

        
        while (internal->node_ptrs[idx].node_nrec == split_nrec) {
            
            if (idx == 0) { 
                if (retries > 0 && (internal->node_ptrs[idx + 1].node_nrec < split_nrec)) {
                    if (H5B2__redistribute2(hdr, depth, internal, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__split1(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to split child node");
                }                             
            }                                 
            else if (idx == internal->nrec) { 
                if (retries > 0 && (internal->node_ptrs[idx - 1].node_nrec < split_nrec)) {
                    if (H5B2__redistribute2(hdr, depth, internal, (idx - 1)) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__split1(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to split child node");
                }  
            }      
            else { 
                if (retries > 0 && ((internal->node_ptrs[idx + 1].node_nrec < split_nrec) ||
                                    (internal->node_ptrs[idx - 1].node_nrec < split_nrec))) {
                    if (H5B2__redistribute3(hdr, depth, internal, &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__split1(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to split child node");
                } 
            }     

            
            
            if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx,
                                    &cmp) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
            if (cmp == 0)
                HGOTO_ERROR(H5E_BTREE, H5E_EXISTS, FAIL, "record is already in B-tree");
            if (cmp > 0)
                idx++;

            
            retries--;
        } 
    }     

    
    if (H5B2_POS_MIDDLE != curr_pos) {
        if (idx == 0) {
            if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos)
                next_pos = H5B2_POS_LEFT;
        } 
        else if (idx == internal->nrec) {
            if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos)
                next_pos = H5B2_POS_RIGHT;
        } 
    }     

    
    if (depth > 1) {
        if (H5B2__insert_internal(hdr, (uint16_t)(depth - 1), &internal_flags, &internal->node_ptrs[idx],
                                  next_pos, internal, udata) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert record into B-tree internal node");
    } 
    else {
        if (H5B2__insert_leaf(hdr, &internal->node_ptrs[idx], next_pos, internal, udata) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert record into B-tree leaf node");
    } 

    
    curr_node_ptr->all_nrec++;

    
    internal_flags |= H5AC__DIRTIED_FLAG;

done:
    
    if (internal) {
        
        if (hdr->swmr_write && (internal_flags & H5AC__DIRTIED_FLAG))
            if (H5B2__shadow_internal(internal, curr_node_ptr) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow internal B-tree node");

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr->addr, internal, internal_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release internal B-tree node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__update_internal(H5B2_hdr_t *hdr, uint16_t depth, unsigned *parent_cache_info_flags_ptr,
                      H5B2_node_ptr_t *curr_node_ptr, H5B2_update_status_t *status, H5B2_nodepos_t curr_pos,
                      void *parent, void *udata, H5B2_modify_t op, void *op_data)
{
    H5B2_internal_t *internal       = NULL; 
    unsigned         internal_flags = H5AC__NO_FLAGS_SET;
    int              cmp;                         
    unsigned         idx       = 0;               
    H5B2_nodepos_t   next_pos  = H5B2_POS_MIDDLE; 
    herr_t           ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(depth > 0);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL ==
        (internal = H5B2__protect_internal(hdr, parent, curr_node_ptr, depth, false, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree internal node");

    
    assert(internal->nrec == curr_node_ptr->node_nrec);

    
    if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx, &cmp) <
        0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");

    
    if (0 == cmp) {
        bool changed = false; 

        
        if ((op)(H5B2_INT_NREC(internal, hdr, idx), op_data, &changed) < 0) {
            
            assert(changed == false);

            HGOTO_ERROR(H5E_BTREE, H5E_CANTMODIFY, FAIL,
                        "'modify' callback failed for B-tree update operation");
        } 

        
        internal_flags |= (changed ? H5AC__DIRTIED_FLAG : 0);

        
        *status = H5B2_UPDATE_MODIFY_DONE;
    } 
    else {
        
        if (cmp > 0)
            idx++;

        
        if (H5B2_POS_MIDDLE != curr_pos) {
            if (idx == 0) {
                if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos)
                    next_pos = H5B2_POS_LEFT;
            } 
            else if (idx == internal->nrec) {
                if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos)
                    next_pos = H5B2_POS_RIGHT;
            } 
        }     

        
        if (depth > 1) {
            if (H5B2__update_internal(hdr, (uint16_t)(depth - 1), &internal_flags, &internal->node_ptrs[idx],
                                      status, next_pos, internal, udata, op, op_data) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL,
                            "unable to update record in internal B-tree node");
        } 
        else {
            if (H5B2__update_leaf(hdr, &internal->node_ptrs[idx], status, next_pos, internal, udata, op,
                                  op_data) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL, "unable to update record in leaf B-tree node");
        } 

        
        switch (*status) {
            case H5B2_UPDATE_MODIFY_DONE:
                
                break;

            case H5B2_UPDATE_SHADOW_DONE:
                
                if (hdr->swmr_write)
                    internal_flags |= H5AC__DIRTIED_FLAG;

                
                *status = H5B2_UPDATE_MODIFY_DONE;
                break;

            case H5B2_UPDATE_INSERT_DONE:
                
                internal_flags |= H5AC__DIRTIED_FLAG;

                
                curr_node_ptr->all_nrec++;
                break;

            case H5B2_UPDATE_INSERT_CHILD_FULL:
                
                if (internal->nrec == hdr->node_info[depth].split_nrec) {
                    bool could_split = false; 

                    if (idx == 0) { 
                        
                        if ((internal->node_ptrs[idx].node_nrec + internal->node_ptrs[idx + 1].node_nrec) >=
                            (unsigned)((hdr->node_info[depth - 1].split_nrec * 2) - 1))
                            could_split = true;
                    }
                    else if (idx == internal->nrec) { 
                        
                        if ((internal->node_ptrs[idx - 1].node_nrec + internal->node_ptrs[idx].node_nrec) >=
                            (unsigned)((hdr->node_info[depth - 1].split_nrec * 2) - 1))
                            could_split = true;
                    }
                    else { 
                        
                        if ((internal->node_ptrs[idx - 1].node_nrec + internal->node_ptrs[idx].node_nrec) >=
                            (unsigned)((hdr->node_info[depth - 1].split_nrec * 2) - 1))
                            could_split = true;
                        
                        else if ((internal->node_ptrs[idx].node_nrec +
                                  internal->node_ptrs[idx + 1].node_nrec) >=
                                 (unsigned)((hdr->node_info[depth - 1].split_nrec * 2) - 1))
                            could_split = true;
                    }

                    
                    if (could_split) {
                        
                        if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr->addr, internal,
                                           internal_flags) < 0)
                            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL,
                                        "unable to release internal B-tree node");
                        internal = NULL;

                        
                        HGOTO_DONE(SUCCEED);
                    }
                }

                
                if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr->addr, internal, internal_flags) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release internal B-tree node");
                internal = NULL;

                
                *status = H5B2_UPDATE_INSERT_DONE;

                
                if (H5B2__insert_internal(hdr, depth, parent_cache_info_flags_ptr, curr_node_ptr, curr_pos,
                                          parent, udata) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL,
                                "unable to insert record into internal B-tree node");
                break;

            case H5B2_UPDATE_UNKNOWN:
            default:
                assert(0 && "Invalid update status");
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL, "invalid update status");
        } 
    }     

done:
    
    if (internal) {
        
        if (hdr->swmr_write && (internal_flags & H5AC__DIRTIED_FLAG)) {
            
            if (H5B2__shadow_internal(internal, curr_node_ptr) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow internal B-tree node");

            
            
            if (*status == H5B2_UPDATE_MODIFY_DONE)
                *status = H5B2_UPDATE_SHADOW_DONE;
        } 

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_INT, curr_node_ptr->addr, internal, internal_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release internal B-tree node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__shadow_internal(H5B2_internal_t *internal, H5B2_node_ptr_t *curr_node_ptr)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(internal);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));
    hdr = internal->hdr;
    assert(hdr);
    assert(hdr->swmr_write);

    
    if (internal->shadow_epoch <= hdr->shadow_epoch) {
        haddr_t new_node_addr; 

        
        
        if (HADDR_UNDEF == (new_node_addr = H5MF_alloc(hdr->f, H5FD_MEM_BTREE, (hsize_t)hdr->node_size)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL, "unable to allocate file space to move B-tree node");

        
        if (H5AC_move_entry(hdr->f, H5AC_BT2_INT, curr_node_ptr->addr, new_node_addr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTMOVE, FAIL, "unable to move B-tree node");
        curr_node_ptr->addr = new_node_addr;

        

        
        internal->shadow_epoch = hdr->shadow_epoch + 1;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__remove_internal(H5B2_hdr_t *hdr, bool *depth_decreased, void *swap_loc, void *swap_parent,
                      uint16_t depth, H5AC_info_t *parent_cache_info, unsigned *parent_cache_info_flags_ptr,
                      H5B2_nodepos_t curr_pos, H5B2_node_ptr_t *curr_node_ptr, void *udata, H5B2_remove_t op,
                      void *op_data)
{
    H5AC_info_t     *new_cache_info; 
    unsigned        *new_cache_info_flags_ptr = NULL;
    H5B2_node_ptr_t *new_node_ptr;                     
    H5B2_internal_t *internal;                         
    H5B2_nodepos_t   next_pos       = H5B2_POS_MIDDLE; 
    unsigned         internal_flags = H5AC__NO_FLAGS_SET;
    haddr_t          internal_addr  = HADDR_UNDEF; 
    size_t           merge_nrec;                   
    bool             collapsed_root = false;       
    herr_t           ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(depth > 0);
    assert(parent_cache_info);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL == (internal = H5B2__protect_internal(hdr, parent_cache_info, curr_node_ptr, depth, false,
                                                   H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree internal node");
    internal_addr = curr_node_ptr->addr;

    
    merge_nrec = hdr->node_info[depth - 1].merge_nrec;

    
    
    if (internal->nrec == 1 &&
        ((internal->node_ptrs[0].node_nrec + internal->node_ptrs[1].node_nrec) <= ((merge_nrec * 2) + 1))) {

        
        if (H5B2__merge2(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal, &internal_flags,
                         0) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");

        
        internal_flags |= H5AC__DELETED_FLAG;
        if (!hdr->swmr_write)
            internal_flags |= H5AC__FREE_FILE_SPACE_FLAG;

        
        curr_node_ptr->addr      = internal->node_ptrs[0].addr;
        curr_node_ptr->node_nrec = internal->node_ptrs[0].node_nrec;

        
        if (hdr->swmr_write)
            if (H5B2__update_flush_depend(hdr, depth, curr_node_ptr, internal, hdr) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL, "unable to update child node to new parent");

        
        *depth_decreased = true;

        
        new_cache_info           = parent_cache_info;
        new_cache_info_flags_ptr = parent_cache_info_flags_ptr;
        new_node_ptr             = curr_node_ptr;

        
        collapsed_root = true;

        
        next_pos = H5B2_POS_ROOT;
    } 
    
    else {
        unsigned idx = 0; 
        int      cmp = 0; 
        unsigned retries; 

        
        if (hdr->swmr_write) {
            if (H5B2__shadow_internal(internal, curr_node_ptr) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow internal node");
            internal_addr = curr_node_ptr->addr;
        } 

        
        if (swap_loc)
            idx = 0;
        else {
            if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata, &idx,
                                    &cmp) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
            if (cmp >= 0)
                idx++;
        } 

        
        
        retries = 2;

        
        while (internal->node_ptrs[idx].node_nrec == merge_nrec) {
            
            
            
            if (idx == 0) { 
                if (retries > 0 && (internal->node_ptrs[idx + 1].node_nrec > merge_nrec)) {
                    if (H5B2__redistribute2(hdr, depth, internal, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__merge2(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");
                }                             
            }                                 
            else if (idx == internal->nrec) { 
                if (retries > 0 && (internal->node_ptrs[idx - 1].node_nrec > merge_nrec)) {
                    if (H5B2__redistribute2(hdr, depth, internal, (idx - 1)) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__merge2(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, (idx - 1)) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");
                }  
            }      
            else { 
                if (retries > 0 && ((internal->node_ptrs[idx + 1].node_nrec > merge_nrec) ||
                                    (internal->node_ptrs[idx - 1].node_nrec > merge_nrec))) {
                    if (H5B2__redistribute3(hdr, depth, internal, &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__merge3(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");
                } 
            }     

            
            if (swap_loc)
                idx = 0;
            else {
                
                if (H5B2__locate_record(hdr->cls, internal->nrec, hdr->nat_off, internal->int_native, udata,
                                        &idx, &cmp) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
                if (cmp >= 0)
                    idx++;
            } 

            
            retries--;
        } 

        
        if (!swap_loc && cmp == 0) {
            swap_loc    = H5B2_INT_NREC(internal, hdr, idx - 1);
            swap_parent = internal;
        } 

        
        if (swap_loc && depth == 1)
            if (H5B2__swap_leaf(hdr, depth, internal, &internal_flags, idx, swap_loc) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTSWAP, FAIL, "Can't swap records in B-tree");

        
        new_cache_info_flags_ptr = &internal_flags;
        new_cache_info           = &internal->cache_info;
        new_node_ptr             = &internal->node_ptrs[idx];

        
        if (H5B2_POS_MIDDLE != curr_pos) {
            if (idx == 0) {
                if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos)
                    next_pos = H5B2_POS_LEFT;
            } 
            else if (idx == internal->nrec) {
                if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos)
                    next_pos = H5B2_POS_RIGHT;
            } 
        }     
    }         

    
    if (depth > 1) {
        if (H5B2__remove_internal(hdr, depth_decreased, swap_loc, swap_parent, (uint16_t)(depth - 1),
                                  new_cache_info, new_cache_info_flags_ptr, next_pos, new_node_ptr, udata, op,
                                  op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree internal node");
    } 
    else {
        if (H5B2__remove_leaf(hdr, new_node_ptr, next_pos, new_cache_info, udata, op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree leaf node");
    } 

    
    if (!collapsed_root)
        new_node_ptr->all_nrec--;

    
    if (!(hdr->swmr_write && collapsed_root))
        internal_flags |= H5AC__DIRTIED_FLAG;

#ifdef H5B2_DEBUG
    H5B2__assert_internal((!collapsed_root ? (curr_node_ptr->all_nrec - 1) : new_node_ptr->all_nrec), hdr,
                          internal);
#endif 

done:
    
    if (internal && H5AC_unprotect(hdr->f, H5AC_BT2_INT, internal_addr, internal, internal_flags) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release internal B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__remove_internal_by_idx(H5B2_hdr_t *hdr, bool *depth_decreased, void *swap_loc, void *swap_parent,
                             uint16_t depth, H5AC_info_t *parent_cache_info,
                             unsigned *parent_cache_info_flags_ptr, H5B2_node_ptr_t *curr_node_ptr,
                             H5B2_nodepos_t curr_pos, hsize_t n, H5B2_remove_t op, void *op_data)
{
    H5AC_info_t     *new_cache_info; 
    unsigned        *new_cache_info_flags_ptr = NULL;
    H5B2_node_ptr_t *new_node_ptr;                     
    H5B2_internal_t *internal;                         
    H5B2_nodepos_t   next_pos       = H5B2_POS_MIDDLE; 
    unsigned         internal_flags = H5AC__NO_FLAGS_SET;
    haddr_t          internal_addr  = HADDR_UNDEF; 
    size_t           merge_nrec;                   
    bool             collapsed_root = false;       
    herr_t           ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(depth > 0);
    assert(parent_cache_info);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL == (internal = H5B2__protect_internal(hdr, parent_cache_info, curr_node_ptr, depth, false,
                                                   H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree internal node");
    internal_addr = curr_node_ptr->addr;
    assert(internal->nrec == curr_node_ptr->node_nrec);
    assert(depth == hdr->depth || internal->nrec > 1);

    
    merge_nrec = hdr->node_info[depth - 1].merge_nrec;

    
    
    if (internal->nrec == 1 &&
        ((internal->node_ptrs[0].node_nrec + internal->node_ptrs[1].node_nrec) <= ((merge_nrec * 2) + 1))) {
        assert(depth == hdr->depth);

        
        if (H5B2__merge2(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal, &internal_flags,
                         0) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");

        
        internal_flags |= H5AC__DELETED_FLAG;
        if (!hdr->swmr_write)
            internal_flags |= H5AC__FREE_FILE_SPACE_FLAG;

        
        curr_node_ptr->addr      = internal->node_ptrs[0].addr;
        curr_node_ptr->node_nrec = internal->node_ptrs[0].node_nrec;

        
        if (hdr->swmr_write)
            if (H5B2__update_flush_depend(hdr, depth, curr_node_ptr, internal, hdr) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTUPDATE, FAIL, "unable to update child node to new parent");

        
        *depth_decreased = true;

        
        new_cache_info           = parent_cache_info;
        new_cache_info_flags_ptr = parent_cache_info_flags_ptr;
        new_node_ptr             = curr_node_ptr;

        
        collapsed_root = true;

        
        next_pos = H5B2_POS_ROOT;
    } 
    
    else {
        hsize_t  orig_n = n;    
        unsigned idx;           
        bool     found = false; 
        unsigned retries;       

        
        if (hdr->swmr_write) {
            if (H5B2__shadow_internal(internal, curr_node_ptr) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow internal node");
            internal_addr = curr_node_ptr->addr;
        } 

        
        if (swap_loc)
            idx = 0;
        else {
            
            for (idx = 0; idx < internal->nrec; idx++) {
                
                if (internal->node_ptrs[idx].all_nrec >= n) {
                    
                    if (internal->node_ptrs[idx].all_nrec == n) {
                        
                        found = true;
                        n     = 0;

                        
                        idx++;
                    } 

                    
                    break;
                } 

                
                n -= (internal->node_ptrs[idx].all_nrec + 1);
            } 
        }     

        
        
        retries = 2;

        
        while (internal->node_ptrs[idx].node_nrec == merge_nrec) {
            
            
            
            if (idx == 0) { 
                if (retries > 0 && (internal->node_ptrs[idx + 1].node_nrec > merge_nrec)) {
                    if (H5B2__redistribute2(hdr, depth, internal, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__merge2(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");
                }                             
            }                                 
            else if (idx == internal->nrec) { 
                if (retries > 0 && (internal->node_ptrs[idx - 1].node_nrec > merge_nrec)) {
                    if (H5B2__redistribute2(hdr, depth, internal, (idx - 1)) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__merge2(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, (idx - 1)) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");
                }  
            }      
            else { 
                if (retries > 0 && ((internal->node_ptrs[idx + 1].node_nrec > merge_nrec) ||
                                    (internal->node_ptrs[idx - 1].node_nrec > merge_nrec))) {
                    if (H5B2__redistribute3(hdr, depth, internal, &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTREDISTRIBUTE, FAIL,
                                    "unable to redistribute child node records");
                } 
                else {
                    if (H5B2__merge3(hdr, depth, curr_node_ptr, parent_cache_info_flags_ptr, internal,
                                     &internal_flags, idx) < 0)
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTSPLIT, FAIL, "unable to merge child node");
                } 
            }     

            
            if (swap_loc)
                idx = 0;
            else {
                
                n = orig_n;

                
                found = false;

                
                for (idx = 0; idx < internal->nrec; idx++) {
                    
                    if (internal->node_ptrs[idx].all_nrec >= n) {
                        
                        if (internal->node_ptrs[idx].all_nrec == n) {
                            
                            found = true;
                            n     = 0;

                            
                            idx++;
                        } 

                        
                        break;
                    } 

                    
                    n -= (internal->node_ptrs[idx].all_nrec + 1);
                } 
            }     

            
            retries--;
        } 

        
        if (!swap_loc && found) {
            swap_loc    = H5B2_INT_NREC(internal, hdr, idx - 1);
            swap_parent = internal;
        } 

        
        if (swap_loc && depth == 1)
            if (H5B2__swap_leaf(hdr, depth, internal, &internal_flags, idx, swap_loc) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTSWAP, FAIL, "can't swap records in B-tree");

        
        new_cache_info_flags_ptr = &internal_flags;
        new_cache_info           = &internal->cache_info;
        new_node_ptr             = &internal->node_ptrs[idx];

        
        if (H5B2_POS_MIDDLE != curr_pos) {
            if (idx == 0) {
                if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos)
                    next_pos = H5B2_POS_LEFT;
            } 
            else if (idx == internal->nrec) {
                if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos)
                    next_pos = H5B2_POS_RIGHT;
            } 
        }     
    }         

    
    if (depth > 1) {
        if (H5B2__remove_internal_by_idx(hdr, depth_decreased, swap_loc, swap_parent, (uint16_t)(depth - 1),
                                         new_cache_info, new_cache_info_flags_ptr, new_node_ptr, next_pos, n,
                                         op, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree internal node");
    } 
    else {
        if (H5B2__remove_leaf_by_idx(hdr, new_node_ptr, next_pos, new_cache_info, (unsigned)n, op, op_data) <
            0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record from B-tree leaf node");
    } 

    
    if (!collapsed_root)
        new_node_ptr->all_nrec--;

    
    if (!(hdr->swmr_write && collapsed_root))
        internal_flags |= H5AC__DIRTIED_FLAG;

#ifdef H5B2_DEBUG
    H5B2__assert_internal((!collapsed_root ? (curr_node_ptr->all_nrec - 1) : new_node_ptr->all_nrec), hdr,
                          internal);
#endif 

done:
    
    if (internal && H5AC_unprotect(hdr->f, H5AC_BT2_INT, internal_addr, internal, internal_flags) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release internal B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__internal_free(H5B2_internal_t *internal)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(internal);

    
    if (internal->int_native)
        internal->int_native = (uint8_t *)H5FL_FAC_FREE(internal->hdr->node_info[internal->depth].nat_rec_fac,
                                                        internal->int_native);

    
    if (internal->node_ptrs)
        internal->node_ptrs = (H5B2_node_ptr_t *)H5FL_FAC_FREE(
            internal->hdr->node_info[internal->depth].node_ptr_fac, internal->node_ptrs);

    
    if (H5B2__hdr_decr(internal->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTDEC, FAIL, "can't decrement ref. count on B-tree header");

    
    assert(NULL == internal->top_proxy);

    
    internal = H5FL_FREE(H5B2_internal_t, internal);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5B2_DEBUG

H5_ATTR_PURE herr_t
H5B2__assert_internal(hsize_t parent_all_nrec, const H5B2_hdr_t H5_ATTR_NDEBUG_UNUSED *hdr,
                      const H5B2_internal_t *internal)
{
    hsize_t  tot_all_nrec; 
    uint16_t u, v;         

    
    assert(internal->nrec <= hdr->node_info->split_nrec);

    
    tot_all_nrec = internal->nrec;
    for (u = 0; u < internal->nrec + 1; u++) {
        tot_all_nrec += internal->node_ptrs[u].all_nrec;

        assert(H5_addr_defined(internal->node_ptrs[u].addr));
        assert(internal->node_ptrs[u].addr > 0);
        for (v = 0; v < u; v++)
            assert(internal->node_ptrs[u].addr != internal->node_ptrs[v].addr);
    } 

    
    if (parent_all_nrec > 0)
        assert(tot_all_nrec == parent_all_nrec);

    return (0);
} 

H5_ATTR_PURE herr_t
H5B2__assert_internal2(hsize_t parent_all_nrec, const H5B2_hdr_t H5_ATTR_NDEBUG_UNUSED *hdr,
                       const H5B2_internal_t *internal, const H5B2_internal_t *internal2)
{
    hsize_t  tot_all_nrec; 
    uint16_t u, v;         

    
    assert(internal->nrec <= hdr->node_info->split_nrec);

    
    tot_all_nrec = internal->nrec;
    for (u = 0; u < internal->nrec + 1; u++) {
        tot_all_nrec += internal->node_ptrs[u].all_nrec;

        assert(H5_addr_defined(internal->node_ptrs[u].addr));
        assert(internal->node_ptrs[u].addr > 0);
        for (v = 0; v < u; v++)
            assert(internal->node_ptrs[u].addr != internal->node_ptrs[v].addr);
        for (v = 0; v < internal2->nrec + 1; v++)
            assert(internal->node_ptrs[u].addr != internal2->node_ptrs[v].addr);
    } 

    
    if (parent_all_nrec > 0)
        assert(tot_all_nrec == parent_all_nrec);

    return (0);
} 
#endif 
