/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

static herr_t H5T__set_order(H5T_t *dtype, H5T_order_t order);

H5T_order_t
H5Tget_order(hid_t type_id)
{
    H5T_t      *dt;        
    H5T_order_t ret_value; 

    FUNC_ENTER_API(H5T_ORDER_ERROR)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, H5T_ORDER_ERROR, "not a datatype");

    
    if (H5T_ORDER_ERROR == (ret_value = H5T_get_order(dt)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, H5T_ORDER_ERROR, "can't get order for specified datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_order_t
H5T_get_order(const H5T_t *dtype)
{
    H5T_order_t ret_value = H5T_ORDER_NONE; 

    FUNC_ENTER_NOAPI(H5T_ORDER_ERROR)

    
    while (dtype->shared->parent)
        dtype = dtype->shared->parent;

    
    if (H5T_IS_ATOMIC(dtype->shared))
        ret_value = dtype->shared->u.atomic.order;
    else {
        
        if (H5T_COMPOUND == dtype->shared->type) {
            H5T_order_t memb_order = H5T_ORDER_NONE;
            int         nmemb; 
            int         i;     

            
            if ((nmemb = H5T_get_nmembers(dtype)) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_ORDER_ERROR,
                            "can't get number of members from compound data type");

            
            for (i = 0; i < nmemb; i++) {
                
                if ((memb_order = H5T_get_order(dtype->shared->u.compnd.memb[i].type)) == H5T_ORDER_ERROR)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5T_ORDER_ERROR,
                                "can't get order for compound member");

                
                if (memb_order != H5T_ORDER_NONE && ret_value == H5T_ORDER_NONE)
                    ret_value = memb_order;

                
                if (memb_order != H5T_ORDER_NONE && ret_value != H5T_ORDER_NONE && memb_order != ret_value) {
                    ret_value = H5T_ORDER_MIXED;
                    break;
                } 
            }     
        }         
    }             

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tset_order(hid_t type_id, H5T_order_t order)
{
    H5T_t *dt        = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a datatype");
    if (order < H5T_ORDER_LE || order > H5T_ORDER_NONE || order == H5T_ORDER_MIXED)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "illegal byte order");
    if (NULL != dt->vol_obj)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is already committed");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "datatype is read-only");

    
    if (H5T__set_order(dt, order) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "can't set order");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5T__set_order(H5T_t *dtype, H5T_order_t order)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5T_ENUM == dtype->shared->type && dtype->shared->u.enumer.nmembs > 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "operation not allowed after enum members are defined");

    
    while (dtype->shared->parent)
        dtype = dtype->shared->parent;

    
    if (order == H5T_ORDER_NONE && !(H5T_REFERENCE == dtype->shared->type ||
                                     H5T_OPAQUE == dtype->shared->type || H5T_IS_FIXED_STRING(dtype->shared)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "illegal byte order for type");

    
    if (H5T_IS_ATOMIC(dtype->shared))
        dtype->shared->u.atomic.order = order;
    else {
        
        if (H5T_COMPOUND == dtype->shared->type) {
            int nmemb; 
            int i;     

            
            if ((nmemb = H5T_get_nmembers(dtype)) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL,
                            "can't get number of members from compound data type");

            
            if (nmemb == 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNINITIALIZED, FAIL, "no member is in the compound data type");

            
            for (i = 0; i < nmemb; i++)
                if (H5T__set_order(dtype->shared->u.compnd.memb[i].type, order) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't set order for compound member");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
