/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__fsinfo_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                 size_t p_size, const uint8_t *p);
static herr_t H5O__fsinfo_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                 const void *_mesg);
static void  *H5O__fsinfo_copy(const void *_mesg, void *_dest);
static size_t H5O__fsinfo_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__fsinfo_free(void *mesg);
static herr_t H5O__fsinfo_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_FSINFO[1] = {{
    H5O_FSINFO_ID,        
    "fsinfo",             
    sizeof(H5O_fsinfo_t), 
    0,                    
    H5O__fsinfo_decode,   
    H5O__fsinfo_encode,   
    H5O__fsinfo_copy,     
    H5O__fsinfo_size,     
    NULL,                 
    H5O__fsinfo_free,     
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    H5O__fsinfo_debug     
}};

static const unsigned H5O_fsinfo_ver_bounds[] = {
    H5O_INVALID_VERSION,      
    H5O_INVALID_VERSION,      
    H5O_FSINFO_VERSION_1,     
    H5O_FSINFO_VERSION_1,     
    H5O_FSINFO_VERSION_1,     
    H5O_FSINFO_VERSION_1,     
    H5O_FSINFO_VERSION_LATEST 
};
#define N_FSINFO_VERSION_BOUNDS H5F_LIBVER_NBOUNDS

H5FL_DEFINE_STATIC(H5O_fsinfo_t);

static void *
H5O__fsinfo_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                   unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_fsinfo_t  *fsinfo = NULL;              
    H5F_mem_page_t ptype;                      
    unsigned       vers;                       
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (NULL == (fsinfo = H5FL_CALLOC(H5O_fsinfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    for (ptype = H5F_MEM_PAGE_SUPER; ptype < H5F_MEM_PAGE_NTYPES; ptype++)
        fsinfo->fs_addr[ptype - 1] = HADDR_UNDEF;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    vers = *p++;

    if (vers == H5O_FSINFO_VERSION_0) {
        H5F_file_space_type_t strategy;      
        hsize_t               threshold = 0; 
        H5FD_mem_t            type;          

        fsinfo->persist             = H5F_FREE_SPACE_PERSIST_DEF;
        fsinfo->threshold           = H5F_FREE_SPACE_THRESHOLD_DEF;
        fsinfo->page_size           = H5F_FILE_SPACE_PAGE_SIZE_DEF;
        fsinfo->pgend_meta_thres    = H5F_FILE_SPACE_PGEND_META_THRES;
        fsinfo->eoa_pre_fsm_fsalloc = HADDR_UNDEF;

        if (H5_IS_BUFFER_OVERFLOW(p, 1 + H5F_sizeof_size(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        strategy = (H5F_file_space_type_t)*p++; 
        H5F_DECODE_LENGTH(f, p, threshold);     

        
        switch (strategy) {
            case H5F_FILE_SPACE_ALL_PERSIST:
                fsinfo->strategy  = H5F_FSPACE_STRATEGY_FSM_AGGR;
                fsinfo->persist   = true;
                fsinfo->threshold = threshold;
                if (HADDR_UNDEF == (fsinfo->eoa_pre_fsm_fsalloc = H5F_get_eoa(f, H5FD_MEM_DEFAULT)))
                    HGOTO_ERROR(H5E_FILE, H5E_CANTGET, NULL, "unable to get file size");
                for (type = H5FD_MEM_SUPER; type < H5FD_MEM_NTYPES; type++) {
                    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    H5F_addr_decode(f, &p, &(fsinfo->fs_addr[type - 1]));
                }
                break;

            case H5F_FILE_SPACE_ALL:
                fsinfo->strategy  = H5F_FSPACE_STRATEGY_FSM_AGGR;
                fsinfo->threshold = threshold;
                break;

            case H5F_FILE_SPACE_AGGR_VFD:
                fsinfo->strategy = H5F_FSPACE_STRATEGY_AGGR;
                break;

            case H5F_FILE_SPACE_VFD:
                fsinfo->strategy = H5F_FSPACE_STRATEGY_NONE;
                break;

            case H5F_FILE_SPACE_NTYPES:
            case H5F_FILE_SPACE_DEFAULT:
            default:
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "invalid file space strategy");
        }

        fsinfo->version = H5O_FSINFO_VERSION_1;
        fsinfo->mapped  = true;
    }
    else {
        if (vers < H5O_FSINFO_VERSION_1)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "bad version number");

        fsinfo->version = vers;
        if (H5_IS_BUFFER_OVERFLOW(p, 1 + 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        fsinfo->strategy = (H5F_fspace_strategy_t)*p++; 
        fsinfo->persist  = *p++;                        

        if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_DECODE_LENGTH(f, p, fsinfo->threshold); 

        if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_DECODE_LENGTH(f, p, fsinfo->page_size); 
        
        if (fsinfo->page_size == 0 || fsinfo->page_size > H5F_FILE_SPACE_PAGE_SIZE_MAX)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "invalid page size in file space info");

        if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, fsinfo->pgend_meta_thres); 

        if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_addr_decode(f, &p,
                        &(fsinfo->eoa_pre_fsm_fsalloc)); 

        
        if (fsinfo->persist)
            for (ptype = H5F_MEM_PAGE_SUPER; ptype < H5F_MEM_PAGE_NTYPES; ptype++) {
                if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                H5F_addr_decode(f, &p, &(fsinfo->fs_addr[ptype - 1]));
            }
        fsinfo->mapped = false;
    }

    
    ret_value = fsinfo;

done:
    if (!ret_value && fsinfo)
        H5FL_FREE(H5O_fsinfo_t, fsinfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__fsinfo_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                   const void *_mesg)
{
    const H5O_fsinfo_t *fsinfo = (const H5O_fsinfo_t *)_mesg;
    H5F_mem_page_t      ptype; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(fsinfo);

    *p++ = (uint8_t)fsinfo->version;            
    *p++ = (uint8_t)fsinfo->strategy;           
    *p++ = (unsigned char)fsinfo->persist;      
    H5F_ENCODE_LENGTH(f, p, fsinfo->threshold); 

    H5F_ENCODE_LENGTH(f, p, fsinfo->page_size);          
    UINT16ENCODE(p, fsinfo->pgend_meta_thres);           
    H5F_addr_encode(f, &p, fsinfo->eoa_pre_fsm_fsalloc); 

    
    if (fsinfo->persist)
        
        for (ptype = H5F_MEM_PAGE_SUPER; ptype < H5F_MEM_PAGE_NTYPES; ptype++)
            H5F_addr_encode(f, &p, fsinfo->fs_addr[ptype - 1]);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__fsinfo_copy(const void *_mesg, void *_dest)
{
    const H5O_fsinfo_t *fsinfo    = (const H5O_fsinfo_t *)_mesg;
    H5O_fsinfo_t       *dest      = (H5O_fsinfo_t *)_dest;
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(fsinfo);
    if (!dest && NULL == (dest = H5FL_CALLOC(H5O_fsinfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *fsinfo;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__fsinfo_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_fsinfo_t *fsinfo    = (const H5O_fsinfo_t *)_mesg;
    size_t              ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    ret_value = 3                            
                + (size_t)H5F_SIZEOF_SIZE(f) 
                + (size_t)H5F_SIZEOF_SIZE(f) 
                + 2                          
                + (size_t)H5F_SIZEOF_ADDR(f);

    
    if (fsinfo->persist)
        ret_value += (H5F_MEM_PAGE_NTYPES - 1) * (size_t)H5F_SIZEOF_ADDR(f);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__fsinfo_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_fsinfo_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__fsinfo_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_fsinfo_t *fsinfo = (const H5O_fsinfo_t *)_mesg;
    H5F_mem_page_t      ptype; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(fsinfo);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "File space strategy:");
    switch (fsinfo->strategy) {
        case H5F_FSPACE_STRATEGY_FSM_AGGR:
            Rfprintf(stream, "%s\n", "H5F_FSPACE_STRATEGY_FSM_AGGR");
            break;

        case H5F_FSPACE_STRATEGY_PAGE:
            Rfprintf(stream, "%s\n", "H5F_FSPACE_STRATEGY_PAGE");
            break;

        case H5F_FSPACE_STRATEGY_AGGR:
            Rfprintf(stream, "%s\n", "H5F_FSPACE_STRATEGY_AGGR");
            break;

        case H5F_FSPACE_STRATEGY_NONE:
            Rfprintf(stream, "%s\n", "H5F_FSPACE_STRATEGY_NONE");
            break;

        case H5F_FSPACE_STRATEGY_NTYPES:
        default:
            Rfprintf(stream, "%s\n", "unknown");
    } 

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Free-space persist:", fsinfo->persist ? "TRUE" : "FALSE");

    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Free-space section threshold:", fsinfo->threshold);

    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "File space page size:", fsinfo->page_size);

    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "Page end metadata threshold:", (unsigned long long)fsinfo->pgend_meta_thres);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "eoa_pre_fsm_fsalloc:", fsinfo->eoa_pre_fsm_fsalloc);

    if (fsinfo->persist) {
        for (ptype = H5F_MEM_PAGE_SUPER; ptype < H5F_MEM_PAGE_NTYPES; ptype++)
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
                    "Free space manager address:", fsinfo->fs_addr[ptype - 1]);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_fsinfo_set_version(H5F_libver_t low, H5F_libver_t high, H5O_fsinfo_t *fsinfo)
{
    unsigned version;             
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    HDcompile_assert(N_FSINFO_VERSION_BOUNDS == H5F_LIBVER_NBOUNDS);
    assert(fsinfo);

    version = H5O_FSINFO_VERSION_1;

    
    if (H5O_fsinfo_ver_bounds[low] != H5O_INVALID_VERSION)
        version = MAX(version, H5O_fsinfo_ver_bounds[low]);

    
    if (H5O_fsinfo_ver_bounds[high] == H5O_INVALID_VERSION || version > H5O_fsinfo_ver_bounds[high])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "File space info message's version out of bounds");

    
    fsinfo->version = version;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_fsinfo_check_version(H5F_libver_t high, H5O_fsinfo_t *fsinfo)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    HDcompile_assert(N_FSINFO_VERSION_BOUNDS == H5F_LIBVER_NBOUNDS);
    assert(fsinfo);

    
    if (H5O_fsinfo_ver_bounds[high] == H5O_INVALID_VERSION || fsinfo->version > H5O_fsinfo_ver_bounds[high])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "File space info message's version out of bounds");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
