% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/error-measures.R
\name{ME}
\alias{ME}
\alias{Errors}
\alias{MAE}
\alias{MSE}
\alias{MRE}
\alias{MIS}
\alias{MPE}
\alias{MAPE}
\alias{MASE}
\alias{RMSSE}
\alias{rMAE}
\alias{rRMSE}
\alias{rAME}
\alias{rMIS}
\alias{sMSE}
\alias{sPIS}
\alias{sCE}
\alias{sMIS}
\alias{GMRAE}
\alias{SAME}
\title{Error measures}
\usage{
ME(holdout, forecast, na.rm = TRUE)

MAE(holdout, forecast, na.rm = TRUE)

MSE(holdout, forecast, na.rm = TRUE)

MRE(holdout, forecast, na.rm = TRUE)

MIS(holdout, lower, upper, level = 0.95, na.rm = TRUE)

MPE(holdout, forecast, na.rm = TRUE)

MAPE(holdout, forecast, na.rm = TRUE)

MASE(holdout, forecast, scale, na.rm = TRUE)

RMSSE(holdout, forecast, scale, na.rm = TRUE)

rMAE(holdout, forecast, benchmark, na.rm = TRUE)

rRMSE(holdout, forecast, benchmark, na.rm = TRUE)

rAME(holdout, forecast, benchmark, na.rm = TRUE)

rMIS(holdout, lower, upper, benchmarkLower, benchmarkUpper, level = 0.95,
  na.rm = TRUE)

sMSE(holdout, forecast, scale, na.rm = TRUE)

sPIS(holdout, forecast, scale, na.rm = TRUE)

sCE(holdout, forecast, scale, na.rm = TRUE)

sMIS(holdout, lower, upper, scale, level = 0.95, na.rm = TRUE)

GMRAE(holdout, forecast, benchmark, na.rm = TRUE)

SAME(holdout, forecast, scale, na.rm = TRUE)
}
\arguments{
\item{holdout}{The vector or matrix of holdout values.}

\item{forecast}{The vector or matrix of forecasts values.}

\item{na.rm}{Logical, defining whether to remove the NAs from the provided data or not.}

\item{lower}{The lower bound of the prediction interval.}

\item{upper}{The upper bound of the prediction interval.}

\item{level}{The confidence level of the constructed interval.}

\item{scale}{The value that should be used in the denominator of MASE. Can
be anything but advised values are: mean absolute deviation of in-sample one
step ahead Naive error or mean absolute value of the in-sample actuals.}

\item{benchmark}{The vector or matrix of the forecasts of the benchmark
model.}

\item{benchmarkLower}{The lower bound of the prediction interval of the
benchmark model.}

\item{benchmarkUpper}{The upper bound of the prediction interval of the
benchmark model.}
}
\value{
All the functions return the scalar value.
}
\description{
Functions allow to calculate different types of error metrics for point and
interval forecasts:
\enumerate{
\item ME - Mean Error,
\item MAE - Mean Absolute Error,
\item MSE - Mean Squared Error,
\item MRE - Mean Root Error (Kourentzes, 2014),
\item MIS - Mean Interval Score (Gneiting & Raftery, 2007),
\item MPE - Mean Percentage Error,
\item MAPE - Mean Absolute Percentage Error (See Svetunkov, 2017 for
the critique),
\item MASE - Mean Absolute Scaled Error (Hyndman & Koehler, 2006),
\item RMSSE - Root Mean Squared Scaled Error (used in M5 Competition),
\item SAME - Scaled Absolute Mean Error (similar to MASE, but measures bias),
\item rMAE - Relative Mean Absolute Error (Davydenko & Fildes, 2013),
\item rRMSE - Relative Root Mean Squared Error,
\item rAME - Relative Absolute Mean Error,
\item rMIS - Relative Mean Interval Score,
\item sMSE - Scaled Mean Squared Error (Petropoulos & Kourentzes, 2015),
\item sPIS- Scaled Periods-In-Stock (Wallstrom & Segerstedt, 2010),
\item sCE - Scaled Cumulative Error,
\item sMIS - Scaled Mean Interval Score,
\item GMRAE - Geometric Mean Relative Absolute Error.
}
}
\details{
In case of \code{sMSE}, \code{scale} needs to be a squared value. Typical
one -- squared mean value of in-sample actuals.

If all the measures are needed, then \link[greybox]{measures} function
can help.

There are several other measures, see details of \link[greybox]{pinball}
and \link[greybox]{hm}.
}
\examples{


y <- rnorm(100,10,2)
testForecast <- rep(mean(y[1:90]),10)

MAE(y[91:100],testForecast)
MSE(y[91:100],testForecast)

MPE(y[91:100],testForecast)
MAPE(y[91:100],testForecast)

# Measures from Petropoulos & Kourentzes (2015)
MASE(y[91:100],testForecast,mean(abs(y[1:90])))
sMSE(y[91:100],testForecast,mean(abs(y[1:90]))^2)
sPIS(y[91:100],testForecast,mean(abs(y[1:90])))
sCE(y[91:100],testForecast,mean(abs(y[1:90])))

# Original MASE from Hyndman & Koehler (2006)
MASE(y[91:100],testForecast,mean(abs(diff(y[1:90]))))

testForecast2 <- rep(y[91],10)
# Relative measures, from and inspired by Davydenko & Fildes (2013)
rMAE(y[91:100],testForecast2,testForecast)
rRMSE(y[91:100],testForecast2,testForecast)
rAME(y[91:100],testForecast2,testForecast)
GMRAE(y[91:100],testForecast2,testForecast)

#### Measures for the prediction intervals
# An example with mtcars data
ourModel <- alm(mpg~., mtcars[1:30,], distribution="dnorm")
ourBenchmark <- alm(mpg~1, mtcars[1:30,], distribution="dnorm")

# Produce predictions with the interval
ourForecast <- predict(ourModel, mtcars[-c(1:30),], interval="p")
ourBenchmarkForecast <- predict(ourBenchmark, mtcars[-c(1:30),], interval="p")

MIS(mtcars$mpg[-c(1:30)],ourForecast$lower,ourForecast$upper,0.95)
sMIS(mtcars$mpg[-c(1:30)],ourForecast$lower,ourForecast$upper,mean(mtcars$mpg[1:30]),0.95)
rMIS(mtcars$mpg[-c(1:30)],ourForecast$lower,ourForecast$upper,
       ourBenchmarkForecast$lower,ourBenchmarkForecast$upper,0.95)

### Also, see pinball function for other measures for the intervals

}
\references{
\itemize{
\item Kourentzes N. (2014). The Bias Coefficient: a new metric for forecast bias
\url{https://kourentzes.com/forecasting/2014/12/17/the-bias-coefficient-a-new-metric-for-forecast-bias/}
\item Svetunkov, I. (2017). Naughty APEs and the quest for the holy grail.
\url{https://openforecast.org/2017/07/29/naughty-apes-and-the-quest-for-the-holy-grail/}
\item Fildes R. (1992). The evaluation of
extrapolative forecasting methods. International Journal of Forecasting, 8,
pp.81-98.
\item Hyndman R.J., Koehler A.B. (2006). Another look at measures of
forecast accuracy. International Journal of Forecasting, 22, pp.679-688.
\item Petropoulos F., Kourentzes N. (2015). Forecast combinations for
intermittent demand. Journal of the Operational Research Society, 66,
pp.914-924.
\item Wallstrom P., Segerstedt A. (2010). Evaluation of forecasting error
measurements and techniques for intermittent demand. International Journal
of Production Economics, 128, pp.625-636.
\item Davydenko, A., Fildes, R. (2013). Measuring Forecasting Accuracy:
The Case Of Judgmental Adjustments To Sku-Level Demand Forecasts.
International Journal of Forecasting, 29(3), 510-522.
\doi{10.1016/j.ijforecast.2012.09.002}
\item Gneiting, T., & Raftery, A. E. (2007). Strictly proper scoring rules,
prediction, and estimation. Journal of the American Statistical Association,
102(477), 359–378. \doi{10.1198/016214506000001437}
}
}
\seealso{
\link[greybox]{pinball}, \link[greybox]{hm}, \link[greybox]{measures}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.com}
}
