% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fusedTree.R
\name{predict.fusedTree}
\alias{predict.fusedTree}
\title{Predict Method for Fused Tree Models}
\usage{
\method{predict}{fusedTree}(object, newX, newZ, newY, ...)
}
\arguments{
\item{object}{An object of class \code{"fusedTree"} returned by the \code{\link{fusedTree}} function.}

\item{newX}{A matrix of new omics covariates for prediction. Must have the same number of columns (variables) as used in the model fitting.}

\item{newZ}{A data frame of new clinical covariates for prediction.}

\item{newY}{A numeric response vector or a \code{\link[survival]{Surv}} object. For linear regression,
and logistic regression, newY is only used to store predictions and response values together. For cox regression,
newY is used to interpolate the baseline hazard to the event times of the test data.}

\item{...}{Currently not used. Included for S3 method consistency.}
}
\value{
A model-specific prediction object:
\itemize{
\item For \code{"linear"} models: a \code{data.frame} with columns \code{Resp} (observed responses) and \code{Ypred} (predicted values).
\item For \code{"logistic"} models: a \code{data.frame} with columns \code{Resp} (observed labels), \code{Ypred} (predicted probabilities), and \code{LinPred} (linear predictor).
\item For \code{"cox"} models: a \code{list} with two elements:
\describe{
\item{data}{A \code{data.frame} with columns \code{Resp} (a \code{Surv} object) and \code{LinPred}.}
\item{Survival}{A matrix of predicted survival probabilities. Rows = test subjects, columns = unique times from \code{newY}.}
}
}
}
\description{
Generates predictions from a fitted \code{fusedTree} object using new clinical and omics data.
}
\examples{
p = 5 # number of omics variables (low for illustration)
p_Clin = 5 # number of clinical variables
N = 100 # sample size
# simulate from Friedman-like function
g <- function(z) {
  15 * sin(pi * z[,1] * z[,2]) + 10 * (z[,3] - 0.5)^2 + 2 * exp(z[,4]) + 2 * z[,5]
}
set.seed(11)
Z <- as.data.frame(matrix(runif(N * p_Clin), nrow = N))
X <- matrix(rnorm(N * p), nrow = N)            # omics data
betas <- c(1,-1,3,4,2)                         # omics effects
Y <- g(Z) + X \%*\% betas + rnorm(N)             # continuous outcome
Y <- as.vector(Y)
dat = cbind.data.frame(Y, Z) #set-up data correctly for rpart
rp <- rpart::rpart(Y ~ ., data = dat,
                   control = rpart::rpart.control(xval = 5, minbucket = 10),
                   model = TRUE)
cp = rp$cptable[,1][which.min(rp$cptable[,4])] # best model according to pruning
Treefit <- rpart::prune(rp, cp = cp)
fit <- fusedTree(Tree = Treefit, X = X, Y = Y, Z = Z,
                    LinVars = FALSE, model = "linear",
                    lambda = 10,
                    alpha = 1000)
Preds <- predict(fit, newX = X, newZ = Z, newY = Y)
}
\seealso{
\code{\link{fusedTree}} for model fitting.
}
