% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{suggest_best_match}
\alias{suggest_best_match}
\title{Suggest Best Matches for a String from a Vector of Choices}
\usage{
suggest_best_match(
  x,
  choices,
  n = 1,
  threshold = 2,
  ignore.case = TRUE,
  return_distance = FALSE
)
}
\arguments{
\item{x}{A single character string; the user input to find matches for.}

\item{choices}{A character vector of available, valid options.}

\item{n}{An integer specifying the maximum number of suggestions to return.
Defaults to 1.}

\item{threshold}{An integer; the maximum Levenshtein distance to consider a
choice a "close" match. A lower value is stricter. Defaults to 2.}

\item{ignore.case}{A logical value. If \code{TRUE}, matching is case-insensitive.
Defaults to \code{TRUE}.}

\item{return_distance}{A logical value. If \code{TRUE}, the output is a data.frame
containing the suggestions and their calculated distance/score. Defaults to
\code{FALSE}.}
}
\value{
By default (\code{return_distance = FALSE}), returns a character vector of the
best \code{n} suggestions. If no suitable match is found, returns \code{NA}.
If \code{return_distance = TRUE}, returns a \code{data.frame} with columns
\code{suggestion} and \code{distance}, or \code{NULL} if no match is found.
}
\description{
This function provides intelligent suggestions for a user's input string by
finding the best matches from a given vector of choices. It follows a
multi-layered approach:
\enumerate{
\item Performs normalization (case-insensitivity, trimming whitespace).
\item Checks for an exact match first for maximum performance and accuracy.
\item If no exact match, it uses a combination of fuzzy string matching
(Levenshtein distance via \code{adist}) to catch typos and partial/substring
matching (\code{grep}) to handle incomplete input.
\item Ranks the potential matches and returns the best suggestion(s).
}
}
\examples{
# --- Setup ---
cell_types <- c(
  "B cell", "T cell", "Macrophage", "Monocyte", "Neutrophil",
  "Natural Killer T-cell", "Dendritic cell"
)

# --- Usage ---
# 1. Exact match (after normalization)
suggest_best_match("t cell", cell_types)
#> [1] "T cell"

# 2. Typo correction (fuzzy match)
suggest_best_match("Macrophaeg", cell_types)
#> [1] "Macrophage"

# 3. Partial input (substring match)
suggest_best_match("Mono", cell_types)
#> [1] "Monocyte"

# 4. Requesting multiple suggestions
suggest_best_match("t", cell_types, n = 3)
#> [1] "T cell" "Neutrophil" "Natural Killer T-cell"

# 5. No good match found
suggest_best_match("Erythrocyte", cell_types)
#> [1] NA

# 6. Returning suggestions with their distance score
suggest_best_match("t ce", cell_types, n = 3, return_distance = TRUE)
#>              suggestion distance
#> 1                T cell        1
#> 2        Dendritic cell        2
#> 3 Natural Killer T-cell        2
}
