% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SQUIRE.R
\name{SQUIRE}
\alias{SQUIRE}
\title{SQUIRE: Statistical Quality-Assured Integrated Response Estimation}
\usage{
SQUIRE(
  data,
  treatments,
  control_treatment = treatments[1],
  response_type = c("germination", "growth", "survival"),
  validation_level = 0.05,
  min_timepoints = 5,
  min_replicates = 3,
  galahad_config = NULL,
  verbose = TRUE
)
}
\arguments{
\item{data}{Data frame with columns: time, response, treatment, replicate}

\item{treatments}{Character vector of treatment names}

\item{control_treatment}{Name of control treatment for comparisons}

\item{response_type}{Type of response: "germination", "growth", "survival"}

\item{validation_level}{Statistical significance level (default: 0.05)}

\item{min_timepoints}{Minimum timepoints required for fitting (default: 5)}

\item{min_replicates}{Minimum replicates per treatment (default: 3)}

\item{galahad_config}{Optional pre-calibrated GALAHAD parameters}

\item{verbose}{Logical, print progress messages}
}
\value{
List with statistical validation results, optimized parameters,
and biological interpretation (only if statistically justified)
}
\description{
Geometry-adaptive biological parameter estimation with built-in
statistical validation. Implements two-cycle optimization: statistical
validation followed by GALAHAD-calibrated parameter estimation.
}
\details{
SQUIRE implements a two-stage validation process:

\strong{Stage 1: Statistical Validation}
\itemize{
\item Tests for significant treatment effects using ANOVA
\item Checks data quality requirements (timepoints, replication)
\item Only proceeds to optimization if biological signals detected
}

\strong{Stage 2: Validated Optimization}
\itemize{
\item Calibrates GALAHAD geometry parameters on significant effects
\item Applies optimized parameters with uncertainty quantification
\item Validates that optimized parameters are statistically meaningful
}
}
\examples{
# Quick data setup example (fast execution)
n_time <- 5
n_rep <- 3

# Simulate example data
example_data <- data.frame(
  time = rep(1:n_time, times = 3 * n_rep),
  treatment = rep(c("Control", "Treatment_A", "Treatment_B"), 
                  each = n_time * n_rep),
  replicate = rep(rep(1:n_rep, each = n_time), times = 3),
  response = c(
    cumsum(rbinom(n_time * n_rep, 1, 0.1)),  # Control
    cumsum(rbinom(n_time * n_rep, 1, 0.15)), # Treatment A  
    cumsum(rbinom(n_time * n_rep, 1, 0.2))   # Treatment B
  )
)

# Inspect data structure (this runs quickly)
head(example_data)
table(example_data$treatment)

\donttest{
# Full analysis (longer computation)
results <- SQUIRE(
  data = example_data,
  treatments = c("Control", "Treatment_A", "Treatment_B"),
  control_treatment = "Control",
  response_type = "germination",
  verbose = FALSE
)

# Check results
if(results$optimization_performed) {
  print("Optimization was justified")
  print(results$parameters)
} else {
  print("No significant effects detected")
  print(results$statistical_advice)
}
}

}
