% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/instrinsicIsotopes.R
\name{isoAssign}
\alias{isoAssign}
\title{Generate probabilistic isotope assignments}
\usage{
isoAssign(
  isovalues,
  isoSTD,
  intercept,
  slope,
  odds = 0.67,
  restrict2Likely = TRUE,
  nSamples = NULL,
  sppShapefile = NULL,
  relAbund = NULL,
  isoWeight = NULL,
  abundWeight = NULL,
  population = NULL,
  assignExtent = c(-179, -60, 15, 89),
  element = "Hydrogen",
  surface = FALSE,
  period = "Annual",
  seed = NULL,
  verbose = 1,
  generateSingleCell = FALSE,
  mapDirectory = NULL
)
}
\arguments{
\item{isovalues}{vector of tissue isotope values}

\item{isoSTD}{standard deviation from calibration}

\item{intercept}{intercept value from calibration}

\item{slope}{value from calibration}

\item{odds}{odds ratio to use to set likely and unlikely locations defaults
to 0.67}

\item{restrict2Likely}{if \code{TRUE} restricts locations to fall within the
'likely' assignment locations}

\item{nSamples}{integer specifying how many random samples to draw from a
multinomial distribution.}

\item{sppShapefile}{A polygon spatial layer (sf - MULTIPOLYGON) defining
species range. Assignments are restricted to these areas.}

\item{relAbund}{raster (\code{SpatRast}) with relative abundance (must match
extent of isotope assignment)}

\item{isoWeight}{weighting value to apply to isotope assignment}

\item{abundWeight}{weighting value to apply to relative abundance prior}

\item{population}{vector identifying location where animal was captured.
Same order as \code{isovalues}}

\item{assignExtent}{definition for the extent of the assignment. Can be used
in place of \code{sppShapefile} to limit assignment. Input should
follow \code{c(xmin,xmax,ymin,ymax)} in degrees longitude and
latitude}

\item{element}{The elemental isotope of interest. Currently the only
elements that are implemented are 'Hydrogen' (default) and 'Oxygen'}

\item{surface}{DEPRECATED function no longer returns surface water values.
Default is 'FALSE' which returns the precipitation isotopes ratio.}

\item{period}{The time period of interest. If 'Annual' returns a raster
of mean annual values in precipitation for the \code{element}. If
'GrowingSeason' returns growing season values in precipitation for
\code{element} of interest}

\item{seed}{numeric value fed to \code{set.seed} for random number
generation. Default = NULL}

\item{verbose}{takes values 0, 1 (default) or 2. 0 prints no output during
run. 1 prints a message detailing where in the process the function
is. 2 prints the animal currently being sampled.}

\item{generateSingleCell}{if 'TRUE' generates a single origin location using
the posterior assignment distribution - this takes a while to run.
If 'FALSE' (default), no coordinates are generated.}

\item{mapDirectory}{Directory to save/read isotope map from. Can use relative
or absolute addressing. The default value (NULL) downloads to a temporary
directory, so we strongly recommend changing this from the default unless
you're sure you're not going to need these data more than once.}
}
\value{
returns an \code{isoAssign} object containing the following:
 \describe{
  \item{\code{probassign}}{SpatRast stack of individual probabilistic assignments}
  \item{\code{oddsassign}}{SpatRast stack that includes likely vs unlikely origin for each animal}
  \item{\code{popassign}}{a SpatRast for population level assignment (sum of \code{oodsassign} if \code{population} = NULL).
  If \code{population} is a vector then returns a raster stack for each unique \code{population} provided}
  \item{\code{probDF}}{data.frame of individual probability surfaces}
  \item{\code{oddsDF}}{data.frame of likely vs unlikely surfaces}
  \item{\code{popDF}}{data.frame of population level assignment}
  \item{\code{SingeCell}}{array of coordinates (longitude,latitude) for
  single cell assignment}
  \item{\code{targetSites}}{\code{sf - MULTIPOLYGON} layer representing
  isotope bands equivalent to \code{isoSTD}}
  \item{\code{RandomSeed}}{the RNG seed used when generating locations from
  the multinomial distribution}
  }
}
\description{
The \code{isoAssign} function generates origin assignments using
stable-hydrogen isotopes in tissue. The function generates a probability
surface of origin assignment from a vector of stable-isotope values for each
animal/sample of interest. Probabilistic assignments are constructed by first
converting observed stable-isotope ratios (isoscape) in either precipitation
or surface waters into a 'tissuescape' using a user-provided intercept, slope
and standard deviation. See
\href{https://journals.plos.org/plosone/article?id=10.1371/journal.pone.0035137}{Hobson et. al. (2012)}.
}
\examples{
\donttest{
extensions <- c("shp", "shx", "dbf", "sbn", "sbx")
tmp <- tempdir()
for (ext in extensions) {
download.file(paste0(
              "https://raw.githubusercontent.com/SMBC-NZP/MigConnectivity",
                     "/master/data-raw/Spatial_Layers/OVENdist.",
                     ext),
              destfile = paste0(tmp, "/OVENdist.", ext), mode = "wb")
}
OVENdist <- sf::st_read(paste0(tmp, "/OVENdist.shp"))
OVENdist <- OVENdist[OVENdist$ORIGIN==2,] # only breeding
sf::st_crs(OVENdist) <- sf::st_crs(4326)

download.file(paste0(
  "https://raw.githubusercontent.com/SMBC-NZP/MigConnectivity",
                     "/master/data-raw/deltaDvalues.csv"),
              destfile = paste0(tmp, "/deltaDvalues.csv"))
OVENvals <- read.csv(paste0(tmp, "/deltaDvalues.csv"))


a <- Sys.time()
b <- isoAssign(isovalues = OVENvals[,2],
              isoSTD = 12,
              intercept = -10,
              slope = 0.8,
              odds = NULL,
              restrict2Likely = TRUE,
              nSamples = 1000,
              sppShapefile = OVENdist,
              assignExtent = c(-179,-60,15,89),
              element = "Hydrogen",
              period = "GrowingSeason") # this setting for demonstration only
Sys.time()-a
}

}
\references{
Cohen, E. B., C. S. Rushing, F. R. Moore, M. T. Hallworth, J. A. Hostetler,
M. Gutierrez Ramirez, and P. P. Marra. 2019. The strength of
migratory connectivity for birds en route to breeding through the Gulf of
Mexico. Ecography 42: 658-669.

Hobson, K. A., S. L. Van Wilgenburg, L. I. Wassenaar, and K. Larson. 2012.
Linking hydrogen isotopes in feathers and precipitation: sources of
variance and consequences for assignment to isoscapes. PLoS ONE 7: e35137.
}
\seealso{
{\code{\link{weightAssign}}}
}
