\name{readParmayMatrix}
\alias{readParmayMatrix}
\title{Read and Process Binary Data from Matrix Devices}
\description{
  Reads a binary file generated by Parmay Tech Matrix devices, processes its header 
  and packet data, validates data integrity using CRC32 checksums, and outputs 
  structured sensor data and quality check information.
}
\usage{
  readParmayMatrix(filename, output = c("all", "sf", "dynrange")[1], 
                   start = 1, end = NULL,
                   desiredtz = "", configtz = NULL, interpolationType = 1,
                   read_acc = TRUE, read_gyro = FALSE, 
                   read_temp = TRUE, read_heart = FALSE)
}
\arguments{
  \item{filename}{Character. Path to the binary file to be read.}
  \item{output}{Character. Specifies the type of output. Options include:
    \describe{
      \item{"all"}{Returns the full processed data.}
      \item{"sf"}{Returns the sampling frequency of the accelerometer data.}
      \item{"dynrange"}{Returns the dynamic range of the accelerometer.}
    }
  }
  \item{start}{Integer. The index of the starting packet to process. Default is 1.}
  \item{end}{Integer. The index of the ending packet to process. Default is \code{NULL}, which means all packets are processed.}
  \item{desiredtz}{Character. Time zone for the returned timestamps. Default is an empty string, which uses the system's default time zone.}
  \item{configtz}{Character. Time zone specified in the file's configuration. Default is \code{NULL}, which means that it uses \code{desiredtz}.}
  \item{interpolationType}{Integer. Specifies the type of interpolation 
    (see \code{\link{resample}}) to use when resampling data: 1 for Linear interpolation,
    2 for Nearest-neighbor interpolation.}
  \item{read_acc}{Logical. Indicates whether accelerometer data should be read.}
  \item{read_gyro}{Logical. Indicates whether gyroscope data should be read.}
  \item{read_temp}{Logical. Indicates whether temperature data should be read.}
  \item{read_heart}{Logical. Indicates whether heart rate data should be read.}
}
\details{
  Matrix devices store binary data in packets, with varying lengths depending on 
  the number of sensor recordings in each packet. The function processes the file's 
  header to extract metadata such as the total number of packets and sensor ranges, 
  validates data integrity using CRC32 checksums, and interpolates data to a 
  consistent sampling frequency.
  
  \strong{Header Information:}
  \itemize{
    \item Remarks: Bytes 1-512.
    \item Total packets: Bytes 513-516.
    \item Header string: Bytes 517-520. If not "MDTC", the file is considered corrupt.
    \item Accelerometer dynamic range: Bytes 521-522.
    \item Gyroscope range: Bytes 523-524.
  }
  
  \strong{Packet Structure:}
  
  Each packet contains accelerometer, gyroscope, temperature, and heart rate data.
  
  \itemize{
    \item 8-byte package header.
    \item 4-byte CRC32 indicator.
    \item 4-byte start timestamp.
    \item 4-byte end timestamp.
    \item 4-byte number of accelerometer recordings in packet.
    \item 4-byte number of gyroscope recordings in packet.
    \item 4-byte number of temperature recordings in packet.
    \item 4-byte number of heart rate recordings in packet.
    \item Sensor data for accelerometer, gyroscope, temperature, and heart rate.
  }
}
\value{
  A list containing the following elements (when \code{output = "all"}):
  \itemize{
    \item \code{QClog}: A data frame with quality control information, including 
    checksum validation and data gaps.
    \item \code{output}: A data frame with resampled sensor data, including:
      \describe{
        \item{\code{time}}{Timestamps of the recordings.}
        \item{\code{acc_x}, \code{acc_y}, \code{acc_z}}{Resampled accelerometer data.}
        \item{\code{gyro_x}, \code{gyro_y}, \code{gyro_z}}{Resampled gyroscope data.}
        \item{\code{bodySurface_temp}, \code{ambient_temp}}{Resampled temperature data.}
        \item{\code{hr_raw}, \code{hr}}{Resampled heart rate data.}
        \item{\code{remarks}}{Remarks extracted from the file header.}
      }
    \item \code{header}: A list with the following elements:
      \describe{
        \item{\code{sf}}{Sampling frequency of the accelerometer data.}
        \item{\code{acc_dynrange}}{Dynamic range of the accelerometer.}
        \item{\code{starttime}}{Start time of the first packet in POSIXct format.}
      }
    \item \code{lastchunk}: Logical, indicating if the processed data includes the last packet in the file.
  }
  
  If \code{output = "sf"}, the function returns only the sampling frequency. 
  
  If \code{output = "dynrange"}, it returns the dynamic range of the accelerometer.
}
\examples{
  \dontrun{
  # Example usage:
  binfile  = system.file("testfiles/mtx_12.5Hz_acc.BIN", package = "GGIRread")
  # Read full data and process all packets
  result <- readParmayMatrix(binfile)
  
  # Get sampling frequency only
  sf <- readParmayMatrix(binfile, output = "sf")
  
  # Get accelerometer dynamic range
  dynrange <- readParmayMatrix(binfile, output = "dynrange")
  
  # Process a subset of packets
  result_subset <- readParmayMatrix(binfile, start = 10, end = 20)
  }
}
\references{
  For more details on Matrix devices', see: 
  \url{https://www.parmaytech.com/devices/en-matrix}
  For additional details on Matrix bin/BIN files structure, please contact manufacturer:
  \url{https://www.parmaytech.com/contact}
}
\seealso{
  \code{\link{resample}} for resampling sensor data.
}
\author{
  Jairo H Migueles <jairo@jhmigueles.com>
}