% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdblockMult}
\alias{bdblockMult}
\title{Block-Based Matrix Multiplication}
\usage{
bdblockMult(
  A,
  B,
  block_size = NULL,
  paral = NULL,
  byBlocks = TRUE,
  threads = NULL
)
}
\arguments{
\item{A}{Matrix or vector. First input operand.}

\item{B}{Matrix or vector. Second input operand.}

\item{block_size}{Integer. Block size for computation. If NULL, uses maximum
allowed block size.}

\item{paral}{Logical. If TRUE, enables parallel computation. Default is FALSE.}

\item{byBlocks}{Logical. If TRUE (default), forces block-based computation for
large matrices. Can be set to FALSE to disable blocking.}

\item{threads}{Integer. Number of threads for parallel computation. If NULL,
uses half of available threads or maximum allowed threads.}
}
\value{
Matrix or vector containing the result of A * B.
}
\description{
Performs efficient matrix multiplication using block-based algorithms. The function
supports various input combinations (matrix-matrix, matrix-vector, vector-vector)
and provides options for parallel processing and block-based computation.
}
\details{
This function implements block-based matrix multiplication algorithms optimized
for cache efficiency and memory usage. Key features:
\itemize{
\item Input combinations supported:
\itemize{
\item Matrix-matrix multiplication
\item Matrix-vector multiplication (both left and right)
\item Vector-vector multiplication
}
\item Performance optimizations:
\itemize{
\item Block-based computation for cache efficiency
\item Parallel processing for large matrices
\item Automatic block size selection
\item Memory-efficient implementation
}
}

The function automatically selects the appropriate multiplication method based
on input types and sizes. For large matrices (>2.25e+08 elements), block-based
computation is used by default.
}
\examples{
\dontrun{
library(BigDataStatMeth)

# Matrix-matrix multiplication
N <- 2500
M <- 400
nc <- 4

set.seed(555)
mat <- matrix(rnorm(N*M, mean=0, sd=10), N, M)

# Parallel block multiplication
result <- bdblockMult(mat, mat,
                      paral = TRUE,
                      threads = nc)

# Matrix-vector multiplication
vec <- rnorm(M)
result_mv <- bdblockMult(mat, vec,
                         paral = TRUE,
                         threads = nc)
}

}
\references{
\itemize{
\item Golub, G. H., & Van Loan, C. F. (2013). Matrix Computations, 4th Edition.
Johns Hopkins University Press.
\item Kumar, V. et al. (1994). Introduction to Parallel Computing: Design and
Analysis of Algorithms. Benjamin/Cummings Publishing Company.
}
}
\seealso{
\itemize{
\item \code{\link{bdblockSum}} for block-based matrix addition
\item \code{\link{bdblockSubstract}} for block-based matrix subtraction
}
}
